﻿/*
 * Copyright 2017 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <stdbool.h>

#include "board.h"
#include "pin_mux.h"
#include "clock_config.h"
#include "fsl_debug_console.h"

#include "led.h"
#include "touch.h"
#include "key_filter.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/
typedef enum
{
    eAppTouchState_Init = 0,
    eAppTouchState_Calib = 1,
    eAppTouchState_Detect = 2,
} app_touch_state_t;

#define APP_TOUCH_CALIB_ERR_MAX  2
#define APP_TOUCH_POS_VAL_RANGE_STEP  100

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Variables7
 ******************************************************************************/
int16_t app_touch_slider_val_raw[TOUCH_USER_CHANNEL_NUM] = {0};
int16_t app_touch_slider_val[TOUCH_USER_CHANNEL_NUM] = {0};
int32_t app_touch_slider_pos;
volatile app_touch_state_t app_touch_state = eAppTouchState_Init;
volatile int app_touch_state_init_counter = 0;
volatile uint8_t app_led_indicator_raw;
volatile uint8_t app_led_indicator = LED_USER_NUM;

Key_FilterHandler_T app_led_indicator_filter;

/*******************************************************************************
 * Code
 ******************************************************************************/
int main(void)
{
    /* platform. */
    BOARD_InitBootClocks();

    /* led. */
    led_init();
    led_on( (1u << LED_USER_NUM)-1 ); /* calib done flag. */

    /* touch. */
    touch_init(); /* initialize all used pins. */
    touch_start(); /* start the scan. */

    /* state machine. */
    app_touch_state = eAppTouchState_Init;
    app_touch_state_init_counter = 0;
    app_led_indicator = LED_USER_NUM;

    Key_Filter_Init(&app_led_indicator_filter, LED_USER_NUM, 3);

    while (1)
    {
        touch_wait_data_ready(app_touch_slider_val_raw);
        touch_start(); /* start new scan for next process. */

        /* process the state machine. */
        switch (app_touch_state)
        {
        case eAppTouchState_Init:
            if ( app_touch_state_init_counter < 5u )
            {
                app_touch_state_init_counter++; /* 启动状态机后先等几个扫描周期, 预热感应电路. */
            }
            else
            {
                app_touch_state_init_counter = 0u;
                app_touch_state = eAppTouchState_Calib;
            }
            break;
        case eAppTouchState_Calib:
            if ( touch_calib(app_touch_slider_val_raw) <= APP_TOUCH_CALIB_ERR_MAX) /* max err <= 2 */
            {
                touch_iir_filter_init(app_touch_slider_val_raw);
                app_touch_state = eAppTouchState_Detect;
                led_off( (1u << LED_USER_NUM)-1 ); /* calib done flag. */
            }
            break;
        case eAppTouchState_Detect:
            /* 验证采样数据. */
            if ( !touch_verify(app_touch_slider_val_raw, 100) )
            {
                //app_touch_state = eAppTouchState_Init; /* 打回原型 */
                break; /* 本次采样数据无效, 跳过本次采样, 状态机保持现状 */
            }
            /* 平滑原始数据. */
            touch_iir_filter_process(app_touch_slider_val_raw, app_touch_slider_val);

            /* 计算坐标 */
            if (eTouch_GetPosErr_YesTouch != touch_get_pos(app_touch_slider_val,
                                                           APP_TOUCH_POS_VAL_RANGE_STEP * (LED_USER_NUM+2),
                                                           &app_touch_slider_pos) )
            {
                app_led_indicator_raw = LED_USER_NUM;
            }
            else
            {
                app_led_indicator_raw = app_touch_slider_pos / APP_TOUCH_POS_VAL_RANGE_STEP;

                if (app_led_indicator_raw <= 1)
                {
                    app_led_indicator_raw = 0; /* 抄底. */
                }
                else if (app_led_indicator_raw >= LED_USER_NUM)
                {
                    app_led_indicator_raw = LED_USER_NUM-1; /* 去顶. */
                }
                else
                {
                    app_led_indicator_raw--;
                }
            }

#if 0 /* enable the key filter. */
            app_led_indicator = Key_Filter_GetSymmetricDebounceOutput(&app_led_indicator_filter, app_led_indicator_raw);
#else
            app_led_indicator = app_led_indicator_raw;
#endif
            /* use led to indicate position. */
            led_off( (1u << LED_USER_NUM)-1 );
            if (app_led_indicator < LED_USER_NUM)
            {
                led_on((1u << app_led_indicator)-1);
            }

            break;
        default:
            app_touch_state = eAppTouchState_Init;
            break;
        }
    }
}

/* EOF. */

