/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright (c) 2016-2021, NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */
#include "stdio.h"
#include "Defines.h"
#include "Application.h"
#include "AppCommon.h"
#include "EEPROMAddrs.h"
#include "UserInterface.h"
#include "IOControls.h"
#include "ComPortDriver.h"
#include "MeteringLPRT.h"
#include "MeteringInterface1Ph.h"
#include "lcd.h"
#include "EEPROMDriver.h"
#include "Timer.h"
#include "PowerModes.h"
/*******************************************************************************
* Definitions
******************************************************************************/

/*******************************************************************************
* Prototypes
******************************************************************************/
static void lcd_scroll_display(void);
static void DisplayActPower(void);
static void DisplayReactPower(void);
static void DisplayAppPower(void);
static void CalculateLCDScrollParam(void);

uint8 AutoScrollParam[MaxAutoDisps] =
{
  DISP_ALL_ON,
  DISP_VRMS,
  DISP_IRMS_PH,
  DISP_IRMS_N
};

uint8 ManualScrollParam[MaxPushDisps] =
{
  DISP_ALL_ON,
  DISP_DATE,
  DISP_TIME,
  DISP_VRMS,
  DISP_IRMS_PH,
  DISP_IRMS_N,
  DISP_POWER_ACT,
  DISP_METER_SNO
};

uint8 BattScrollParam[nBattModeDisps] =
{
  DISP_ALL_ON,
  DISP_VRMS,
  DISP_IRMS_PH,
  DISP_IRMS_N
};

/*
 * ---------------------------------------
 * LCD Driver Global Variables & Structure
 * ---------------------------------------
 */
/** Load count for Refresh LCD */
vuint16 lcd_refresh_count = 1;
/** Load count for auto scroll of 10 seconds */
vuint16 lcd_scroll_count = 0;

uint8  DispMDTypes;
uint8  DispPowerTypes;
uint8  PowerIndex;
uint8  DisplayMode;
uint8  PhStateToggle;
tDispConfig DispConfig;
stTime Time;
stDate Date;

uint8  DispIndex;
uint8  DispParam = 0x10;

char  disp_string[NUM_DIGITS];
uint8 UIBuffer[MaxPushDisps];

/*
 * ---------------------------------------
 * User Interface Global Variables & Structure
 * ---------------------------------------
 */

/** Flag for user switch **/
uint8 DownScroll = 0;
uint8 UpScroll = 0;

/*******************************************************************************
 * Code
 ******************************************************************************/
/*!
 * @brief Intitializes the LCD module in the meter.
 */
void LCDModuleInit(void)
{
  LCDInit();
  LCDClear();
}

/*!
 * @brief Displays start up messsage in the meter LCD.
 */
void LCDPrintInitString(void)
{
  sprintf(disp_string,"1PHSM");
  lcd_PrintString( );
  SDK_DelayAtLeastUs(1000, SYSTEM_CLOCK);
}

/*!
* @brief Intitializes the LCD module in the meter.
*/
void LCDModuleInitVal(uint32 Val)
{
  LCDInit();
  LCDClear();
  sprintf(disp_string,"%6d", Val);
  lcd_PrintString( );
}

/*!
* @brief Initializes the user interface relted data objects.
*/
void UserInterfaceInit(void)
{
  uint16 i;
  uint16 j;
  
  /* Cleanr all LCD segments */
  LCDClear();
  
  /* Turn on all LCD segments, indicative LCD segments are working */
  LCD_ALLFP_ON;
  
  /* Load count for auto scroll of 10 seconds */
  lcd_scroll_count = (15);
  
  /* Load LCD refresh count for 1 */
  lcd_refresh_count = 1;
  
  /* Set the default display modes */
  DisplayMode = DISP_AUTO;
    
  /* Reset a starting display index from a list of display parameter */
  DispIndex = 0x0;
  
  /* Set the display configuration parameters as per application/tender requirement */
  NVReadIIC(DispConfigAddr, (uint8 *)&DispConfig, sizeof(DispConfig));
  if (DispConfig.DispSign != DISPSIGN)
  {
    DispConfig.DispSign = DISPSIGN;

    DispConfig.DispAutoSize = 4;
    DispConfig.DispManualSize = 8;
    DispConfig.DispAutoTime = 15;
    DispConfig.DispManualTime = 10;
    DispConfig.DispBattCounts = 0;
    // Remove if using upgradable display list
    if (SystemState.MetMode == METMODE_NONE)
    {
      for (i = 0; i < nBattModeDisps; i++)
      {
        AutoScrollParam[i] = BattScrollParam[i];
        ManualScrollParam[i] = BattScrollParam[i];
      }
      DispConfig.DispAutoSize = nBattModeDisps;
      DispConfig.DispManualSize = nBattModeDisps;
    }
    NVWriteIIC(AutoDispTable, &AutoScrollParam[0], sizeof(AutoScrollParam));
    NVWriteIIC(BattDispTable, &ManualScrollParam[0], sizeof(ManualScrollParam));
    NVWriteIIC(ManualDispTable, &ManualScrollParam[0], sizeof(ManualScrollParam));
    NVWriteIIC(DispConfigAddr, (uint8 *)&DispConfig, sizeof(DispConfig));
  }
  else
  {
    /* Manual params */
    NVReadIIC(ManualDispTable, UIBuffer, sizeof(ManualScrollParam));
    if (UIBuffer[0] >= 0x10)
    {
      j = 0;
      for (i = 0; i < MaxPushDisps; i++)
      {
        ManualScrollParam[j++] = UIBuffer[i];
      }
    }
    if (SystemState.PowerMode == POWERMODE_MAINS)
    {
      // Mains mode
      NVReadIIC(AutoDispTable, UIBuffer, sizeof(AutoScrollParam));
      if (UIBuffer[0] >= 0x10)
      {
        j = 0;
        for (i = 0; i < MaxAutoDisps; i++)
        {
          AutoScrollParam[j++] = UIBuffer[i];
        }
      }
    }
    else
    {
      if (SystemState.MetMode == METMODE_NONE)
      {
        for (i = 0; i < nBattModeDisps; i++)
        {
          AutoScrollParam[i] = BattScrollParam[i];
          ManualScrollParam[i] = BattScrollParam[i];
        }
        DispConfig.DispAutoSize = nBattModeDisps;
        DispConfig.DispManualSize = nBattModeDisps;
      }
      else
      {
        NVReadIIC(AutoDispTable, UIBuffer, sizeof(AutoScrollParam));
        if (UIBuffer[0] >= 0x10)
        {
          j = 0;
          for (i = 0; i < MaxAutoDisps; i++)
          {
            AutoScrollParam[j++] = UIBuffer[i];
          }
        }
      }
    }
  }

  DispParam = AutoScrollParam[DispIndex];
}

/*!
 * @brief This function LCD display related tasks of the meter.
 */
void Display(void)
{
  if (DownScroll)
  {
    /* Down scroll button press was detected */
    if (SystemState.PowerMode == POWERMODE_BAT)
    {
      if (BattModeTimeout < BattModeTO)
      {
        /* Battery mode timeout should be reset to keep the meter running */
        BattModeTimeout = BattModeTO;
      }
    }
    if (DisplayMode == DISP_AUTO)
    {
      DispIndex = 0xFF;
      DisplayMode = DISP_MANUAL;
    }
    
    /* Calculate the scroll index parameter */
    CalculateLCDScrollParam();
    DownScroll = 0;
  }
  else if(0 == lcd_refresh_count)
  {
    lcd_scroll_display();
  }
  else if(0 == lcd_scroll_count)
  {
    if (DisplayMode == DISP_MANUAL)
    {
      DispIndex = 0xFF;
      DisplayMode = DISP_AUTO;
    }
    /* Calculate the scroll index parameter */
    CalculateLCDScrollParam();
  }
}

/*!
 * @brief This function calculates the scroll LCD display indices periodically 
 * for the scroll display.
 */
static void CalculateLCDScrollParam(void)
{
  uint8 *ScreenParam = AutoScrollParam;
  uint8 Size = DispConfig.DispAutoSize;
  
  lcd_scroll_count = DispConfig.DispAutoTime;
  if (DisplayMode == DISP_MANUAL)
  {
    lcd_scroll_count = DispConfig.DispManualTime;
    ScreenParam = ManualScrollParam;
    Size = DispConfig.DispManualSize;
  }
  
  DispIndex++;
  if (DispIndex >= Size)
  {
    if (SystemState.PowerMode == POWERMODE_BAT)
    {
      TurnLatchOff = TRUE;
    }
    if (DisplayMode == DISP_AUTO)
    {
      DispIndex = 1;
    }
    else
    {
      DispIndex = 0;
    }
  }
  
  DispParam = ScreenParam[DispIndex];
  lcd_scroll_display();
}

/*!
 * @brief This function displays meter parameter message on LCD screen based on 
 * a pre-calculated display index, .
 */
void lcd_scroll_display(void)
{
  uint8 Index;
  uint32 temp1 = 0;
  uint32 temp2 = 0;
  
  for (Index = 0; Index < NUM_FRONTPLANEPINS; Index++)
  {
    LCDArray[Index] = 0;
  }

  lcd_refresh_count = 1; 
  
  if ((SystemState.PowerMode == POWERMODE_MAINS) && (MainsOn == TRUE))
  {
    ON_ON();
  }
  for (Index = 0; Index < NUM_DIGITS; Index++)
  {
    disp_string[Index] = 0;
  }
  
  switch(DispParam)
  {
  case DISP_ALL_ON:
    LCD_ALLFP_ON;
    break;

  case DISP_VRMS:
    if(mlib1phdata.MetOnImax == TRUE)
    {
      mlib1phdata.Vrms = 240.0;
    }
    temp1 = (uint32)(mlib1phdata.Vrms * 100);
    sprintf(disp_string," %05ld1n", temp1);
    ON_DP3();
    ON_V();
    break;
    
  case DISP_IRMS_PH:
    temp1 = (uint32)(mlib1phdata.Irms[CURRENT_PHASE] * 100);
    sprintf(disp_string,"  %04ld P", temp1);
    ON_A();
    ON_DP3();
    break;

  case DISP_IRMS_N:
    temp1 = (uint32)(mlib1phdata.Irms[CURRENT_NEUTRAL] * 100);
    sprintf(disp_string,"  %04ld N", temp1);
    ON_A();
    ON_DP3();
    break;

  case DISP_IRMS:
    temp1 = (uint32)(mlib1phdata.Irms[mlib1phdata.CurToUse] * 100);
    sprintf(disp_string,"  %04ld1n", temp1);
    if (mlib1phdata.ISigns[mlib1phdata.CurToUse] == -1)
    {
      disp_string[0] = '-';
    }
    ON_A();
    ON_DP3();
    break;

  case DISP_POWER_ACT:
    DisplayActPower();
    break; 

  case DISP_POWER_REACT:
    DisplayReactPower();
    break;
    
  case DISP_POWER_APP:  
    DisplayAppPower();
    break;
    
  case DISP_FREQ:
   if (MainsOn == FALSE)
    {
      mlib1phdata.Frequency = 0;
    }
    sprintf(disp_string,"  %04ld1n",(uint32)(mlib1phdata.Frequency*100));
    ON_DP3();
    break;

  case DISP_PF:
    
    if((mlib1phdata.PowerFactors[mlib1phdata.CurToUse] < 0.90) && (mlib1phdata.PowerFactors[mlib1phdata.CurToUse] > 0.10))
    {
      if(mlib1phdata.ReactPowers[mlib1phdata.CurToUse] < 0)
      {
        sprintf(disp_string,"LD %03ld1n",(uint32)(mlib1phdata.PowerFactors[mlib1phdata.CurToUse]*100));  
      }
      else
      {
        sprintf(disp_string,"LG %03ld1n",(uint32)(mlib1phdata.PowerFactors[mlib1phdata.CurToUse]*100));
      }
    }
    else
    sprintf(disp_string,"PF %03ld1n",(uint32)(mlib1phdata.PowerFactors[mlib1phdata.CurToUse]*100));
    ON_DP3();
    break;

  case DISP_TIME:
    sprintf(disp_string,"%02ld%02ld%02ld",(uint32_t)g_stTime.m_ucHr,\
              (uint32_t)g_stTime.m_ucMin, (uint32_t)g_stTime.m_ucSec);
    ON_TIME();
    ON_COL1();
    ON_COL2();
    break;

  case DISP_DATE:
    temp1 = g_stDate.m_ucYear - 2000;
    sprintf(disp_string,"%02ld%02ld%02ld", g_stDate.m_ucDay, g_stDate.m_ucMonth, temp1);
    ON_DATE();
    ON_COL1();
    ON_COL2();
    break; 
 
  case DISP_METER_SNO:
    NVReadIIC(SRNOAddr, (uint8 *)disp_string, zSRNO);
    disp_string[7] = disp_string[6];
    disp_string[6] = ' ';
    break;
       
  case DISP_METER_REVNO:
    sprintf(disp_string,"REU%3d",(uint32)VER_NO);
    ON_DP3();
    break;
    
  case DISP_MANU_DATE:
    NVReadIIC(MeterManuAddr, (uint8 *)&temp1, 2);
    temp2 = (temp1 >> 8) & 0xFF;
    temp1 &= 0xFF;
    sprintf(disp_string,"MF%02ld%02ld", temp1, temp2);
    ON_DATE();
    ON_COL1();
    ON_COL2();
    break;

  default:
    break;
  }
  if (DispParam != DISP_ALL_ON)
  {
    for (Index = 0; Index < NUM_DIGITS; Index++)
    {
      if (disp_string[Index] == 0)
      {
        disp_string[Index] = 0x20;
      }
    }

    if (mlib1phdata.MetOnImax == TRUE)
    {
      ON_MAG();
    }
    else
    {
      if ((mlib1phdata.ISigns[0] == -1) || (mlib1phdata.ISigns[1] == -1))
      {
        ON_REV();
      }
    }
    if ((SystemState.PowerMode != POWERMODE_MAINS) && (mlib1phdata.Irms[mlib1phdata.CurToUse] > 0.45))
    {
      ON_NM();
    }
    LCDClear();
    lcd_PrintString( );
  }
  
  return;
}

/*!
 * @brief Displays active power kW.
 */
static void DisplayActPower(void)
{
  uint32 Temp;
  Temp = (uint32)(mlib1phdata.ActPowers[mlib1phdata.CurToUse]);
  sprintf(disp_string,"%05ld1n", Temp);
  ON_DP1();
  ON_kW();
}

/*!
 * @brief Displays reactive power kVAR.
 */
static void DisplayReactPower(void)
{
  sprintf(disp_string,"%05ld 1n",(uint32)(mlib1phdata.ReactPowers[mlib1phdata.CurToUse]));
  ON_kVA();
  ON_DP1();
}

/*!
 * @brief Displays apparent power kVA.
 */
static void DisplayAppPower(void)
{
  sprintf(disp_string,"%05ld 1n",(uint32)(mlib1phdata.AppPowers[mlib1phdata.CurToUse]));
  ON_kVA();
  ON_DP1();
}

/*!
 * @brief Displays "Erase" during Memory erase operation.
 */
void DisplayErase(void)
{
  uint16 i;
  for (i = 0; i < NUM_FRONTPLANEPINS; i++)
  {
    LCDArray[i] = 0;
  }

  LCDClear();
  sprintf(disp_string,"ERASE   ");
  lcd_PrintString( );
}

/*!
 * @brief Displays "Done".
 */
void DisplayDone(void)
{
  uint16 i;
  
  for (i = 0; i < NUM_FRONTPLANEPINS; i++)
  {
    LCDArray[i] = 0;
  }
  LCDClear();
  sprintf(disp_string,"DONE");
}
