/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright (c) 2016-2021, NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */
#include "lcd.h"
#include "common.h"
#include "string.h"
#include "fsl_clock.h"
#include "fsl_slcd.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/

/*******************************************************************************
* Prototypes
******************************************************************************/
slcd_config_t lcdconfig;
slcd_clock_config_t slcdClkConfig = {kSLCD_DefaultClk, kSLCD_AltClkDivFactor1, kSLCD_ClkPrescaler00
#if FSL_FEATURE_SLCD_HAS_FAST_FRAME_RATE
                                     ,
                                     false
#endif
};

void LCDInit(void);
void LCDClear(void);
void LCDScrollNumbersAndAlphabetbets(unsigned char digit);

/*******************************************************************************
 * Code
 ******************************************************************************/
/*!
 * @brief Initializes LCD module in the function driver level.
 */
void LCDInit(void)
{ 
  /* SLCD get default configure. */
  /*
  * lcdconfig.displayMode = kSLCD_NormalMode;
  * lcdconfig.powerSupply = kSLCD_InternalVll3UseChargePump;
  * lcdconfig.voltageTrim = kSLCD_RegulatedVolatgeTrim08;
  * lcdconfig.lowPowerBehavior = kSLCD_EnabledInWaitStop;
  * lcdconfig.frameFreqIntEnable = false;
  * lcdconfig.faultConfig = NULL;
  */
  SLCD_GetDefaultConfig(&lcdconfig);
  
  /* Verify and Complete the configuration structure. */
  lcdconfig.clkConfig          = &slcdClkConfig;
  lcdconfig.loadAdjust         = kSLCD_HighLoadOrSlowestClkSrc;
  lcdconfig.dutyCycle          = (slcd_duty_cycle_t)(LCD_NO_OF_BACK_PLANES - 1);
  lcdconfig.faultConfig        = NULL;
  config_lcd_mode();
  
  /* SLCD Initialize. */
  SLCD_Init(LCD, &lcdconfig);
  
  SLCD_SetBackPlanePhase(LCD, BackPlanePins[0], kSLCD_PhaseHActivate);
  SLCD_SetBackPlanePhase(LCD, BackPlanePins[1], kSLCD_PhaseGActivate);
  SLCD_SetBackPlanePhase(LCD, BackPlanePins[2], kSLCD_PhaseFActivate);
  SLCD_SetBackPlanePhase(LCD, BackPlanePins[3], kSLCD_PhaseEActivate);
  SLCD_SetBackPlanePhase(LCD, BackPlanePins[4], kSLCD_PhaseDActivate);
  SLCD_SetBackPlanePhase(LCD, BackPlanePins[5], kSLCD_PhaseCActivate);
  SLCD_SetBackPlanePhase(LCD, BackPlanePins[6], kSLCD_PhaseBActivate);
  SLCD_SetBackPlanePhase(LCD, BackPlanePins[7], kSLCD_PhaseAActivate);
  
  /* Starts SLCD display. */
  SLCD_StartDisplay(LCD);
}

/*!
 * @brief Configures LCD mode.
 */
void config_lcd_mode(void)
{
  uint8 i;

  lcdconfig.slcdLowPinEnabled = 0;
  lcdconfig.slcdHighPinEnabled = 0;
  lcdconfig.backPlaneLowPin = 0;
  lcdconfig.backPlaneHighPin = 0;

  for (i = 0; i < LCD_NO_OF_FRONT_PLANES; i++)
  {
    if(FrontPlanePins[i] < 32)
      lcdconfig.slcdLowPinEnabled|=(0x00000001) << FrontPlanePins[i];
    else
      lcdconfig.slcdHighPinEnabled|=(0x00000001) << (FrontPlanePins[i]-32);
  }

  i = 0;
  for (i = 0; i < LCD_NO_OF_BACK_PLANES; i++)
  {
    if(BackPlanePins[i] < 32)
    {
      lcdconfig.backPlaneLowPin|=0x00000001 << BackPlanePins[i];
      lcdconfig.slcdLowPinEnabled|=(0x00000001) << BackPlanePins[i];
    }
    else
    {
      lcdconfig.backPlaneHighPin|=0x00000001 << (BackPlanePins[i]-32);
      lcdconfig.slcdHighPinEnabled|=(0x00000001) << (BackPlanePins[i]-32);
    }
  }
}

/*!
 * @brief Clears the LCD screen.
 */
void LCDClear(void)
{
  LCD_ALLFP_OFF      //all segments are off
}

/*!
 * @brief Prints a string to the LCD.
 */
void lcd_PrintString(void)
{
  uint8 i;
  uint8 table_pos;

  for (i = 0; i < NUM_DIGITS; i++)
  {
    table_pos = LCDFindChar(disp_string[i]);
    LCDArray[i+2] |= lcd_char_table[table_pos];
  }
  LCDPIN9  = LCDArray[0];
  LCDPIN10 = LCDArray[1];
  LCDPIN11 = LCDArray[2];
  LCDPIN12 = LCDArray[3];
  LCDPIN13 = LCDArray[4];
  LCDPIN14 = LCDArray[5];
  LCDPIN15 = LCDArray[6];
  LCDPIN16 = LCDArray[7];
  LCDPIN17 = LCDArray[8];
  LCDPIN18 = LCDArray[9];
  LCDPIN19 = LCDArray[10];
  LCDPIN20 = LCDArray[11];
  LCDPIN21 = LCDArray[12];
  LCDPIN22 = LCDArray[13];
}
