/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright (c) 2016-2019, NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef APPSTRUCTURES_H
#define APPSTRUCTURES_H


#include "common.h"
/*******************************************************************************
 * Definitions
 ******************************************************************************/

/******************************************************************************
* SIM system 32Khz Oscillator clock select for LPTMR , CLKOUT , LCD , EWM, WDOG
*
*//*! @addtogroup sim_32Khz_source
* @{
*******************************************************************************/
#define RTC_32KHZ	     (uint8)0x00   ///< divide by 01
#define OSC_32KHZ        (uint8)0x01   ///< divide by 02
#define MCG_IRCLK	     (uint8)0x02   ///< divide by 03
#define LPO_1KHZ         (uint8)0x03   ///< divide by 04

/*! @} End of sim_32Khz_source                                                */


/******************************************************************************
* SIM default configurations used by SIM_init() function
*
*//*! @addtogroup sim_config
* @{
*******************************************************************************/
/**************************************************************************/ /*!
 * @brief   Full configuration.
 * @details Configures System Integration Module with the following characteristics:
 *  - All peripherals clocked.
 *  - Selects RTC 32 kHz clock source for LPTMR, CLKOUT, LCD, EWM and WDOG.
 *  - Bitline precharge of system SRAM disabled during VLPR and VLPW modes.
 *  - Source bias of System SRAM enabled during VLPR and VLPW modes.
 *  - CMP[1] output selected as LPTMR input[0].
 *  - LP timer channel3 sourced from PTD5.
 *  - LP timer channel2 sourced from PTD6.
 *  - LP timer channel1 sourced from PTE4.
 *  - Clock out Select Disabled.
 *  - NMI Disabled (enabled after POR)
 *  - Self-time adjust for System RAM enabled.
 *  - All ports are clocked.
 *  - Below Peripheral are not clocked.
      a) UART0,UART2,UART3
      b) I2C0
      d) All Low power Timer
      e)AFE Channel 4
      f) All PIT
      g) SAR ADC
      h) RNGA
      i) MPU
      j) SPI0, SPI1

 *  - VrefBuffer enabled.
 *  - Internal Reference selected as Buffer Input.
 *  - VrefBuffer does not drive PAD.
*  - SysClock, BusClock and Flask Clock ratio is 1:1:1
 *  - RTC OSC_32K clock selected as the RTC clock source.
 *  - Bus Clock selected as timer Channel3 Primary Count Source.
 *  - Bus Clock selected as timer Channel2 Primary Count Source.
 *  - Bus Clock selected as timer Channel1 Primary Count Source.
 *  - Bus Clock selected as timer Channel0 Primary Count Source.
 *  - (XBAR) Output[8] drives timer Channel3 Secondary Count Source.
 *  - (XBAR) Output[7] drives timer Channel3 Secondary Count Source.
 *  - (XBAR) Output[6] drives timer Channel3 Secondary Count Source.
 *  - (XBAR) Output[5] drives timer Channel3 Secondary Count Source.
 *  - Bus Clock selected as source for the Timer CH0.
 *  - External watchdog monitor input sourced from PTE2 or PTE4
 *  - PIT0[0] drives XBAR_IN[16].
 *  - IR modulation disabled on all UARTs.
 *  - AFE CLK PAD is input.
 *  - AFE clocked from MCG PLL Clock Source.
 *  - DMA Done flag driven by DMA0.
 *  - Sigma Delta Modulator 0 data signal drives XBAR_IN[3].
 *****************************************************************************/
#define SIM_MODULE_CONFIG_OPTIMIZED_MODE_ONE                                   \
(tSIM){                                                                         \
/* SOPT1    */ SET(SIM_SOPT1_OSC32KSEL(0)),                                     \
/* SOPT1_CFG*/ CLR(SIM_SOPT1_CFG_RAMBPEN_MASK)|CLR(SIM_SOPT1_CFG_RAMSBDIS_MASK)|\
               CLR(SIM_SOPT1_CFG_CMPOLPTMR0SEL_MASK)|                           \
               SET(SIM_SOPT1_CFG_LPTMRSEL3(0))|SET(SIM_SOPT1_CFG_LPTMRSEL2(0))| \
               SET(SIM_SOPT1_CFG_LPTMRSEL1(0)),                                 \
/* CTRL_REG */ SET(SIM_CTRL_REG_CLKOUT(0))|                                     \
               SET(SIM_CTRL_REG_ADCTRGSEL(0))|                                  \
               CLR(SIM_CTRL_REG_PTC2_HD_EN_MASK)|                               \
               CLR(SIM_CTRL_REG_PLL_VLP_EN_MASK)|SET(SIM_CTRL_REG_NMIDIS_MASK), \
/* SCGC4    */ CLR(SIM_SCGC4_SPI1_MASK)|CLR(SIM_SCGC4_SPI0_MASK)|               \
               SET(SIM_SCGC4_CMP_MASK)|                                         \
               SET(SIM_SCGC4_VREF_MASK)|SET(SIM_SCGC4_UART3_MASK)|              \
               CLR(SIM_SCGC4_UART2_MASK)|SET(SIM_SCGC4_UART1_MASK)|             \
               SET(SIM_SCGC4_UART0_MASK)|SET(SIM_SCGC4_I2C1_MASK)|              \
               SET(SIM_SCGC4_I2C0_MASK)|                                        \
               CLR(SIM_SCGC4_EWM_MASK),                                         \
/* SCGC5    */ SET(SIM_SCGC5_TMR3_MASK)|SET(SIM_SCGC5_TMR2_MASK)|               \
               SET(SIM_SCGC5_TMR1_MASK)|SET(SIM_SCGC5_TMR0_MASK)|               \
               SET(SIM_SCGC5_XBAR_MASK)|                                        \
               SET(SIM_SCGC5_RTCREG_MASK)|SET(SIM_SCGC5_RTC_MASK)|              \
               SET(SIM_SCGC5_PORTI_MASK)|SET(SIM_SCGC5_PORTH_MASK)|             \
               SET(SIM_SCGC5_PORTG_MASK)|SET(SIM_SCGC5_PORTF_MASK)|             \
               SET(SIM_SCGC5_PORTE_MASK)|SET(SIM_SCGC5_PORTD_MASK)|             \
               SET(SIM_SCGC5_PORTC_MASK)|SET(SIM_SCGC5_PORTB_MASK)|             \
               SET(SIM_SCGC5_PORTA_MASK)|SET(SIM_SCGC5_SLCD_MASK),              \
/* SCGC6    */ SET(SIM_SCGC6_LPTMR0_MASK)|SET(SIM_SCGC6_CRC_MASK)|               \
               SET(SIM_SCGC6_AFE_MASK)|SET(SIM_SCGC6_PIT1_MASK)|                \
               SET(SIM_SCGC6_PIT0_MASK)|SET(SIM_SCGC6_ADC_MASK)|                \
               SET(SIM_SCGC6_RNGA_MASK)|SET(SIM_SCGC6_DMACHMUX_MASK)|            \
               SET(SIM_SCGC6_FTFA_MASK),            \
/* SCGC7    */ SET(SIM_SCGC7_DMA_MASK)|CLR(SIM_SCGC7_MPU_MASK),                 \
/* CLKDIV1  */ SET(SIM_CLKDIV1_CLKDIVSYS(SYSCLK_DIV1))|                         \
               SET(SIM_CLKDIV1_CLKDIVBUS(0x01))|                                \
               CLR(SIM_CLKDIV1_FLASHCLKMODE_MASK),                              \
/* FCCFG1   */ CLR(SIM_FCFG1_FLASHDOZE_MASK)|CLR(SIM_FCFG1_FLASHDIS_MASK),      \
/* MISC_CTL */ CLR(SIM_MISC_CTL_VREFBUFPD_MASK)|                                \
               CLR(SIM_MISC_CTL_VREFBUFINSEL_MASK)|                             \
               CLR(SIM_MISC_CTL_VREFBUFOUTEN_MASK)|                             \
               CLR(SIM_MISC_CTL_RTCCLKSEL_MASK)|                                \
               CLR(SIM_MISC_CTL_TMR3PCSSEL(0))|                                 \
               CLR(SIM_MISC_CTL_TMR2PCSSEL(0))|                                 \
               CLR(SIM_MISC_CTL_TMR1PCSSEL(0))|                                 \
               CLR(SIM_MISC_CTL_TMR0PCSSEL(0))|                                 \
               SET(SIM_MISC_CTL_TMR3SCSEL_MASK)|                                \
               SET(SIM_MISC_CTL_TMR2SCSEL_MASK)|                                \
               SET(SIM_MISC_CTL_TMR1SCSEL_MASK)|                                \
               SET(SIM_MISC_CTL_TMR0SCSEL_MASK)|                                \
               CLR(SIM_MISC_CTL_TMR0PLLCLKSEL_MASK)|                            \
               CLR(SIM_MISC_CTL_EWMINSEL_MASK)|                                 \
               CLR(SIM_MISC_CTL_XBARPITOUTSEL(0))|                              \
               CLR(SIM_MISC_CTL_UART3IRSEL_MASK)|                               \
               CLR(SIM_MISC_CTL_UART2IRSEL_MASK)|                               \
               CLR(SIM_MISC_CTL_UART1IRSEL_MASK)|                               \
               CLR(SIM_MISC_CTL_UART0IRSEL_MASK)|                               \
               CLR(SIM_MISC_CTL_UARTMODTYPE_MASK)|                              \
               CLR(SIM_MISC_CTL_AFECLKPADDIR_MASK)|                             \
               SET(SIM_MISC_CTL_AFECLKSEL(0))|                                  \
               SET(SIM_MISC_CTL_DMADONESEL(0)),                                 \
}

#define SIM_MODULE_CONFIG_OPTIMIZED_MODE_BAT                                   \
(tSIM){                                                                         \
/* SOPT1    */ SET(SIM_SOPT1_OSC32KSEL(0)),                                     \
/* SOPT1_CFG*/ CLR(SIM_SOPT1_CFG_RAMBPEN_MASK)|CLR(SIM_SOPT1_CFG_RAMSBDIS_MASK)|\
               CLR(SIM_SOPT1_CFG_CMPOLPTMR0SEL_MASK)|                           \
               SET(SIM_SOPT1_CFG_LPTMRSEL3(0))|SET(SIM_SOPT1_CFG_LPTMRSEL2(0))| \
               SET(SIM_SOPT1_CFG_LPTMRSEL1(0)),                                 \
/* CTRL_REG */ SET(SIM_CTRL_REG_CLKOUT(0))|                                     \
               SET(SIM_CTRL_REG_ADCTRGSEL(0))|                                  \
               CLR(SIM_CTRL_REG_PTC2_HD_EN_MASK)|                               \
               CLR(SIM_CTRL_REG_PLL_VLP_EN_MASK)|SET(SIM_CTRL_REG_NMIDIS_MASK), \
/* SCGC4    */ CLR(SIM_SCGC4_SPI1_MASK)|CLR(SIM_SCGC4_SPI0_MASK)|               \
               SET(SIM_SCGC4_CMP_MASK)|                                          \
               SET(SIM_SCGC4_VREF_MASK)|SET(SIM_SCGC4_UART3_MASK)|              \
               CLR(SIM_SCGC4_UART2_MASK)|SET(SIM_SCGC4_UART1_MASK)|             \
               SET(SIM_SCGC4_UART0_MASK)|SET(SIM_SCGC4_I2C1_MASK)|              \
               SET(SIM_SCGC4_I2C0_MASK)|                                         \
               CLR(SIM_SCGC4_EWM_MASK),                                         \
/* SCGC5    */ SET(SIM_SCGC5_TMR3_MASK)|SET(SIM_SCGC5_TMR2_MASK)|               \
               SET(SIM_SCGC5_TMR1_MASK)|SET(SIM_SCGC5_TMR0_MASK)|               \
               SET(SIM_SCGC5_XBAR_MASK)|                                         \
               SET(SIM_SCGC5_RTCREG_MASK)|SET(SIM_SCGC5_RTC_MASK)|              \
               SET(SIM_SCGC5_PORTI_MASK)|SET(SIM_SCGC5_PORTH_MASK)|             \
               SET(SIM_SCGC5_PORTG_MASK)|SET(SIM_SCGC5_PORTF_MASK)|             \
               SET(SIM_SCGC5_PORTE_MASK)|SET(SIM_SCGC5_PORTD_MASK)|             \
               SET(SIM_SCGC5_PORTC_MASK)|SET(SIM_SCGC5_PORTB_MASK)|             \
               SET(SIM_SCGC5_PORTA_MASK)|SET(SIM_SCGC5_SLCD_MASK),              \
/* SCGC6    */ SET(SIM_SCGC6_LPTMR0_MASK)|SET(SIM_SCGC6_CRC_MASK)|               \
               SET(SIM_SCGC6_AFE_MASK)|SET(SIM_SCGC6_PIT1_MASK)|                \
               SET(SIM_SCGC6_PIT0_MASK)|SET(SIM_SCGC6_ADC_MASK)|                \
               SET(SIM_SCGC6_RNGA_MASK)|SET(SIM_SCGC6_DMACHMUX_MASK)|           \
               SET(SIM_SCGC6_FTFA_MASK),            							\
/* SCGC7    */ CLR(SIM_SCGC7_DMA_MASK)|CLR(SIM_SCGC7_MPU_MASK),                 \
/* CLKDIV1  */ SET(SIM_CLKDIV1_CLKDIVSYS(SYSCLK_DIV1))|                         \
               SET(SIM_CLKDIV1_CLKDIVBUS(0x01))|                                \
               CLR(SIM_CLKDIV1_FLASHCLKMODE_MASK),                              \
/* FCCFG1   */ CLR(SIM_FCFG1_FLASHDOZE_MASK)|CLR(SIM_FCFG1_FLASHDIS_MASK),      \
/* MISC_CTL */ SET(SIM_MISC_CTL_VREFBUFPD_MASK)|                                \
               CLR(SIM_MISC_CTL_VREFBUFINSEL_MASK)|                             \
               CLR(SIM_MISC_CTL_VREFBUFOUTEN_MASK)|                             \
               CLR(SIM_MISC_CTL_RTCCLKSEL_MASK)|                                \
               CLR(SIM_MISC_CTL_TMR3PCSSEL(0))|                                 \
               CLR(SIM_MISC_CTL_TMR2PCSSEL(0))|                                 \
               CLR(SIM_MISC_CTL_TMR1PCSSEL(0))|                                 \
               CLR(SIM_MISC_CTL_TMR0PCSSEL(0))|                                 \
               SET(SIM_MISC_CTL_TMR3SCSEL_MASK)|                               \
               SET(SIM_MISC_CTL_TMR2SCSEL_MASK)|                               \
               SET(SIM_MISC_CTL_TMR1SCSEL_MASK)|                               \
               SET(SIM_MISC_CTL_TMR0SCSEL_MASK)|                               \
               CLR(SIM_MISC_CTL_TMR0PLLCLKSEL_MASK)|                            \
               CLR(SIM_MISC_CTL_EWMINSEL_MASK)|                                 \
               CLR(SIM_MISC_CTL_XBARPITOUTSEL(0))|                              \
               CLR(SIM_MISC_CTL_UART3IRSEL_MASK)|                               \
               CLR(SIM_MISC_CTL_UART2IRSEL_MASK)|                               \
               CLR(SIM_MISC_CTL_UART1IRSEL_MASK)|                               \
               CLR(SIM_MISC_CTL_UART0IRSEL_MASK)|                               \
               CLR(SIM_MISC_CTL_UARTMODTYPE_MASK)|                              \
               CLR(SIM_MISC_CTL_AFECLKPADDIR_MASK)|                             \
               SET(SIM_MISC_CTL_AFECLKSEL(0))|                                  \
               SET(SIM_MISC_CTL_DMADONESEL(0)),                                 \
}

/**************************************************************************/ /*!
 * @brief UART configured to operate in interrupt mode.
 * DMA is enabled
 * @param   brate     Baud rate.
 * @param   bclk      Buss clock in Hz.
 *****************************************************************************/
#define UART_MODULE_INTRMODE_DMA_CONFIG(brate,bclk)                     \
(tUART){                                                                      \
/* BDH      */ CLR(UART_BDH_RXEDGIE_MASK)|((CALC_SBR(brate,bclk)>>8)&0x1f),   \
/* BDL      */ ((CALC_SBR(brate,bclk)>>0)&0xff),                              \
/* C1       */ CLR(UART_C1_LOOPS_MASK)|CLR(UART_C1_RSRC_MASK)|                \
               CLR(UART_C1_M_MASK)|CLR(UART_C1_WAKE_MASK)|                    \
               CLR(UART_C1_ILT_MASK)|CLR(UART_C1_PE_MASK)|                    \
               CLR(UART_C1_PT_MASK),                                          \
/* C2       */ CLR(UART_C2_TIE_MASK)|CLR(UART_C2_TCIE_MASK)|                  \
               SET(UART_C2_RIE_MASK)|CLR(UART_C2_ILIE_MASK)|                  \
               SET(UART_C2_TE_MASK)|SET(UART_C2_RE_MASK)|                     \
               CLR(UART_C2_RWU_MASK)|CLR(UART_C2_SBK_MASK),                   \
/* S2       */ CLR(UART_S2_RXEDGIF_MASK)|CLR(UART_S2_MSBF_MASK)|              \
               CLR(UART_S2_RXINV_MASK)|CLR(UART_S2_RWUID_MASK)|               \
               CLR(UART_S2_BRK13_MASK),                                       \
/* C3       */ CLR(UART_C3_T8_MASK)|CLR(UART_C3_TXDIR_MASK)|                  \
               CLR(UART_C3_TXINV_MASK)|CLR(UART_C3_ORIE_MASK)|                \
               CLR(UART_C3_NEIE_MASK)|CLR(UART_C3_FEIE_MASK)|                 \
               CLR(UART_C3_PEIE_MASK),                                        \
/* MA1      */ SET(UART_MA1_MA(0x00)),                                        \
/* MA2      */ SET(UART_MA2_MA(0x00)),                                        \
/* C4       */ CLR(UART_C4_MEAN1(0))|CLR(UART_C4_MEAN1_MASK)|                 \
               CLR(UART_C4_M10_MASK)|((CALC_BRFA(brate,bclk)>>0)&0x1f),       \
/* C5       */ SET(UART_C5_TDMAS_MASK)|CLR(UART_C5_RDMAS_MASK),               \
/* C7816    */ CLR(UART_C78160_ONACK_MASK)|CLR(UART_C78160_ANACK_MASK)|       \
               CLR(UART_C78160_INIT_MASK)|CLR(UART_C78160_TTYPE_MASK)|        \
               CLR(UART_C78160_ISO_7816E_MASK),                               \
/* IE7816   */ CLR(UART_IE7816_WTE_MASK)|CLR(UART_IE7816_CWTE_MASK)|          \
               CLR(UART_IE7816_BWTE_MASK)|CLR(UART_IE7816_INITDE_MASK)|       \
               CLR(UART_IE7816_GTVE_MASK)|CLR(UART_IE7816_TXTE_MASK)|         \
               CLR(UART_IE7816_RXTE_MASK),                                    \
/* IS7816   */ CLR(UART_IS7816_WT_MASK)|CLR(UART_IS7816_CWT_MASK)|            \
               CLR(UART_IS7816_BWT_MASK)|CLR(UART_IS7816_INITD_MASK)|         \
               CLR(UART_IS7816_GTV_MASK)|CLR(UART_IS7816_TXT_MASK)|           \
               CLR(UART_IS7816_RXT_MASK),                                     \
/* WP7816T0 */ SET(UART_WP7816_T_TYPE0_WI(0x0a)),                             \
/* WP7816T1 */ SET(UART_WP7816_T_TYPE1_CWI(0x00))|                            \
               SET(UART_WP7816_T_TYPE1_BWI(0x0a)),                            \
/* WN7816   */ SET(UART_WN7816_GTN(0x00)),                                    \
/* WF7816   */ SET(UART_WF7816_GTFD(0x00)),                                   \
/* ET7816   */ SET(UART_ET7816_TXTHRESHOLD(0x00))|                            \
               SET(UART_ET7816_RXTHRESHOLD(0x00)),                            \
/* TL7816   */ SET(UART_TL7816_TLEN(0x00))                                    \
}
/**************************************************************************/ /*!
  * @brief Comparator setting:
  * - filter enable 7 counts,
  * - module enable after initialization,
  * - hi speed mode,
  * - irq. Disable
  * @param   dac       DAC output voltage select from 0 to 63
  *****************************************************************************/
#define CMP_FILTER7_MODULE_EN_SAMPLE_PER_255_IRQ_DIS_CONFIG(dac)               \
(tCMP){                                                                       \
/* CR0    */  SET(CMP_CR0_FILTER_CNT(7)) |SET(CMP_CR0_HYSTCTR(0)),            \
/* CR1    */  CLR(CMP_CR1_SE_MASK)       |CLR(CMP_CR1_WE_MASK)|               \
              CLR(CMP_CR1_TRIGM_MASK)    |SET(CMP_CR1_PMODE_MASK)|            \
              CLR(CMP_CR1_INV_MASK)      |CLR(CMP_CR1_COS_MASK)|              \
              CLR(CMP_CR1_OPE_MASK)      |SET(CMP_CR1_EN_MASK),               \
/* FPR    */  SET(CMP_FPR_FILT_PER(0xFF)),                                    \
/* SCR    */  CLR(CMP_SCR_DMAEN_MASK)    |CLR(CMP_SCR_IER_MASK)|              \
              CLR(CMP_SCR_IEF_MASK),                                          \
/* DACCR  */  CLR(CMP_DACCR_DACEN_MASK)  |CLR(CMP_DACCR_VRSEL_MASK)|dac       \
}
/**************************************************************************/ /*!
  * @brief Comparator setting:
  * - filter enable 7 counts,
  * - module enable after initialization,
  * - hi speed mode,
  * - irq. Disable
  * @param   dac       DAC output voltage select from 0 to 63
  *****************************************************************************/
#define CMP_FILTER7_MODULE_EN_SAMPLE_PER_255_IRQ_DIS_LP_DAC_EN_CONFIG(dac)    \
(tCMP){                                                                       \
/* CR0    */  SET(CMP_CR0_FILTER_CNT(7)) |SET(CMP_CR0_HYSTCTR(0)),            \
/* CR1    */  CLR(CMP_CR1_SE_MASK)       |CLR(CMP_CR1_WE_MASK)|               \
              CLR(CMP_CR1_TRIGM_MASK)    |CLR(CMP_CR1_PMODE_MASK)|            \
              CLR(CMP_CR1_INV_MASK)      |CLR(CMP_CR1_COS_MASK)|              \
              CLR(CMP_CR1_OPE_MASK)      |SET(CMP_CR1_EN_MASK),               \
/* FPR    */  SET(CMP_FPR_FILT_PER(0xff)),                                    \
/* SCR    */  CLR(CMP_SCR_DMAEN_MASK)    |CLR(CMP_SCR_IER_MASK)|              \
              CLR(CMP_SCR_IEF_MASK),                                          \
/* DACCR  */  SET(CMP_DACCR_DACEN_MASK)  |SET(CMP_DACCR_VRSEL_MASK)|dac       \
}

/**************************************************************************/ /*!
  * @brief Comparator setting:
  * - filter enable 7 counts,
  * - module enable after initialization,
  * - hi speed mode,
  * - irq. Disable
  *****************************************************************************/
#define CMP_FILTER7_MODULE_EN_SAMPLE_PER_255_IRQ_DIS_LP_DAC_DI_CONFIG         \
(tCMP){                                                                       \
/* CR0    */  SET(CMP_CR0_FILTER_CNT(7)) |SET(CMP_CR0_HYSTCTR(0)),            \
/* CR1    */  CLR(CMP_CR1_SE_MASK)       |CLR(CMP_CR1_WE_MASK)|               \
              CLR(CMP_CR1_TRIGM_MASK)    |CLR(CMP_CR1_PMODE_MASK)|            \
              CLR(CMP_CR1_INV_MASK)      |CLR(CMP_CR1_COS_MASK)|              \
              CLR(CMP_CR1_OPE_MASK)      |SET(CMP_CR1_EN_MASK),               \
/* FPR    */  SET(CMP_FPR_FILT_PER(0xff)),                                    \
/* SCR    */  CLR(CMP_SCR_DMAEN_MASK)    |CLR(CMP_SCR_IER_MASK)|              \
              CLR(CMP_SCR_IEF_MASK),                                          \
/* DACCR  */  CLR(CMP_DACCR_DACEN_MASK)  |CLR(CMP_DACCR_VRSEL_MASK)           \
}

/**************************************************************************/ /*!
 * @brief Minimum configuration with PTD0 (LLWU_P11) enabled as wakeup source 
 *        (falling edge active). 
 *****************************************************************************/
#define LLWU_MODULE_RTC_CMP1_P4_P10_ENABLED                                  \
(tLLWU){                                                                      \
/* PE1  */ SET(LLWU_PE1_WUPE0 (0x00))|SET(LLWU_PE1_WUPE1 (0x00))|             \
           SET(LLWU_PE1_WUPE2 (0x00))|SET(LLWU_PE1_WUPE3 (0x00)),             \
/* PE2  */ SET(LLWU_PE2_WUPE4 (0x02))|SET(LLWU_PE2_WUPE5 (0x00))|             \
           SET(LLWU_PE2_WUPE6 (0x00))|SET(LLWU_PE2_WUPE7 (0x00)),             \
/* PE3  */ SET(LLWU_PE3_WUPE8 (0x00))|SET(LLWU_PE3_WUPE9 (0x00))|             \
           SET(LLWU_PE3_WUPE10(0x01))|SET(LLWU_PE3_WUPE11(0x00)),             \
/* PE4  */ SET(LLWU_PE4_WUPE12(0x00))|SET(LLWU_PE4_WUPE13(0x00))|             \
           SET(LLWU_PE4_WUPE14(0x00))|SET(LLWU_PE4_WUPE15(0x00)),             \
/* ME   */ CLR(LLWU_ME_WUME0_MASK)   |CLR(LLWU_ME_WUME1_MASK)   |             \
           CLR(LLWU_ME_WUME2_MASK)   |SET(LLWU_ME_WUME3_MASK)   |             \
           SET(LLWU_ME_WUME4_MASK),                                           \
/* FILT1*/ SET(LLWU_FILT1_FILTE(0x00))|SET(LLWU_FILT1_FILTSEL(0x00)),         \
/* FILT1*/ SET(LLWU_FILT2_FILTE(0x00))|SET(LLWU_FILT2_FILTSEL(0x00)),         \
}

/**************************************************************************/ /*!
 * @brief Selects 12-bit software triggered mode with external voltage reference.
 *****************************************************************************/
#define ADC_MODULE_12B_SWTRG_XREF_CONFIG                                      \
(tADC){                                                                       \
/* CFG1   */  CLR(ADC_CFG1_ADLPC_MASK)|SET(ADC_CFG1_ADIV(3))|                 \
              SET(ADC_CFG1_ADLSMP_MASK)|SET(ADC_CFG1_MODE(1))|                \
              SET(ADC_CFG1_ADICLK(0)),                                        \
/* CFG2   */  CLR(ADC_CFG2_MUXSEL_MASK)|CLR(ADC_CFG2_ADACKEN_MASK)|           \
              SET(ADC_CFG2_ADHSC_MASK)|SET(ADC_CFG2_ADLSTS(0)),               \
/* CV1    */  0l,                                                             \
/* CV2    */  0l,                                                             \
/* SC2    */  CLR(ADC_SC2_ADTRG_MASK)|CLR(ADC_SC2_ACFE_MASK)|                 \
              CLR(ADC_SC2_ACFGT_MASK)|CLR(ADC_SC2_ACREN_MASK)|                \
              CLR(ADC_SC2_DMAEN_MASK)|SET(ADC_SC2_REFSEL(2)),                 \
/* SC3    */  CLR(ADC_SC3_CAL_MASK)|CLR(ADC_SC3_ADCO_MASK)|                   \
              CLR(ADC_SC3_AVGE_MASK)|SET(ADC_SC3_AVGS(0)),                    \
}

/***************************************************************************//*!
 * @brief   Reads coarse compensation interval value. 
 * @details This function reads coarse interval value.
 ******************************************************************************/
#define IRTC_ReadCoarseCompIntervalVal()        (uint8)(RTC_COMPEN >>8)

/***************************************************************************//*!
 * @brief   Reads coarse compensation  value. 
 * @details This function reads coarse compensation value.
 ******************************************************************************/
#define IRTC_ReadCoarseCompVal()                 (int8)(RTC_COMPEN)

/***************************************************************************//*!
 * @brief   Reads Fine compensation integer value. 
 * @details This function reads fine compensation integer value.
 ******************************************************************************/
#define IRTC_ReadFineCompIntegerVal()         (int8)(((int16)RTC_COMPEN >>12))

/***************************************************************************//*!
 * @brief   Reads Fine compensation fractional value. 
 * @details This function reads fine compensation fractional value.
 ******************************************************************************/
#define IRTC_ReadFineCompFractionalVal()        (uint8)(((uint16)RTC_COMPEN&0x007f))

#define LPTMR_WAKEUP_SOURCE                     LLWU_MWUF0
#define IRTC_ALARM_WAKEUP_SOURCE                LLWU_MWUF1
#define CMP0_WAKEUP_SOURCE                      LLWU_MWUF2
#define IRTC_INTERRUPT_WAKEUP_SOURCE            LLWU_MWUF3
#define CMP1_WAKEUP_SOURCE                      LLWU_MWUF4

/***************************************************************************//*!
 * @brief   Macro disables LVW Interrupt.
 * @details This macro disables LVW Interrupt.
 * @note    Implemented as inlined macro.
 ******************************************************************************/
#define PMC_ClrLVW()         { PMC_LVDSC2 &= ~PMC_LVDSC2_LVWIE_MASK;  }

/***************************************************************************//*!
 * @brief   Macro Enables LVW Interrupt.
 * @details This macro disables LVW Interrupt.
 * @note    Implemented as inlined macro.
 ******************************************************************************/
#define PMC_SetLVW()         { PMC_LVDSC2 |=  PMC_LVDSC2_LVWIE_MASK;  }

/**************************************************************************/ /*!
 * @brief   Voltage reference switch setting.
 * @details Configures voltage reference switches:
 *        - Buffer enable,
 *        - Buffer does not drive PAD,     S1 = H,
 *        - Internal reference select,     S2 = H,
 *        - AFE Internal reference select, S3 = L,
 ******************************************************************************/
#define VREF_SWITCH_THREE_PHASE_METER                                          \
(tVREF_S){                                                                     \
/* SIM_MISC  */  CLR(SIM_MISC_CTL_VREFBUFPD_MASK)|                             \
                 CLR(SIM_MISC_CTL_VREFBUFINSEL_MASK)|                          \
                 SET(SIM_MISC_CTL_VREFBUFOUTEN_MASK),                          \
/* VERFL_TRM */  CLR(VREF_VREFL_TRM_VREFL_SEL_MASK),                           \
}

/**************************************************************************/ /*!
 * @brief Selects 12-bit hardware triggered mode with internal voltage reference.
 *****************************************************************************/
#define ADC_MODULE_THREE_PHASE_METER_1_POINT_2V_CONFIG                        \
(tADC){                                                                       \
/* CFG1   */  CLR(ADC_CFG1_ADLPC_MASK)|SET(ADC_CFG1_ADIV(2))|                 \
              CLR(ADC_CFG1_ADLSMP_MASK)|SET(ADC_CFG1_MODE(3))|                \
              SET(ADC_CFG1_ADICLK(1)),                                        \
/* CFG2   */  CLR(ADC_CFG2_MUXSEL_MASK)|CLR(ADC_CFG2_ADACKEN_MASK)|           \
              CLR(ADC_CFG2_ADHSC_MASK)|SET(ADC_CFG2_ADLSTS(0)),               \
/* CV1    */  0l,                                                             \
/* CV2    */  0l,                                                             \
/* SC2    */  SET(ADC_SC2_ADTRG_MASK)|CLR(ADC_SC2_ACFE_MASK)|                 \
              CLR(ADC_SC2_ACFGT_MASK)|CLR(ADC_SC2_ACREN_MASK)|                \
              CLR(ADC_SC2_DMAEN_MASK)|SET(ADC_SC2_REFSEL(1)),                 \
/* SC3    */  CLR(ADC_SC3_CAL_MASK)|CLR(ADC_SC3_ADCO_MASK)|                   \
              CLR(ADC_SC3_AVGE_MASK)|SET(ADC_SC3_AVGS(0)),                    \
}

#endif