/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 *
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdbool.h>
#include "board.h"
#include "led.h"
#include "fsl_debug_console.h"
#include "serial_manager.h"
#include "fsl_os_abstraction.h"
#include "fsl_port.h"
#include "fsl_gpio.h"
#include "fsl_common.h"

#include "fsl_device_registers.h"
#include "pin_mux.h"
#include "clock_config.h"

#include "ApplMain.h"
#include "BLE_Abstraction_main.h"
#include "FSCI_main.h"
#include "temperature_sensor.h"
#include "SPI.h"

#include "task.h"
#include "UART_main.h"
/************************************************************************************
*************************************************************************************
* Private macros
*************************************************************************************
************************************************************************************/
#define FSL_RTOS_FREE_RTOS 1
#define MSG_QUEUE_LEN 10
#define MAXIMUM_BUFF_SIZE 128

/* Application Events */
#define gAppEvtMsgFromAbs_c (1 << 0)

/************************************************************************************
*************************************************************************************
* Private type definitions
*************************************************************************************
************************************************************************************/
typedef uint8_t appFSCIMsgType_t;

/* FSCI to Application Connection Message */
typedef struct connectionMsg_tag{
    deviceId_t              deviceId;
    gapConnectionEvent_t    connEvent;
}connectionMsg_t;

typedef struct disconnectionMsg_tag{
	deviceId_t               deviceId;
	gapConnectionEvent_t     disconnEvent;
}disconnectionMsg_t;

/* Host to Application GATT Server Message */
typedef struct gattServerMsg_tag{
    deviceId_t          deviceId;
    gattServerEvent_t   serverEvent;
}gattServerMsg_t;

typedef struct appMsgFromFSCI_tag
{
	appFSCIMsgType_t msgType;
	union {
		gapGenericEvent_t     genericMsg;
		gapAdvertisingEvent_t advmsg;
		connectionMsg_t       connMsg;
		disconnectionMsg_t    disconnMsg;
		gattServerMsg_t		  gattServerMsg;
	}msgData;
}appMsgFromFSCI_t;

typedef enum {
	gAppGapGenericMsg_c = 0,
	gAppGapConnectionMsg_c,
	gAppGapAdvertisementMsg_c,
	gAppGapScanMsg_c,
	gAppGattServerMsg_c,
	gAppGattClientProcedureMsg_c,
	gAppGattClientNotificationMsg_c,
	gAppGattClientIndicationMsg_c,
	gAppCharacteristicCccdWritten_c,
}appFSCIMsgType_tag;

/*******************************************************************************
 * Private memory declarations
 ******************************************************************************/
static uint8_t platformInitialized = 0;

/* Adv State */
//static osa_msgq_handle_t msgqHandle = NULL;
osa_msgq_handle_t rsp_msg_queue = NULL;
osa_event_handle_t eventHandle = NULL;
osa_event_handle_t appEventHandle = NULL;
static gapAdvertisingCallback_t pfAdvCallback = NULL;
static gapConnectionCallback_t pfConnCallback = NULL;
static uint8_t s_ledMonochromeHandleBuffer[1][LED_HANDLE_SIZE];
/*******************************************************************************
 * Private functions Prototypes
*************************************************************************************
************************************************************************************/

static void App_ConnectionCallback(deviceId_t peerDeviceId, gapConnectionEvent_t* pConnectionEvent);
static void APP_AdvertisingCallback(gapAdvertisingEvent_t* pAdvertisingEvent);
static void App_Thread (uint32_t param);
static void App_HandleFSCIMessageInput(appMsgFromFSCI_t *pMsg);
static void Led_Initialization();
/************************************************************************************
*************************************************************************************
* Public memory declarations
*************************************************************************************
************************************************************************************/
TaskHandle_t 		task_handle;
led_flash_config_t 	ledFlash;
led_handle_t 		s_ledMonochromeHandle[1];
led_config_t 		g_ledMonochrome[1] = {
    {
        .type = kLED_TypeRgb,
        .ledRgb =
            {
                .redPin =
                    {
                        .dimmingEnable = 0,
                        .gpio =
                            {
                                1,
                                BOARD_LED_RED_GPIO_PIN,
                                1,
                            },
                    },

				.greenPin =
                    {
                        .dimmingEnable = 0,
                        .gpio =
                            {
                                4,
                                BOARD_LED_GREEN_GPIO_PIN,
                                1,
                            },
                    },

				.bluePin =
                    {
                        .dimmingEnable = 0,
                        .gpio =
                            {
                                1,
                                BOARD_LED_BLUE_GPIO_PIN,
                                1,
                            },
                    },
            },
    },
};
/*! *********************************************************************************
*************************************************************************************
* Public functions prototypes
*************************************************************************************
********************************************************************************** */

/*******************************************************************************
 ********************************************************************************
 * Public functions
 ******************************************************************************
********************************************************************************/

/*****************************************************************************
* \brief  Handles all messages received from the application task.
*
* \param[in] pMsg Pointer to appMsgFromFSCI_t.
*****************************************************************************/
static void App_HandleFSCIMessageInput(appMsgFromFSCI_t *pMsg)
{
	switch ( pMsg->msgType )
	{
		case gAppGapGenericMsg_c:
		{
			BleApp_GenericCallback(&pMsg->msgData.genericMsg);
		}
		break;

		case gAppGapAdvertisementMsg_c:
		{
			if(pfAdvCallback)
				pfAdvCallback(&pMsg->msgData.advmsg);
		}
		break;		

		case gAppGapConnectionMsg_c:
		{
			if(pfConnCallback)
				pfConnCallback(pMsg->msgData.connMsg.deviceId, &pMsg->msgData.connMsg.connEvent);
		}
		break;

		case gAppGattServerMsg_c:
		{
			BleApp_GattServerCallback(pMsg->msgData.gattServerMsg.deviceId, &pMsg->msgData.gattServerMsg.serverEvent);
		}
		break;

		default:
			break;
	}
}

/*! *********************************************************************************
 * \brief  This function represents the Application task.
 *         This task reuses the stack allocated for the MainThread.
 *         This function is called to process all events for the task.
 * \param[in]  argument
 *
 ********************************************************************************** */
static void App_Thread (uint32_t param)
{
	osa_event_flags_t event = 0;

	while(1)
	{
		OSA_EventWait(&appEventHandle, osaEventFlagsAll_c, FALSE, 0, &event);

		if((g_ButtonPress == TRUE) || (event & gAppEvtMsgFromAbs_c) || (g_ButtonLongPress == TRUE) || (is3SecTimerExpired == TRUE))
		{
#ifdef gTempTimerSupported
			if(is3SecTimerExpired == TRUE)
			{
				is3SecTimerExpired = FALSE;
				if(mPeerDeviceId != gInvalidDeviceId_c)
				{
					BleApp_SendTemperature();
				}
			}
#endif

			if(g_ButtonPress == TRUE)
			{
				App_Start();
			}

			if(g_ButtonLongPress == TRUE)
			{
				g_ButtonLongPress = FALSE;

				if (mPeerDeviceId != gInvalidDeviceId_c)
				{
#ifdef gTempTimerSupported
					/* Stop 3 sec timer that sends packet to connected device */
					xTimerStop(sendTempTimerHandle, 0);
#endif

					(void)Gap_Disconnect(mPeerDeviceId);
				}
			}

			appMsgFromFSCI_t *msg_buff = malloc(MAXIMUM_BUFF_SIZE);

			if((OSA_MsgQGet(&rsp_msg_queue, msg_buff, 0)) == KOSA_StatusSuccess)
			{
				if(msg_buff)
				{
					/* Process it */
					App_HandleFSCIMessageInput(msg_buff);
				}
			}

			free(msg_buff);
			msg_buff = NULL;
		}
		vTaskDelay(pdMS_TO_TICKS(900));
	}
}

/*! *********************************************************************************
* \brief        Advertising callback function called when advertising events
* 				coming from FSCI layer.
*
* \param[in]    pAdvertisingEvent Pointer to gapAdvertisingEvent_t.
********************************************************************************** */
static void APP_AdvertisingCallback(gapAdvertisingEvent_t* pAdvertisingEvent)
{
#ifdef DEBUG_PRINT_ENABLE
	PRINTF("at app_adv_cb \r\n");
#endif

	appMsgFromFSCI_t *pMsgIn = NULL;

	pMsgIn = malloc((uint32_t)&(pMsgIn->msgData) + sizeof(gapAdvertisingEvent_t));

	if (pMsgIn == NULL)
	{
		return;
	}

	pMsgIn->msgType = gAppGapAdvertisementMsg_c;
	memcpy(&pMsgIn->msgData.advmsg, pAdvertisingEvent, sizeof(gapAdvertisingEvent_t));

	OSA_MsgQPut(&rsp_msg_queue, pMsgIn);

	/* Signal App_Thread for message into message queue */
	OSA_EventSet(&appEventHandle, gAppEvtMsgFromAbs_c);

	vTaskDelay(pdMS_TO_TICKS(10));

	free(pMsgIn);
	pMsgIn = NULL;
}

/*! *********************************************************************************
* \brief        Connection callback function called when connection events
* 				coming from FSCI layer.
*
* \param[in]    peerDeviceId     Peer Device ID.
* \param[in]    pConnectionEvent Pointer to gapConnectionEvent_t.
********************************************************************************** */
static void App_ConnectionCallback(deviceId_t peerDeviceId, gapConnectionEvent_t* pConnectionEvent)
{
#ifdef DEBUG_PRINT_ENABLE
	PRINTF("At connection cb \r\n");
#endif

	appMsgFromFSCI_t *pMsgIn = NULL;

	pMsgIn = malloc((uint32_t)&(pMsgIn->msgData) + sizeof(connectionMsg_t));

	if(pMsgIn == NULL)
	{
		return;
	}

	pMsgIn->msgType = gAppGapConnectionMsg_c;
	pMsgIn->msgData.connMsg.deviceId = peerDeviceId;
	memcpy(&pMsgIn->msgData.connMsg.connEvent, pConnectionEvent, sizeof(gapConnectionEvent_t));

	OSA_MsgQPut(&rsp_msg_queue, pMsgIn);

	/* Signal App_Thread for message into message queue */
	OSA_EventSet(&appEventHandle, gAppEvtMsgFromAbs_c);

	vTaskDelay(pdMS_TO_TICKS(10));

	free(pMsgIn);
	pMsgIn = NULL;
}

/*! *********************************************************************************
* \brief        Function Starts advertising and register advertise and connection callbacks.
*
* \param[in]    advertisingCallback advertising callback function.
* \param[in]    connectionCallback  connection callback function.
********************************************************************************** */
void App_StartAdvertising(
    gapAdvertisingCallback_t advertisingCallback,
	gapConnectionCallback_t  connectionCallback)
{
	pfAdvCallback = advertisingCallback;
	pfConnCallback = connectionCallback;

	Gap_StartAdvertising(APP_AdvertisingCallback, App_ConnectionCallback);
}

/*! *********************************************************************************
* \brief        Callback function that receives Generic events coming from FSCI layer.
*
* \param[in]    GenericCallback callback function.
********************************************************************************** */
void App_GenericCallback(gapGenericEvent_t *pGenericEvent)
{
	appMsgFromFSCI_t *pMsgIn = NULL;

	pMsgIn = malloc((uint32_t)&(pMsgIn->msgData) + sizeof(gapGenericEvent_t));

	if(pMsgIn == NULL)
	{
		return;
	}

	pMsgIn->msgType = gAppGapGenericMsg_c;
	memcpy(&pMsgIn->msgData.genericMsg, pGenericEvent, sizeof(gapGenericEvent_t));

	OSA_MsgQPut(&rsp_msg_queue, pMsgIn);

	/* Signal App_Thread for message into message queue */
	OSA_EventSet(&appEventHandle, gAppEvtMsgFromAbs_c);

	vTaskDelay(pdMS_TO_TICKS(10));

	free(pMsgIn);
	pMsgIn = NULL;
}

void App_GattServerCallback
(
    deviceId_t          deviceId,
    gattServerEvent_t*  pServerEvent)
{
	appMsgFromFSCI_t *pMsgIn = NULL;

	pMsgIn = malloc((uint32_t)&(pMsgIn->msgData) + sizeof(gattServerMsg_t));

	if(pMsgIn == NULL)
	{
		return;
	}

	pMsgIn->msgType = gAppGattServerMsg_c;
	pMsgIn->msgData.gattServerMsg.deviceId = deviceId;
	memcpy(&pMsgIn->msgData.gattServerMsg.serverEvent, pServerEvent, sizeof(gattServerEvent_t));

	OSA_MsgQPut(&rsp_msg_queue, pMsgIn);

	/* Signal App_Thread for message into message queue */
	OSA_EventSet(&appEventHandle, gAppEvtMsgFromAbs_c);

	vTaskDelay(pdMS_TO_TICKS(10));

	free(pMsgIn);
	pMsgIn = NULL;
}

static void Led_Initialization()
{
    s_ledMonochromeHandle[0] = &s_ledMonochromeHandleBuffer[0];

    ledFlash.times = LED_FLASH_CYCLE_FOREVER;
    ledFlash.period = 10;
    ledFlash.flashType = kLED_FlashOneColor;
    ledFlash.duty = 10;

    if (kStatus_LED_Success != LED_Init(s_ledMonochromeHandle[0], &g_ledMonochrome[0]))
	{
		PRINTF("LED Init failed\r\n");
	}
}

/*! *********************************************************************************
* \brief  This is the first task created by the OS. This task will initialize
*         the system
*
* \param[in]  param
*
********************************************************************************** */
void main_task (uint32_t param)
{
	if(!platformInitialized)
	{
		platformInitialized = 1;
		osa_status_t status;

		task_handle = xTaskGetCurrentTaskHandle();

#ifdef gSPISupported
		BOARD_SPIInitPins();
		SPI_Init();
#endif

#ifdef gUARTSupported
		BOARD_UARTInitPins();
		UARTInit();
#endif

		ADC_Hardware_Init();

		/* create application event */
		status = OSA_EventCreate(&eventHandle, TRUE);
		if(status != KOSA_StatusSuccess)
		{
			return;
		}

		status = OSA_EventCreate(&appEventHandle, TRUE);
		if(status != KOSA_StatusSuccess)
		{
			return;
		}

		status = mBLEAbsInit();
		if(status != KOSA_StatusSuccess)
		{
			return;
		}

		status = FSCIInit();
		if(status != KOSA_StatusSuccess)
		{
			return;
		}

		/* create application message queue */
		status = OSA_MsgQCreate(&rsp_msg_queue, MSG_QUEUE_LEN, MSG_SIZE);
		if(status != KOSA_StatusSuccess)
		{
			return;
		}

		App_config();

		Button_Initialization();

		Led_Initialization();
		PRINTF("\r\n---------- CONFIGURATION SUCCESS ----------\r\n");
		LED_SetColor(s_ledMonochromeHandle[0], kLED_White);
		LED_Flash(s_ledMonochromeHandle[0], &ledFlash);
	}

	App_Thread( param );
}
