/*
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*System includes.*/
#include "ApplMain.h"
#include "BLE_Abstraction_main.h"
#include "ble_FSCI.h"
#include "FSCI_main.h"
#include "SPI.h"
#include "gap_interface.h"
#include "fsl_debug_console.h"
#include "fsl_os_abstraction_free_rtos.h"
#include "gattdb_interface.h"
#include "gatt_server.h"

/* queue handle */
osa_msgq_handle_t tx_msg_queue = NULL;

osa_event_handle_t evt_handle = NULL;

gapGenericCallback_t genericCallback = NULL;

TaskHandle_t recv_task_handle;

typedef PACKED_STRUCT packet_tag
{
	uint8_t    startMarker;
	uint8_t    opGroup;
	uint8_t    opCode;
	uint16_t   len;
}packet_t;

/*******************************************************************************
 * Code
 ******************************************************************************/

/*! *********************************************************************************
* \brief   Callback function to register GAP generic events.
*
* \param[in]   GenericCallback
********************************************************************************** */
void GAP_RegisterGenericCallback (gapGenericCallback_t GenericCallback)
{
	genericCallback = GenericCallback;
}

/*!
 * @brief mBLEAbsInput_init function
 */
osa_status_t mBLEAbsInit(void)
{
	osa_status_t result = kStatus_Success;

	result = OSA_MsgQCreate(&tx_msg_queue, MAX_QUEUE_LENGTH, MSG_SIZE);
    /* Enable queue view in MCUX IDE FreeRTOS TAD plugin. */
    if (result != KOSA_StatusSuccess)
    {
        return result;
    }

    result = OSA_EventCreate(&evt_handle, TRUE);
	if(result != KOSA_StatusSuccess)
	{
		return result;
	}

    return result;
}

/*! *********************************************************************************
* \brief   Handles all messages received from FSCI layer.
*
* \param[in]    pMsg  Pointer to FSCIResponseMsg_t.
********************************************************************************** */
void FSCI_HandleResponse(FSCIResponseMsg_t *pMsg, uint8_t *payload, uint8_t crc)
{
	PRINTF("\r\n RESPONSE : opCode = %X opGroup = %X\r\n", pMsg->opCode, pMsg->opGroup);

	switch(pMsg->opGroup)
	{
		case bleGAP:
		{
			switch(pMsg->opCode)
			{
				case gapConfirm:
				{
					GapConfirm_t *resp = (GapConfirm_t*)payload;
					uint16_t status = resp->status;
					xTaskNotify(task_handle, status, eSetValueWithOverwrite );
					vTaskDelay(pdMS_TO_TICKS(10));
				}
				break;

				case gapAdvDataSetupComp:
				{
					gapGenericEvent_t genEvent;
					genEvent.eventType = gAdvertisingDataSetupComplete_c;
					genericCallback(&genEvent);
				}
				break;

				case gapAdvParamSetupComp:
				{
					gapGenericEvent_t genEvent;
					genEvent.eventType = gAdvertisingParametersSetupComplete_c;
					genericCallback(&genEvent);
				}
				break;

				case gapAdvSetupFailed:
				{
					gapGenericEvent_t genEvent;
					genEvent.eventType = gAdvertisingSetupFailed_c;
					genEvent.eventData.setupFailError = gGapAdvDataTooLong_c;
					genericCallback(&genEvent);
				}
				break;

				case gapAdvEventStateChanged:
				{
					gapAdvertisingEvent_t advEvent;
					advEvent.eventType = gAdvertisingStateChanged_c;
					pAdvCallback(&advEvent);
				}
				break;

				case gapAdvEventFailed:
				{
					gapAdvertisingEvent_t advEvent;
					advEvent.eventType = gAdvertisingCommandFailed_c;
					advEvent.eventData.failReason = pMsg->ResponseType.advEventFail.failReason;
					pAdvCallback(&advEvent);
				}
				break;

				case gapCheckNotfStatusResp:
				{
					*isNotfActive = pMsg->ResponseType.checkNotfStatResp.isActive;
				}
				break;

				case gapConnectionEventConnected:
				{
					gapConnectionEvent_t connEvent;
					connEvent.eventType = gConnEvtConnected_c;
					memcpy(&connEvent.eventData.connectedEvent, &payload[1], pMsg->length);
					pConnCallback(payload[0], &connEvent);
				}
				break;

				case gapConnectionEventDisConnected:
				{
					gapConnectionEvent_t connEvent;
					connEvent.eventType = gConnEvtDisconnected_c;
					connEvent.eventData.disconnectedEvent.reason = (payload[1] | payload[2] << 8);
					pConnCallback(payload[0], &connEvent);
				}
				break;

				case gapConnEventLEDataLenChanged:
				{
					PRINTF("\r\n---------- DATA LENGTH CHANGE ----------\r\n");
				}
				break;

				default:
				{
				}
				break;
			}
		}
		break;

		case bleGATT:
		{
			switch(pMsg->opCode)
			{
				case gattConfirm:
				{
					GattConfirm_t *resp = (GattConfirm_t*)payload;
					uint16_t status = resp->status;
					xTaskNotify(task_handle, status, eSetValueWithOverwrite );
				}
				break;

				case gattSendNotification:
				{
				}
				break;

				case gattCharCccdWritten:
				{
					gattServerEvent_t serverEvent;
					serverEvent.eventType = gEvtCharacteristicCccdWritten_c;
					serverEvent.eventData.charCccdWrittenEvent.handle = (payload[2] | (payload[1] << 8));
					serverEvent.eventData.charCccdWrittenEvent.newCccd = payload[3];
					pfGattServerCallback(payload[0], &serverEvent);
				}
				break;

				case gattServerCallbackRegister:
				{
				}
				break;

				case gattMTUChanged:
				{
					PRINTF("\r\n---------- MTU CHANGED ----------\r\n");
				}
				break;

				default:
				{
				}
				break;
			}
		}
		break;

		case bleGATTDB:
		{
			switch (pMsg->opCode)
			{
				case gattDBConfirm:
				{
					gattDBConfirm_t *resp = (gattDBConfirm_t*)payload;
					uint16_t status = resp->status;
					xTaskNotify(task_handle, status, eSetValueWithOverwrite );
				}
				break;

				case gattDBAddPrimServResp:
				{
					*primServiceHandle = pMsg->ResponseType.gattDBAddPrimServResp.servHandle;
				}
				break;

				case gattDBAddSecServResp:
				{
					*secServiceHandle = pMsg->ResponseType.gattDBAddSecServResp.servHandle;
				}
				break;

				case gattDBAddCharDeclandValResp:
				{
					*charHandle = pMsg->ResponseType.gattDBAddCharDeclandValResp.charHandle;
				}
				break;

				case gattDBAddCharDescResp:
				{
					*charDescHandle = pMsg->ResponseType.gattDBAddCharDescResp.descHandle;
				}
				break;

				case gattDBAddCccdResp:
				{
					*cccdHandle = pMsg->ResponseType.gattDBAddCccdResp.CccdHandle;
				}
				break;

				case gattDBReadAttrResp:
				{
					*attributeVallen = pMsg->ResponseType.gattDBReadAttrResp.valueLen;
					attributeVal = pMsg->ResponseType.gattDBReadAttrResp.value;
				}
				break;

				case gattDBFindSerHandleResp:
				{
					*servHandle = pMsg->ResponseType.gattDBFindServHandleResp.servHandle;
				}
				break;

				case gattDBFindCharValHandleResp:
				{
					*charValHandle = pMsg->ResponseType.gattDBFindCharValHandleResp.charValHandle;
				}
				break;

				case gattDBFindCccdHandleResp:
				{
					*cccdHandleCharVal = pMsg->ResponseType.gattDBFindCccdHandleResp.CccdHandle;
				}
				break;

				case gattDBFindDescHandleResp:
				{
					*descHandle = pMsg->ResponseType.gattDBFindDescHandleResp.descHandle;
				}
				break;

				default:
				{
				}
				break;
			}
		}
		break;

		default:
		{
		}
		break;
	}
}
/*! *********************************************************************************
* \brief   This thread checks continuously whether there is any message
* 		   available in receive message queue from FSCI layer.
*
* \param[in]    param        argument passed to the task function.
********************************************************************************** */
void FSCI_ReceivePacketTask(void *argument)
{
	osa_event_flags_t event = 0;
	recv_task_handle = xTaskGetCurrentTaskHandle();

	while(1)
	{
		OSA_EventWait(&evt_handle, osaEventFlagsAll_c, FALSE, 0, &event);

		if(event & gEvtMsgFromFSCI_c || (item_cnt > 0))
		{
			uint8_t *resp_buff = malloc(DATA_SIZE);

			/* Check for FSCI responses from FSCI layer */
			if((OSA_MsgQGet(&rx_msg_queue, resp_buff, 0)) == KOSA_StatusSuccess)
			{
				if(resp_buff)
				{
					uint8_t *data = resp_buff + sizeof(packet_t);
					uint16_t len = ((uint16_t)resp_buff[4] << 8) | resp_buff[3];
					uint8_t *payload = malloc(len);

#ifdef DEBUG_PRINT_ENABLE
					PRINTF("\r\nRECV_RESP:\r\n");
					for(int i = 0; i < (sizeof(packet_t) + len + 1); i++)
					{
						PRINTF("%2X ", resp_buff[i]);
					}
					PRINTF("\r\n");
#endif

					for(int i = 0; i < len; i++)
					{
						payload[i] = data[i];
					}

					FSCI_HandleResponse((FSCIResponseMsg_t*)resp_buff, payload, data[len]);

					/* Decrement event item count as event is received */
					item_cnt--;

					/* Messages must always be freed. */
					free(resp_buff);
					free(payload);
					resp_buff = NULL;
					payload = NULL;
				}
			}
			else
			{

			}
		}

		vTaskDelay(pdMS_TO_TICKS(20));
	}
}
