/*
 * FreeRTOS Kernel V10.2.0
 * Copyright (C) 2019 Amazon.com, Inc. or its affiliates.  All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of
 * this software and associated documentation files (the "Software"), to deal in
 * the Software without restriction, including without limitation the rights to
 * use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
 * the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
 * COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * http://www.FreeRTOS.org
 * http://aws.amazon.com/freertos
 *
 * 1 tab == 4 spaces!
 */

#ifndef GAP_INTERFACE_H
#define GAP_INTERFACE_H

#include "ble_general.h"
#include "gap_types.h"

extern bool_t *isNotfActive;
extern gapAdvertisingCallback_t pAdvCallback;
extern gapConnectionCallback_t pConnCallback;


/*! *********************************************************************************
* \brief        Wrapper function to register generic callback function.
*
* \param[in]    GenericCallback callback function.
********************************************************************************** */
void GAP_RegisterGenericCallback (gapGenericCallback_t GenericCallback);

/*! *********************************************************************************
* \brief  Sets up the Advertising Parameters.
*
* \param[in] pAdvertisingParameters   Pointer to gapAdvertisingParameters_t structure.
*
* \return  gBleSuccess_c or error.
*
* \remarks GAP Peripheral-only API function.
*
********************************************************************************** */
bleResult_t Gap_SetAdvertisingParameters
(
    gapAdvertisingParameters_t*   pAdvertisingParameters
);

/*! *********************************************************************************
* \brief  Sets up the Advertising and Scan Response Data.
*
* \param[in] pAdvertisingData   Pointer to gapAdvertisingData_t structure or NULL.
* \param[in] pScanResponseData  Pointer to gapScanResponseData_t structure or NULL.
*
* \return  gBleSuccess_c or error.
*
* \remarks Any of the parameters may be NULL, in which case they are ignored.
* Therefore, this function can be used to set any of the parameters individually or both at once.
* The standard advertising packet payload is 37 bytes. Some of the payload may be occupied by
* the Advertisser Address which takes up 6 bytes and for some advertising PDU types also by
* the Initiator Address which takes another 6 bytes. This leaves 25-31 bytes to the application
* to include advertising structures (Length [1Byte], AD Type [1 Byte], AD Data[Length-1 Bytes])
*
* \remarks GAP Peripheral-only API function.
*
********************************************************************************** */
bleResult_t Gap_SetAdvertisingData
(
    gapAdvertisingData_t*         pAdvertisingData,
    gapScanResponseData_t*        pScanResponseData
);

/*! *********************************************************************************
* \brief  Commands the KW36 FSCI application to start advertising.
*
* \param[in] advertisingCallback   Callback used by the application to receive advertising events.
*                                  Can be NULL.
* \param[in] connectionCallback    Callback used by the application to receive connection events.
*                                  Can be NULL.
*
* \return  gBleSuccess_c or error.
*
* \remarks The advertisingCallback confirms or denies whether the advertising has started.
* The connectionCallback is only used if a connection gets established during advertising.
*
* \remarks GAP Peripheral-only API function.
*
********************************************************************************** */
bleResult_t Gap_StartAdvertising
(
    gapAdvertisingCallback_t    advertisingCallback,
    gapConnectionCallback_t     connectionCallback
);


/*! *********************************************************************************
* \brief  Commands the KW36 FSCI application to stop advertising.
*
* \return  gBleSuccess_c or error.
*
* \remarks GAP Peripheral-only API function.
*
********************************************************************************** */
bleResult_t Gap_StopAdvertising(void);

/*! *********************************************************************************
* \brief  Retrieves the notification status for a given Client and a given CCCD handle.
*
* \param[in]  deviceId      The peer GATT Client.
* \param[in]  handle        The handle of the CCCD.
*
* \return  gBleSuccess_c or error.
*
* \remarks GATT Server-only API function.
*
********************************************************************************** */
bleResult_t Gap_CheckNotificationStatus
(
    deviceId_t  deviceId,
    uint16_t    handle,
	bool_t*     pOutIsActive
);

/*! *********************************************************************************
* \brief  Initiates disconnection from a connected peer device.
*
* \param[in] deviceId The connected peer to disconnect from.
*
* \return  gBleSuccess_c or error.
*
********************************************************************************** */
bleResult_t Gap_Disconnect
(
    deviceId_t  deviceId
);
#endif /* GAP_INTERFACE_H */
