/*
 * FreeRTOS Kernel V10.2.0
 * Copyright (C) 2019 Amazon.com, Inc. or its affiliates.  All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of
 * this software and associated documentation files (the "Software"), to deal in
 * the Software without restriction, including without limitation the rights to
 * use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
 * the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
 * COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * http://www.FreeRTOS.org
 * http://aws.amazon.com/freertos
 *
 * 1 tab == 4 spaces!
 */

#ifndef GATT_SERVER_H
#define GATT_SERVER_H

#include "BLE_Abstraction_main.h"
#include "ble_general.h"
#include "gatt_types.h"

/*! GATT Server Event type enumeration */
typedef enum gattServerEventType_tag {
    gEvtMtuChanged_c                        = 0x00, /*!< ATT_MTU was changed after the MTU exchange. */
    gEvtHandleValueConfirmation_c           = 0x01, /*!< Received a Handle Value Confirmation from the Client. */
    gEvtAttributeWritten_c                  = 0x02, /*!< An attribute registered with GattServer_RegisterHandlesForWriteNotifications was written.
                                                         After receiving this event, application must call GattServer_SendAttributeWrittenStatus.
                                                         Application must write the Attribute in the Database if it considers necessary. */
    gEvtCharacteristicCccdWritten_c         = 0x03, /*!< A CCCD was written. Application should save the CCCD value with Gap_SaveCccd. */
    gEvtAttributeWrittenWithoutResponse_c   = 0x04, /*!< An attribute registered with GattServer_RegisterHandlesForWriteNotifications was written without response (with ATT Write Command).
                                                         Application must write the Attribute Value in the Database if it considers necessary. */
    gEvtError_c                             = 0x05, /*!< An error appeared during a Server-initiated procedure. */
    gEvtLongCharacteristicWritten_c         = 0x06, /*!< A long characteristic was written. */
    gEvtAttributeRead_c                     = 0x07, /*!< An attribute registered with GattServer_RegisterHandlesForReadNotifications is being read.
                                                         After receiving this event, application must call GattServer_SendAttributeReadStatus. */
} gattServerEventType_t;

/*! GATT Server MTU Changed Event structure */
typedef struct gattServerMtuChangedEvent_tag {
    uint16_t newMtu;    /*!< Value of the agreed ATT_MTU for this connection. */
} gattServerMtuChangedEvent_t;

/*! GATT Server Attribute Written Event structure */
typedef struct gattServerAttributeWrittenEvent_tag {
    uint16_t    handle;         /*!< Handle of the attribute. */
    uint16_t    cValueLength;   /*!< Length of the attribute value array. */
    uint8_t*    aValue;         /*!< Attribute value array attempted to be written. */
} gattServerAttributeWrittenEvent_t;

/*! GATT Server Long Characteristic Written Event structure */
typedef struct gattServerLongCharacteristicWrittenEvent_tag {
    uint16_t    handle;         /*!< Handle of the Characteristic Value. */
    uint16_t    cValueLength;   /*!< Length of the value written. */
    uint8_t*    aValue;         /*!< Pointer to the attribute value in the database. */
} gattServerLongCharacteristicWrittenEvent_t;

/*! GATT Server CCCD Written Event structure */
typedef PACKED_STRUCT gattServerCccdWrittenEvent_tag {
    uint16_t          handle;       /*!< Handle of the CCCD attribute. */
    gattCccdFlags_t   newCccd;      /*!< New value of the CCCD. */
} gattServerCccdWrittenEvent_t;

/*! GATT Server Attribute Read Event structure */
typedef struct gattServerAttributeReadEvent_tag {
    uint16_t    handle;         /*!< Handle of the attribute. */
} gattServerAttributeReadEvent_t;

/*! Server-initiated procedure type enumeration */
typedef enum gattServerProcedureType_tag {
    gSendAttributeWrittenStatus_c,      /*!< Procedure initiated by GattServer_SendAttributeWrittenStatus. */
    gSendAttributeReadStatus_c,         /*!< Procedure initiated by GattServer_SendAttributeReadStatus. */
    gSendNotification_c,                /*!< Procedure initiated by GattServer_SendNotification. */
    gSendIndication_c                   /*!< Procedure initiated by GattServer_SendIndication. */
} gattServerProcedureType_t;

/*! Server-initiated procedure error structure */
typedef PACKED_STRUCT gattProcedureError_tag {
    gattServerProcedureType_t   procedureType;  /*!< Procedure that generated error. */
    bleResult_t                 error;          /*!< Error generated. */
} gattServerProcedureError_t;

/*! GATT Server Event structure: type + data. */
typedef PACKED_STRUCT gattServerEvent_tag {
    gattServerEventType_t eventType;    /*!< Event type. */
    union {
        gattServerMtuChangedEvent_t                 mtuChangedEvent;        /*!< For event type gEvtMtuChanged_c: the new value of the ATT_MTU. */
        gattServerAttributeWrittenEvent_t           attributeWrittenEvent;  /*!< For event types gEvtAttributeWritten_c, gEvtAttributeWrittenWithoutResponse_c: handle and value of the attempted write. */
        gattServerCccdWrittenEvent_t                charCccdWrittenEvent;   /*!< For event type gEvtCharacteristicCccdWritten_c: handle and value of the CCCD. */
        gattServerProcedureError_t                  procedureError;         /*!< For event type gEvtError_c: error that terminated a Server-initiated procedure. */
        gattServerLongCharacteristicWrittenEvent_t  longCharWrittenEvent;   /*!< For event type gEvtLongCharacteristicWritten_c: handle and value. */
        gattServerAttributeReadEvent_t              attributeReadEvent;     /*!< For event types gEvtAttributeRead_c: handle of the attempted read. */
    } eventData;                            /*!< Event data : selected according to event type. */
} gattServerEvent_t;

/*  Send Notification FSCI frame payload*/
typedef struct gattServerSendNotificationReq_Req {
	deviceId_t deviceId;
	uint16_t handle;
} gattServerSendNotificationReq_t;

/*! GATT Server Callback prototype */
typedef void (*gattServerCallback_t)
(
    deviceId_t          deviceId,       /*!< Device ID identifying the active connection. */
    gattServerEvent_t*  pServerEvent    /*!< Server event. */
);

extern gattServerCallback_t pfGattServerCallback;

/*! *********************************************************************************
* \brief  Installs an application callback for the GATT Server module.
*
* \param[in] callback Application-defined callback to be triggered by this module.
*
* \return  gBleSuccess_c or error.
***********************************************************************************/
bleResult_t Gatt_RegisterCallback
(
    gattServerCallback_t serverCallback
);

/*! *********************************************************************************
* \brief  Sends a notification to a peer GATT Client using the Characteristic Value
* from the GATT Database.
*
* \param[in] deviceId           The device ID of the connected peer.
* \param[in] handle             Handle of the Value of the Characteristic to be notified.
*
* \return  gBleSuccess_c or error.
*
********************************************************************************** */
bleResult_t Gatt_SendNotification
(
    deviceId_t              deviceId,
    uint16_t                handle
);

#endif /* GATT_SERVER_H */
