/*
 * FreeRTOS Kernel V10.2.0
 * Copyright (C) 2019 Amazon.com, Inc. or its affiliates.  All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of
 * this software and associated documentation files (the "Software"), to deal in
 * the Software without restriction, including without limitation the rights to
 * use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
 * the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
 * COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * http://www.FreeRTOS.org
 * http://aws.amazon.com/freertos
 *
 * 1 tab == 4 spaces!
 */

#ifndef GATTDB_INTERFACE_H
#define GATTDB_INTERFACE_H

#include "BLE_Abstraction_main.h"
#include "ble_general.h"
#include "gatt_database.h"

/************************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
************************************************************************************/

#ifdef __cplusplus
extern "C" {
#endif

extern uint16_t *primServiceHandle;
extern uint16_t *secServiceHandle;
extern uint16_t *charHandle;
extern uint16_t *charDescHandle;
extern uint16_t *cccdHandle;
extern uint8_t  *attributeVal;
extern uint16_t  *attributeVallen;
extern uint16_t *servHandle;
extern uint16_t *charValHandle;
extern uint16_t *cccdHandleCharVal;
extern uint16_t *descHandle;

typedef PACKED_STRUCT gattDBAddPrimaryService_Req {
	uint16_t desireHandle;
	bleUuidType_t   serviceUuidType;
	bleUuid_t*      pServiceUuid;
} gattDBAddPrimaryServiceReq_t;

typedef gattDBAddPrimaryServiceReq_t gattDBAddSecondaryServiceReq_t;

typedef PACKED_STRUCT gattDBAddCharacDecandValue_Req {
	bleUuidType_t                               characteristicUuidType;
	bleUuid_t*                                  pCharacteristicUuid;
	gattCharacteristicPropertiesBitFields_t     characteristicProperties;
	uint16_t                                    maxValueLength;
	uint16_t                                    initialValueLength;
	uint8_t*                                    aInitialValue;
	gattAttributePermissionsBitFields_t         valueAccessPermissions;
}gattDBAddCharDecandValueReq_t;

typedef PACKED_STRUCT gattDBAddCharacDescriptor_Req {
	bleUuidType_t                               descriptorUuidType;
    bleUuid_t*                                  pDescriptorUuid;
    uint16_t                                    descriptorValueLength;
    uint8_t*                                    aInitialValue;
    gattAttributePermissionsBitFields_t         descriptorAccessPermission;
}gattDBAddCharacDescriptorReq_t;

typedef PACKED_STRUCT gattDBWriteAttribute_Req {
	uint16_t    handle;
    uint16_t    valueLength;
    uint8_t*    aValue;
}gattDBWriteAttributeReq_t;

typedef struct gattDBReadAttribute_Req {
	uint16_t    handle;
    uint16_t    maxBytes;
}gattDBReadAttributeReq_t;

typedef struct gattDBFindSeriveHandle_Req {
	uint16_t        startHandle;
    bleUuidType_t   serviceUuidType;
    bleUuid_t*      pServiceUuid;
}gattDBFindSeriveHandleReq_t;

typedef PACKED_STRUCT gattDBFindCharValueHandleInService_Req {
	uint16_t        serviceHandle;
    bleUuidType_t   characteristicUuidType;
    bleUuid_t*      pCharacteristicUuid;
}gattDBFindCharValueHandleInServiceReq_t;

typedef struct gattDBFindCccdHandleForCharValueHandle_Req {
	uint16_t        charValueHandle;
}gattDBFindCccdHandleForCharValueHandleReq_t;

typedef struct gattDBFindDescHandleForCharValueHandle_Req {
	uint16_t        charValueHandle;
    bleUuidType_t   descriptorUuidType;
    bleUuid_t*      pDescriptorUuid;
}gattDBFindDescHandleForCharValueHandleReq_t;

typedef struct gattDBConfirm_tag
{
	uint16_t status;
}gattDBConfirm_t;

typedef struct gattDBAddPrimServResp_tag
{
	uint16_t servHandle;
}gattDBAddPrimServResp_t;

typedef struct gattDBAddSecServResp_tag
{
	uint16_t servHandle;
}gattDBAddSecServResp_t;

typedef struct gattDBAddCharDeclandValResp_tag
{
	uint16_t charHandle;
}gattDBAddCharDeclandValResp_t;

typedef struct gattDBAddCharDescResp_tag
{
	uint16_t descHandle;
}gattDBAddCharDescResp_t;

typedef struct gattDBAddCccdResp_tag
{
	uint16_t CccdHandle;
}gattDBAddCccdResp_t;

typedef struct gattDBReadAttrResp_tag
{
	uint16_t valueLen;
	uint8_t *value;
}gattDBReadAttrResp_t;

typedef struct gattDBFindServHandleResp_tag
{
	uint16_t servHandle;
}gattDBFindServHandleResp_t;

typedef struct gattDBFindCharValHandleResp_tag
{
	uint16_t charValHandle;
}gattDBFindCharValHandleResp_t;

typedef struct gattDBFindCccdHandleResp_tag
{
	uint16_t CccdHandle;
}gattDBFindCccdHandleResp_t;

typedef struct gattDBFindDescHandleResp_tag
{
	uint16_t descHandle;
}gattDBFindDescHandleResp_t;

/*! *********************************************************************************
* \brief  Allocates smallest possible buffers and initializes an empty database.
*
* \return  gBleSuccess_c or gBleOutOfMemory_c.
*
********************************************************************************** */
bleResult_t GattDbDynamic_Init(void);

/*! *********************************************************************************
* \brief  Releases allocated buffers.
*
* \return  gBleSuccess_c or gBleOsError_c.
*
********************************************************************************** */
bleResult_t GattDb_ReleaseDatabase(void);

/*! *********************************************************************************
* \brief  Adds a Primary Service declaration into the database.
*
* \param[in]  desiredHandle     Desired attribute handle of the Service declaration.
*                               Ignored if 0x0000 and allocated by the GATT Database
*                               Dynamic module.
* \param[in]  serviceUuidType   Service UUID type (16-bit, 32-bit or 128-bit).
* \param[in]  pServiceUuid      Service UUID.
*
* \return  gBleSuccess_c or gBleOutOfMemory_c.
*
********************************************************************************** */
bleResult_t GattDb_AddPrimaryService
(
    uint16_t        desiredHandle,
    bleUuidType_t   serviceUuidType,
    bleUuid_t*      pServiceUuid,
	uint16_t*       pOutHandle
);

/*! *********************************************************************************
* \brief  Adds a Secondary Service declaration into the database.
*
* \param[in]  desiredHandle     Desired attribute handle of the Service declaration.
*                               Ignored if 0x0000 and allocated by the GATT Database
*                               Dynamic module.
* \param[in]  serviceUuidType   Service UUID type (16-bit, 32-bit or 128-bit).
* \param[in]  pServiceUuid      Service UUID.
*
* \return  gBleSuccess_c or gBleOutOfMemory_c.
*
********************************************************************************** */
bleResult_t GattDb_AddSecondaryService
(
    uint16_t        desiredHandle,
    bleUuidType_t   serviceUuidType,
    bleUuid_t*      pServiceUuid,
	uint16_t*       pOutHandle
);

/*! *********************************************************************************
* \brief  Adds a Characteristic declaration and its Value into the database.
*
* \param[in]  characteristicUuidType    Characteristic UUID type.
* \param[in]  pCharacteristicUuid       Characteristic UUID.
* \param[in]  characteristicProperties  Characteristic Properties.
* \param[in]  maxValueLength            If the Characteristic Value length is variable,
*                                       this is the maximum length. For fixed lengths,
*                                       this must be set to 0.
* \param[in]  initialValueLength        Value length at initialization. Remains fixed
*                                       if maxValueLength is set to 0, otherwise cannot be
*                                       greater than maxValueLength.
* \param[in]  aInitialValue             Contains the initial value of the Characteristic.
* \param[in]  valueAccessPermissions    Access permissions for the value attribute.

* \return  gBleSuccess_c or gBleOutOfMemory_c.
*
********************************************************************************** */
bleResult_t GattDb_AddCharacteristicDeclarationAndValue
(
    bleUuidType_t                               characteristicUuidType,
    bleUuid_t*                                  pCharacteristicUuid,
    gattCharacteristicPropertiesBitFields_t     characteristicProperties,
    uint16_t                                    maxValueLength,
    uint16_t                                    initialValueLength,
    uint8_t*                                    aInitialValue,
    gattAttributePermissionsBitFields_t         valueAccessPermissions,
	uint16_t*       							pOutHandle
);

/*! *********************************************************************************
* \brief  Adds a Characteristic descriptor into the database.
*
* \param[in]  descriptorUuidType            Descriptor UUID type.
* \param[in]  pDescriptorUuid               Descriptor UUID.
* \param[in]  descriptorValueLength         Fixed length of the descriptor attribute value.
* \param[in]  aInitialValue                 Initial value of the descriptor attribute.
* \param[in]  descriptorAccessPermissions   Access permissions for the descriptor attribute.
* \return  gBleSuccess_c or gBleOutOfMemory_c.
*
*
********************************************************************************** */
bleResult_t GattDb_AddCharacteristicDescriptor
(
    bleUuidType_t                               descriptorUuidType,
    bleUuid_t*                                  pDescriptorUuid,
    uint16_t                                    descriptorValueLength,
    uint8_t*                                    aInitialValue,
    gattAttributePermissionsBitFields_t         descriptorAccessPermissions,
	uint16_t*       							pOutHandle
);

/*! *********************************************************************************
* \brief  Adds a CCCD in the database.
*
* \return  gBleSuccess_c or gBleOutOfMemory_c.
*
********************************************************************************** */
bleResult_t GattDb_AddCccd
(
		uint16_t*       pOutHandle
);

/*!
 * \brief Writes an attribute from the application level.
 *
 * This function can be called by the application code to modify an attribute in the database.
 * It should only be used by the application to modify a Characteristic's value
 * based on the application logic (e.g., external sensor readings).
 *
 * \param[in] handle             The handle of the attribute to be written.
 * \param[in] valueLength        The number of bytes to be written.
 * \param[in] aValue             The source buffer containing the value to be written.
 *
 * \return                       gBleSuccess_c or error.
 *
 */
bleResult_t GattDb_WriteAttribute
(
    uint16_t    handle,
    uint16_t    valueLength,
    uint8_t*    aValue
);

/*!
 * \brief Reads an attribute from the application level.
 *
 * This function can be called by the application code to read an attribute in the database.
 *
 * \param[in]  handle           The handle of the attribute to be read.
 * \param[in]  maxBytes         The maximum number of bytes to be received.
 *
 * \return                  gBleSuccess_c or error.
 *
 */
bleResult_t GattDb_ReadAttribute
(
    uint16_t    handle,
    uint16_t    maxBytes,
	uint8_t*    aOutValue,
	uint16_t*   pOutValueLength
);

/*!
 * \brief Finds the handle of a Service Declaration with a given UUID inside the database.
 *
 * \param[in]  startHandle              The handle to start the search. Should be 0x0001 on the first call.
 * \param[in]  serviceUuidType          Service UUID type.
 * \param[in]  pServiceUuid             Service UUID.
 *
 * \return                  gBleSuccess_c or error.
 */
bleResult_t GattDb_FindServiceHandle
(
    uint16_t        startHandle,
    bleUuidType_t   serviceUuidType,
    bleUuid_t*      pServiceUuid,
	uint16_t*       pOutServiceHandle
);

/*!
 * \brief Finds the handle of a Characteristic Value with a given UUID inside a Service.
 *
 * The Service is input by its declaration handle.
 *
 * \param[in]  serviceHandle            The handle of the Service declaration.
 * \param[in]  characteristicUuidType   Characteristic UUID type.
 * \param[in]  pCharacteristicUuid      Characteristic UUID.
 *
 * \return                          gBleSuccess_c or error.
 *
 */
bleResult_t GattDb_FindCharValueHandleInService
(
    uint16_t        serviceHandle,
    bleUuidType_t   characteristicUuidType,
    bleUuid_t*      pCharacteristicUuid,
	uint16_t*       pOutCharValueHandle
);

/*!
 * \brief Finds the handle of a Characteristic's CCCD given the Characteristic's Value handle.
 *
 * \param[in]  charValueHandle      The handle of the Service declaration.
 *
 * \return                          gBleSuccess_c or error.
 */
bleResult_t GattDb_FindCccdHandleForCharValueHandle
(
    uint16_t        charValueHandle,
	uint16_t*       pOutCccdHandle
);

/*!
 * \brief Finds the handle of a Characteristic Descriptor given the Characteristic's Value handle and Descriptor's UUID.
 *
 * \param[in]  charValueHandle          The handle of the Service declaration.
 * \param[in]  descriptorUuidType       Descriptor's UUID type.
 * \param[in]  pDescriptorUuid          Descriptor's UUID.
 *
 * \return                          gBleSuccess_c or error.
 */
bleResult_t GattDb_FindDescriptorHandleForCharValueHandle
(
    uint16_t        charValueHandle,
    bleUuidType_t   descriptorUuidType,
    bleUuid_t*      pDescriptorUuid,
	uint16_t*       pOutDescriptorHandle
);

#endif /* GATTDB_INTERFACE_H */
