/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 *
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "fsl_common.h"
#include "fsl_port.h"

#include "fsl_device_registers.h"
#include "fsl_debug_console.h"
#include "board.h"

#include "pin_mux.h"
#include "clock_config.h"
#include "EmbeddedTypes.h"
#include "SPI.h"
#include "ble_FSCI.h"
#include "FSCI_main.h"
#include "BLE_Abstraction_main.h"
/************************************************************************************
*************************************************************************************
* Private macros
*************************************************************************************
************************************************************************************/
#define PIN0_IDX                         0u   /*!< Pin number for pin 0 in a port */
#define PIN1_IDX                         1u   /*!< Pin number for pin 1 in a port */
#define PIN2_IDX                         2u   /*!< Pin number for pin 2 in a port */
#define PIN3_IDX                         3u   /*!< Pin number for pin 3 in a port */
#define PIN16_IDX                       16u   /*!< Pin number for pin 16 in a port */
#define PIN17_IDX                       17u   /*!< Pin number for pin 17 in a port */
#define SOPT5_UART0TXSRC_UART_TX      0x00u   /*!< UART 0 transmit data source select: UART0_TX pin */

#define FSCI_SYNC_BYTE 0x02

#define TRANSFER_BAUDRATE 500000U /*! Transfer baudrate - 500k */

#define DSPI_DELAY_COUNT 0XFFFFFU
/************************************************************************************
*************************************************************************************
* Private type definitions
*************************************************************************************
************************************************************************************/

/*******************************************************************************
 * Private memory declarations
 ******************************************************************************/
volatile uint32_t masterCommand;

uint32_t TRANSFER_SIZE;
uint32_t masterFifoSize;

uint8_t masterRxData[DATA_SIZE] = {0U};
uint8_t masterTxData[DATA_SIZE] = {0U};

/* Variable that indicates number of times event is set */
uint32_t item_cnt = 0;

volatile uint32_t masterTxCount;
volatile uint32_t masterRxCount;
volatile bool isTransferCompleted = false;

dspi_command_data_config_t commandData;
/*******************************************************************************
 * Private functions Prototypes
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Private memory declarations
*************************************************************************************
************************************************************************************/
uint8_t buff_data[DATA_SIZE] = {0};

/*! *********************************************************************************
*************************************************************************************
* Public functions prototypes
*************************************************************************************
********************************************************************************** */

/*******************************************************************************
 ********************************************************************************
 * Public functions
 ******************************************************************************
********************************************************************************/

/*FUNCTION**********************************************************************
 *
 * Function Name : BOARD_SPIInitPins
 * Description   : Configures pin routing and optionally pin electrical features for SPI.
 *
 *END**************************************************************************/

void BOARD_SPIInitPins(void)
{
	CLOCK_EnableClock(kCLOCK_PortB);                           /* Port B Clock Gate Control: Clock enabled */
	CLOCK_EnableClock(kCLOCK_PortD);                           /* Port D Clock Gate Control: Clock enabled */

	PORT_SetPinMux(PORTB, PIN16_IDX, kPORT_MuxAlt3);           /* PORTB16 (pin 62) is configured as UART0_RX */
	PORT_SetPinMux(PORTB, PIN17_IDX, kPORT_MuxAlt3);           /* PORTB17 (pin 63) is configured as UART0_TX */
	PORT_SetPinMux(PORTD, PIN0_IDX, kPORT_MuxAlt2);            /* PORTD0 (pin 93) is configured as SPI0_PCS0 */
	PORT_SetPinMux(PORTD, PIN1_IDX, kPORT_MuxAlt2);            /* PORTD1 (pin 94) is configured as SPI0_SCK */
	PORT_SetPinMux(PORTD, PIN2_IDX, kPORT_MuxAlt2);            /* PORTD2 (pin 95) is configured as SPI0_SOUT */
	PORT_SetPinMux(PORTD, PIN3_IDX, kPORT_MuxAlt2);            /* PORTD3 (pin 96) is configured as SPI0_SIN */
	SIM->SOPT5 = ((SIM->SOPT5 &
	(~(SIM_SOPT5_UART0TXSRC_MASK)))                          /* Mask bits to zero which are setting */
	  | SIM_SOPT5_UART0TXSRC(SOPT5_UART0TXSRC_UART_TX)       /* UART 0 transmit data source select: UART0_TX pin */
	);
}

void SPI_Init(void)
{
	uint32_t srcClock_Hz;
	dspi_master_config_t masterConfig;

	/* Master config */
	masterConfig.whichCtar                                = kDSPI_Ctar0;
	masterConfig.ctarConfig.baudRate                      = TRANSFER_BAUDRATE;
	masterConfig.ctarConfig.bitsPerFrame                  = 8U;
	masterConfig.ctarConfig.cpol                          = kDSPI_ClockPolarityActiveHigh;
	masterConfig.ctarConfig.cpha                          = kDSPI_ClockPhaseFirstEdge;
	masterConfig.ctarConfig.direction                     = kDSPI_MsbFirst;
	masterConfig.ctarConfig.pcsToSckDelayInNanoSec        = 1000000000U / TRANSFER_BAUDRATE;
	masterConfig.ctarConfig.lastSckToPcsDelayInNanoSec    = 1000000000U / TRANSFER_BAUDRATE;
	masterConfig.ctarConfig.betweenTransferDelayInNanoSec = 1000000000U / TRANSFER_BAUDRATE;

	masterConfig.whichPcs           = EXAMPLE_DSPI_MASTER_PCS;
	masterConfig.pcsActiveHighOrLow = kDSPI_PcsActiveLow;

	masterConfig.enableContinuousSCK        = false;
	masterConfig.enableRxFifoOverWrite      = false;
	masterConfig.enableModifiedTimingFormat = false;
	masterConfig.samplePoint                = kDSPI_SckToSin0Clock;

	srcClock_Hz = EXAMPLE_DSPI_MASTER_CLK_FREQ;

	DSPI_MasterInit(EXAMPLE_DSPI_MASTER_BASEADDR, &masterConfig, srcClock_Hz);

	/* Enable the NVIC for DSPI peripheral. */
	EnableIRQ(EXAMPLE_DSPI_MASTER_IRQ);

	commandData.isPcsContinuous    = false;
	commandData.whichCtar          = kDSPI_Ctar0;
	commandData.whichPcs           = EXAMPLE_DSPI_MASTER_PCS;
	commandData.isEndOfQueue       = false;
	commandData.clearTransferCount = false;

	masterCommand  = DSPI_MasterGetFormattedCommand(&commandData);
	masterFifoSize = FSL_FEATURE_DSPI_FIFO_SIZEn(EXAMPLE_DSPI_MASTER_BASEADDR);
}

void EXAMPLE_DSPI_MASTER_IRQHandler(void)
{
    if (masterRxCount < TRANSFER_SIZE)
    {
        while (DSPI_GetStatusFlags(EXAMPLE_DSPI_MASTER_BASEADDR) & kDSPI_RxFifoDrainRequestFlag)
        {
            masterRxData[masterRxCount] = DSPI_ReadData(EXAMPLE_DSPI_MASTER_BASEADDR);
            ++masterRxCount;

            DSPI_ClearStatusFlags(EXAMPLE_DSPI_MASTER_BASEADDR, kDSPI_RxFifoDrainRequestFlag);

            if (masterRxCount == TRANSFER_SIZE)
            {
                break;
            }
        }
    }

    if (masterTxCount < TRANSFER_SIZE)
    {
        while ((DSPI_GetStatusFlags(EXAMPLE_DSPI_MASTER_BASEADDR) & kDSPI_TxFifoFillRequestFlag) &&
               ((masterTxCount - masterRxCount) < masterFifoSize))
        {
            if (masterTxCount < TRANSFER_SIZE)
            {
                EXAMPLE_DSPI_MASTER_BASEADDR->PUSHR = masterCommand | masterTxData[masterTxCount];
                ++masterTxCount;
            }
            else
            {
                break;
            }

            /* Try to clear the TFFF; if the TX FIFO is full this will clear */
            DSPI_ClearStatusFlags(EXAMPLE_DSPI_MASTER_BASEADDR, kDSPI_TxFifoFillRequestFlag);
        }
    }

    /* Check if we're done with this transfer.*/
    if ((masterTxCount == TRANSFER_SIZE) && (masterRxCount == TRANSFER_SIZE))
    {
        isTransferCompleted = true;
        /* Complete the transfer and disable the interrupts */
        DSPI_DisableInterrupts(EXAMPLE_DSPI_MASTER_BASEADDR,
                               kDSPI_RxFifoDrainRequestInterruptEnable | kDSPI_TxFifoFillRequestInterruptEnable);
    }
/* Add for ARM errata 838869, affects Cortex-M4, Cortex-M4F Store immediate overlapping
  exception return operation might vector to incorrect interrupt */
#if defined __CORTEX_M && (__CORTEX_M == 4U)
    __DSB();
#endif
}

void Prepare_FSCI_Response(uint8_t *Rxdata)
{
	int i = 0, j = 0, index = 0, temp = 0;
	uint16_t len;

	memset(buff_data, '\0', DATA_SIZE);

	for(i = 0; i < DATA_SIZE; i++)
	{
		if(Rxdata[i] == FSCI_SYNC_BYTE)
		{
			i++;
			if((Rxdata[i] == bleGATT) || (Rxdata[i] == bleGATTDB) || (Rxdata[i] == bleGAP))
			{
				temp = i - 1;

				/* Increment i by 4, as opgroup-1 byte,opcode-1 byte and len-2 bytes */
				i = i + 4;
				len = ((uint16_t)Rxdata[i-1] << 8) | Rxdata[i-2];

				for(j = 0; temp < i; j++,temp++)
				{
					buff_data[j] = Rxdata[temp];
				}

				for(index = 0; index < len; index++)
				{
					buff_data[j++] = Rxdata[i++];
				}

				buff_data[j] = Rxdata[i];
				uint8_t size = j + 1;
				uint8_t *send_data = malloc(size);
				uint8_t data = 0;
				memset(send_data, '\0', size);

				for(data = 0; data < size; data++)
				{
					send_data[data] = buff_data[data];
				}

				OSA_MsgQPut(&rx_msg_queue, send_data);

				/* Signal BLE Abstraction thread for response  message into message queue */
				OSA_EventSet(&evt_handle, gEvtMsgFromFSCI_c);

				/* Variable that indicates number of times event is set */
				item_cnt += 1;

				/* Let FSCI receive thread handle previous response */
				vTaskDelay(pdMS_TO_TICKS(20));

				free(send_data);
				send_data = NULL;
			}
		}
	}
}

void SPI_SendData(uint8_t *buffer_ptr, uint16_t buffer_size)
{
	uint64_t i;

	TRANSFER_SIZE = buffer_size;

	for(i = 0; i < TRANSFER_SIZE; i++)
	{
		masterTxData[i] = buffer_ptr[i];
	}

	masterTxCount = 0;
	masterRxCount = 0;

	isTransferCompleted = false;
	DSPI_StopTransfer(EXAMPLE_DSPI_MASTER_BASEADDR);
	DSPI_FlushFifo(EXAMPLE_DSPI_MASTER_BASEADDR, true, true);
	DSPI_ClearStatusFlags(EXAMPLE_DSPI_MASTER_BASEADDR, (uint32_t)kDSPI_AllStatusFlag);

	/*Fill up the master Tx data*/
	while (DSPI_GetStatusFlags(EXAMPLE_DSPI_MASTER_BASEADDR) & kDSPI_TxFifoFillRequestFlag)
	{
		if (masterTxCount < TRANSFER_SIZE)
		{
			DSPI_MasterWriteData(EXAMPLE_DSPI_MASTER_BASEADDR, &commandData, masterTxData[masterTxCount]);
			++masterTxCount;
		}
		else
		{
			break;
		}
		/* Try to clear the TFFF; if the TX FIFO is full this will clear */
		DSPI_ClearStatusFlags(EXAMPLE_DSPI_MASTER_BASEADDR, kDSPI_TxFifoFillRequestFlag);
	}

	/*Enable master RX interrupt*/
	DSPI_EnableInterrupts(EXAMPLE_DSPI_MASTER_BASEADDR, kDSPI_RxFifoDrainRequestInterruptEnable);
	DSPI_StartTransfer(EXAMPLE_DSPI_MASTER_BASEADDR);

	/* Wait 1st round transfer complete */
	while (!isTransferCompleted)
	{
	}

	/* Delay to wait slave is ready */
	for (i = 0; i < (DSPI_DELAY_COUNT/4); i++)
	{
		__NOP();
	}
}

/*! *********************************************************************************
* \brief        SPI response monitoring timer callback.
*
* \param[in]    xTimer        Timer handle.
********************************************************************************** */
void RespMonitorTimerCallback(TimerHandle_t xTimer)
{
	TRANSFER_SIZE = DATA_SIZE;
	uint64_t i;

	/* Clear the TX data.*/
	for (i = 0U; i < TRANSFER_SIZE; i++)
	{
		masterTxData[i] = 0xFF;
	}

	/* Clear the RX data.*/
	for (i = 0U; i < TRANSFER_SIZE; i++)
	{
		masterRxData[i] = 0U;
	}

	 masterTxCount       = 0;
	 masterRxCount       = 0;

	 isTransferCompleted = false;
	 DSPI_StopTransfer(EXAMPLE_DSPI_MASTER_BASEADDR);
	 DSPI_FlushFifo(EXAMPLE_DSPI_MASTER_BASEADDR, true, true);
	 DSPI_ClearStatusFlags(EXAMPLE_DSPI_MASTER_BASEADDR, (uint32_t)kDSPI_AllStatusFlag);

	while (DSPI_GetStatusFlags(EXAMPLE_DSPI_MASTER_BASEADDR) & kDSPI_TxFifoFillRequestFlag)
	{
	 if (masterTxCount < TRANSFER_SIZE)
	 {
		 DSPI_MasterWriteData(EXAMPLE_DSPI_MASTER_BASEADDR, &commandData, masterTxData[masterTxCount]);
		 ++masterTxCount;
	 }
	 else
	 {
		 break;
	 }

	 /* Try to clear the TFFF; if the TX FIFO is full this will clear */
	 DSPI_ClearStatusFlags(EXAMPLE_DSPI_MASTER_BASEADDR, kDSPI_TxFifoFillRequestFlag);
	}

	/*Enable master RX interrupt*/
	DSPI_EnableInterrupts(EXAMPLE_DSPI_MASTER_BASEADDR, kDSPI_RxFifoDrainRequestInterruptEnable);
	DSPI_StartTransfer(EXAMPLE_DSPI_MASTER_BASEADDR);

	/* Wait 2nd round transfer complete */
	while (!isTransferCompleted)
	{
	}

	Prepare_FSCI_Response(masterRxData);
}
