/* main.c */
#include "fsl_common.h"
#include "clock_config.h"

#include "led.h"
#include "touch.h"
#include "touch_2d.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/
/* for cpu counter.
 * systick is using core_clk on ke1xz platform.
 */
#define CPU_COUNTER_START()   do { SysTick->LOAD = 0xffffff; SysTick->VAL = 0; SysTick->CTRL = 0x5; } while(0)
#define CPU_COUNTER_STOP(val) do { SysTick->CTRL = 0; val = SysTick->VAL; val = 0xFFFFFF - val; } while(0)
volatile uint32_t app_cpu_counter_val, app_cpu_counter_us;
volatile uint32_t app_touch_scan_us = 0u, app_touch_scan_ms = 0u;
volatile uint32_t app_touch_calc_us = 0u, app_touch_calc_ms = 0u;

/* touch state machine. */
typedef enum
{
    eAppTouchState_Init   = 0,
    eAppTouchState_Calib  = 1,
    eAppTouchState_Detect = 2,
} app_touch_state_t;

/*******************************************************************************
 * Variables.
 ******************************************************************************/
volatile app_touch_state_t app_touch_state = eAppTouchState_Init;
int32_t app_touch_state_init_counter = 0;     /* for warm up.. */

int32_t app_touch_channel_values[TOUCH_2D_ALL_COUNT];
int32_t app_touch_2d_x_pos_raw;
int32_t app_touch_2d_y_pos_raw;
volatile int32_t app_touch_2d_x_pos;
volatile int32_t app_touch_2d_y_pos;
int32_t app_touch_err;

int main(void)
{
    BOARD_InitBootClocks(); /* setup clocks. */

    led_init();
    led_on(LED_RED_MASK);

    touch_init_pins();
    touch_init();

    /* measure the sampling time. */
    CPU_COUNTER_START();
    touch_start();
    touch_wait_data_ready(app_touch_channel_values);
    CPU_COUNTER_STOP(app_cpu_counter_val);
    app_touch_scan_us = app_cpu_counter_val / 48u;
    app_touch_scan_ms = app_touch_scan_us / 1000u;

    /* start the scan. */
    touch_start();

    app_touch_state_init_counter = 0u;
    app_touch_state = eAppTouchState_Init;

    while (1)
    {
        touch_wait_data_ready(app_touch_channel_values);
        touch_start();

        /* todo: using app_touch_channel_values for calculation.*/
        switch (app_touch_state)
        {
        case eAppTouchState_Init:
            if (app_touch_state_init_counter < 10u)
            {
                app_touch_state_init_counter++;
            }
            else
            {
                app_touch_state = eAppTouchState_Calib;
            }
            break;

        case eAppTouchState_Calib:
            if (touch_2d_calib(app_touch_channel_values))
            {
                app_touch_state = eAppTouchState_Detect;
                led_off(LED_RED_MASK); /* calib done. */
            }
            break;

        case eAppTouchState_Detect:
            CPU_COUNTER_START();
            app_touch_err = touch_2d_calc_position(app_touch_channel_values, &app_touch_2d_x_pos_raw, &app_touch_2d_y_pos_raw);
            CPU_COUNTER_STOP(app_cpu_counter_val);
            app_touch_calc_us = app_cpu_counter_val / 48u;
            app_touch_calc_ms = app_touch_calc_us / 1000u;
            if (app_touch_err == 0)
            {
                app_touch_2d_x_pos = app_touch_2d_x_pos_raw;
                app_touch_2d_y_pos = app_touch_2d_y_pos_raw;
                led_on(LED_GREEN_MASK);
            }
            else
            {
                app_touch_2d_x_pos = 0;
                app_touch_2d_y_pos = 0;
                led_off(LED_GREEN_MASK);
            }
            break;

        default:
            app_touch_state_init_counter = 0u;
            app_touch_state = eAppTouchState_Init;
            break;
        }
    }
}
/* EOF. */

