#include <hidef.h>                  /* for EnableInterrupts macro */
#include "derivative.h"             /* include peripheral declarations */
#include "mc9s08pb16.h"
#include "motor.h"

/*------------------------------------
 * User state machine functions
 * ----------------------------------*/

extern FCN_POINTER BLDCProcess_MainIndexFcn[];
 
/**********************************************************************************
 * Local variable
 **********************************************************************************/
BLDC_STATE_ENUM      eBldcStateIndex; 
BLDC_FAULT_FLAGS     unBldcFaultFlags; 
CTR_sDrive gsMC_Drive;

/* Hall_Signal = PTA2 * 0x04 + PTA1 * 0x02 + PTA0 * 0x01
 * 
 *        PTA2:    1    0    0    0    1    1
 *        PTA1:    1    1    1    0    0    0
 *        PTA0:    0    0    1    1    1    0       
 * Hall_Signal:    6    2    3    1    5    4
 * Hall Vector:    1    2    3    4    5    6
 */
static uint8_t vector_Table[8] = {0,4,2,3,6,5,1,0};

/* Demo mode enabled/disabled */
uint8_t bDemoMode = FALSE;
uint16_t uw16Duty =0;
uint8_t Limit_Current = 40;     /* Current:4A, Limit_Current value set to: 4A * 10 */
uint8_t ACMPvalue = 0;

/* Counter for button pressing */
unsigned char buttonStates2 = 0;
#define Button3_Is_Pressed()	!(buttonStates2 & (1<<2))

/************************************************************************
*       main - Program main entry point                                                                         *
*************************************************************************/
void main(void) 
{ 
    /*Initialize system including clocks
     *Watchdog: disabled
     *Default FEI: BUS clock 16MHz      
     */
    Sys_Init();
    EnableInterrupts; 

    PRINTF("\n-- S08PB16-EVK BLDC DEMO BEGINS --\n");
    PRINTF("\n");

    /* Initialize MTIM1, 1ms interrupt interval */
    MTIM1_Init(); 
    /* Initialize ADC, 12bit mode */
    Init_ADC();
    /* Configure the priority of FTM2,MTIM1 */
    IPCInit();
    /* Configure PTA0/PTA1/PTA2 as input */
    Init_HALL();
    /* Initialize PB16 internal OPAMP */
//    Init_OPAMP();   
    /* Initialize ACMP1, ACMP1+ input: PTA3 input (this pin is connect external OPAMP) */
    Init_ACMP1();
    /* Initialize fault detection and shutdowm module to control 6 pins */
    Init_FDS();    
    /* Initialize KBI, press SW3 to turn on/off demo mode */
    Init_Buttons();
    /* Initialize FTM2, output 6 PWMs */
    Init_FTM2();    
    MTIM1_Start();
    
    while (1)
    	{
    	
    	}
}


/* Slow loop control (1ms period) */
interrupt VectorNumber_Vmtim1 void  Mtim1_ISR(void)
{
    if(MTIM1_SC_TOF)
    {
        /* clear the ISR flag */
        MTIM1_SC_TOF = 0;	
    }
    EnableInterrupts;
    
    /* Button detection */
    if(KBI0_SC_KBF)
    {
        KBI0_SC_KBACK = 1;                       /* clear the KBI interrupt flag */
        KBI0_PE &= (~KBI0_PE_KBIPE6_MASK);       /* disable KBI pin */
        PORT_PTBIE |= (1<<2);                    /* enable GPIO input */
        asm(nop);
        asm(nop);
        buttonStates2 = PORT_PTBD;               /* read PORT data register to recognize which button is pushed */

        if(Button3_Is_Pressed())
        {	
        	/* SW3 is pressed. */
        	if (bDemoMode)
        	{
        	    /* turn off demo mode */
        		bDemoMode = FALSE;
              	gsMC_Drive.w16Speed_req = 0;
        	}
        	else
        	{
        	    /* turn on demo mode */
        		bDemoMode = TRUE;
        	}
        }
        /* restore KBI pin function after PORT data register */
        KBI0_PE = KBI0_PE_KBIPE6_MASK; /* enable KBI pin */
        KBI0_SC_KBACK = 1;	
    }
    /* demo mode function */
    DemoSpeedStimulator();
   
    /* Change the limit current, default limit current is about 4A*/
    if(Limit_Current < 40)
    {
        /* Calculate DACVAL.*/
        /* (DACVAL+1)/64 * Vin = 1.65V + 0.2V/A * Limit_Current */
        gsMC_Drive.ACMP_DACValue = (uint8_t)(((21120 + 256 * Limit_Current)/1000)-1);
       
        /* Update ACMP1_C1_DACVAL */
    	ACMP1_C1_DACVAL = gsMC_Drive.ACMP_DACValue;
        ACMPvalue = ACMP1_C1_DACVAL;
    }

    BLDCProcess_MainIndexFcn[eBldcStateIndex]();
      
	IPC_SC_PULIPM = 1;//reload interrupt priority settings
}

interrupt VectorNumber_Vftm2ovf void  FTM2_ISR(void)
{
	/*Clear TOF bit*/
    if(FTM2_SC_TOF)
    {
    	FTM2_SC_TOF = 0;
    	gsMC_Drive.uw16FTM_cntr ++;	
    }
    EnableInterrupts;

    PORT_PTBD_PTBD2   = 1;
	
    gsMC_Drive.HALL_signal = 0;
    
	if(HALL_3) 
		/* If Hall sensor is active, store 4 in HALL_vector variable */
		gsMC_Drive.HALL_signal |= 0x04;  
	if(HALL_2) 
		/* If Hall sensor is active, store 4 in HALL_vector variable */
		gsMC_Drive.HALL_signal |= 0x02;  /* If Hall sensor is active, store 2 in HALL_vector variable */
	if(HALL_1) 
		/* If Hall sensor is active, store 4 in HALL_vector variable */
		gsMC_Drive.HALL_signal |= 0x01;  /* If Hall sensor is active, store 1 in HALL_vector variable */
	
	gsMC_Drive.HALL_vector = vector_Table[gsMC_Drive.HALL_signal];
	if(uw16Duty ==0)
	{
		gsMC_Drive.Direction = gsMC_Drive.w16Speed_req >0?1:0;
	}
	if(eBldcStateIndex == BLDC_RUN)
	{
		Commutation(gsMC_Drive.HALL_vector,gsMC_Drive.Direction);
	    PWM_DutyUpdate(uw16Duty);
	}
	else
	{
		PWM_Disable();
	    PWM_DutyUpdate(0);
	}
	/* Measure actual speed */
	Speed_Measure(&gsMC_Drive);
    
	IPC_SC_PULIPM = 1;//reload interrupt priority settings
}

/* ACMP1 generate interrupt on raising / falling edge */
interrupt VectorNumber_Vacmp1 void ACMP1_ISR(void)
{
	/* clear ACMP1 interrupt flag */
	ACMP1_CS_ACF = 0;          
	unBldcFaultFlags.Bits.Hard_OCFault = 1;
}
