#include "mc9s08pb16.h"
#include "main.h"
#include "motor.h"


/*******************************************************************************
 * Local functions
 ******************************************************************************/


static void FaultDetect(CTR_sDrive * gsMC_Drive,BLDC_FAULT_FLAGS * unBLDCFaultFlags);
static void Variable_Init(void);
static void GetADCValue(CTR_sDrive * gsMC_Drive);

static void AppInit(void);
static void AppStop(void);
static void AppRun(void);
static void AppFault(void);

/*------------------------------------
 * User state machine functions
 * ----------------------------------*/

FCN_POINTER BLDCProcess_MainIndexFcn[] = { 	AppInit,
											AppStop,                                            
                                            AppRun,
                                            AppFault
										  };

/*------------------------------------
 * User state transition functions
 * ----------------------------------*/
static void AppInitToStop(void);
static void AppStopToRun(void);
static void AppStopToFault(void);
static void AppRunToStop(void);
static void AppRunToFault(void);
static void AppFaultToStop(void);

/**********************************************************************************
 * Local variable
 **********************************************************************************/
extern BLDC_STATE_ENUM      eBldcStateIndex;
extern BLDC_FAULT_FLAGS     unBldcFaultFlags;  
extern CTR_sDrive gsMC_Drive;
extern uint16_t uw16Duty;
static uint16_t uw16FaultReleaseTime = 2000;
extern uint8_t bDemoMode;

/* Test the speed 1000,2000,1000,-1000,-2000,-1000 in demo mode,
 * Scaled speed = speed value/N_MAX*32768, N_MAX: 100000
 * The calculation results: 328,655,328,-328,-655,-328  */
static uint16_t speed_Demo[6] = {328,655,328,-328,-655,-328};


void PWM_Disable(void)
{
	/* set all pwm channel to software control and set the level to 0*/
	  SYS_SOPT8 = 0;
	  SYS_SOPT7 = SYS_SOPT7_FTM2CH0OC_MASK | SYS_SOPT7_FTM2CH1OC_MASK \
			  | SYS_SOPT7_FTM2CH2OC_MASK | SYS_SOPT7_FTM2CH3OC_MASK \
			  | SYS_SOPT7_FTM2CH4OC_MASK | SYS_SOPT7_FTM2CH5OC_MASK;
}
void PWM_DutyUpdate(uint16_t uw16pwmduty)
{
    FTM2_C0V = uw16pwmduty;                  
    FTM2_C2V = uw16pwmduty;                   
    FTM2_C4V = uw16pwmduty; 	
}

/***************************************************************************//*!
*
* @brief   INIT state
*
* @param   void
*
* @return  none
*
******************************************************************************/
static void AppInit()
{
	Variable_Init();
	GetADCValue(&gsMC_Drive);
	AppInitToStop();
}

/***************************************************************************//*!
*
* @brief   STOP state
*
* @param   void
*
* @return  none
*
******************************************************************************/
static void AppStop()
{
	GetADCValue(&gsMC_Drive);
	FaultDetect(&gsMC_Drive,&unBldcFaultFlags);
	if (abs(gsMC_Drive.w16Speed_req) >= (signed int)(VELOCITY_MIN))	
	{
		AppStopToRun();
	}
	if(unBldcFaultFlags.Byte)
	{
		AppStopToFault();
	}
}


/***************************************************************************//*!
*
* @brief   RUN state
*
* @param   void
*
* @return  none
*
******************************************************************************/
static void AppRun()
{	
	GetADCValue(&gsMC_Drive);
	FaultDetect(&gsMC_Drive,&unBldcFaultFlags);	

	gsMC_Drive.w16Speed_ramp = ECLIB_Ramp16(RAMP_SPEED, gsMC_Drive.w16Speed_req, gsMC_Drive.w16Speed_ramp);    
	gsMC_Drive.w16Speed_err =  sub16(gsMC_Drive.w16Speed_ramp,gsMC_Drive.w16Speed_Act_flt);

	if(gsMC_Drive.Direction ==0)
	{
		gsMC_Drive.w16Speed_err = -gsMC_Drive.w16Speed_err;
	}
	uw16Duty = CTR_ControllerPI16(gsMC_Drive.w16Speed_err,&gsMC_Drive.sSpeedPiParams);

	
	if ((abs(gsMC_Drive.w16Speed_req) < (signed int)(VELOCITY_MIN))&&\
			(abs(gsMC_Drive.w16Speed_ramp) < (signed int)(VELOCITY_MIN)))	
	{
		AppRunToStop();
	}
	
	if(unBldcFaultFlags.Byte)
	{
		AppRunToFault();
	}
}

/***************************************************************************//*!
*
* @brief   Fault state
*
* @param   void
*
* @return  none
*
******************************************************************************/
static void AppFault()
{
	/*six channels disable*/
	PWM_Disable();
	GetADCValue(&gsMC_Drive);
	FaultDetect(&gsMC_Drive,&unBldcFaultFlags);
	/* when ACMP1 output is 0(fault disappear), remove Hardware over current fault flag */
	if(ACMP1_CS_ACO == 0)
	{
		unBldcFaultFlags.Bits.Hard_OCFault = 0;
		FDS_CS_FDF = 0;    // clear FDS interrupt flag
	}
	if(unBldcFaultFlags.Byte ==0)
	{
		if(--uw16FaultReleaseTime == 0)
		{
			AppFaultToStop();
		}
	}	
}


/***************************************************************************//*!
*
* @brief   INIT to STOP transition
*
* @param   void
*
* @return  none
*
******************************************************************************/
static void AppInitToStop()
{
	eBldcStateIndex = BLDC_STOP;
}

/***************************************************************************//*!
*
* @brief   STOP to RUN transition
*
* @param   void
*
* @return  none
*
******************************************************************************/
static void AppStopToRun()
{
	eBldcStateIndex = BLDC_RUN;
}

/***************************************************************************//*!
*
* @brief   STOP to FAULT transition
*
* @param   void
*
* @return  none
*
******************************************************************************/
static void AppStopToFault()
{
	eBldcStateIndex = BLDC_FAULT;
}

/***************************************************************************//*!
*
* @brief   RUN to STOP transition
*
* @param   void
*
* @return  none
*
******************************************************************************/
static void AppRunToStop()
{
	Variable_Init();
	/* Disable PWM */
	PWM_Disable();
	eBldcStateIndex = BLDC_STOP;
}


/***************************************************************************//*!
*
* @brief   RUN to FAULT transition
*
* @param   void
*
* @return  none
*
******************************************************************************/
static void AppRunToFault()
{
	eBldcStateIndex = BLDC_FAULT;
}

/***************************************************************************//*!
*
* @brief   FAULT to STOP transition
*
* @param   void
*
* @return  none
*
******************************************************************************/
static void AppFaultToStop()
{
	Variable_Init();
	eBldcStateIndex = BLDC_STOP;
	uw16FaultReleaseTime = 2000;//2s fault release time

}


/*!
* @brief   DemoSpeedStimulator
*           - When demo mode is enabled it changes the required speed according
*             to predefined profile
*
* @param   void
*
* @return  none
*/
void DemoSpeedStimulator(void)
{
    /* Increase push button pressing counter  */ 
    if (bDemoMode)
    {
    	/* Counters used for demo mode */
    	gsMC_Drive.uw32SpeedStimulatorCnt++;
        switch (gsMC_Drive.uw32SpeedStimulatorCnt)
        {
                             
            case 100:
              {
            	  gsMC_Drive.w16Speed_req = speed_Demo[0];
                break;
              }
            case 3000:
              {
              	  gsMC_Drive.w16Speed_req = speed_Demo[1];
                break;
              }
            case 6000:
              {
              	  gsMC_Drive.w16Speed_req = speed_Demo[2];
                break;
              }
            case 9000:
              {
              	  gsMC_Drive.w16Speed_req = speed_Demo[3];
                break;
              }
            case 12000:
              {
              	  gsMC_Drive.w16Speed_req = speed_Demo[4];
                break;
              }
            case 15000:
              {
              	gsMC_Drive.w16Speed_req = speed_Demo[5];
                break;
              }
            case 18000:
              {
              	gsMC_Drive.uw32SpeedStimulatorCnt = 0;
                break;
              }
            default:
                break;
        }
    }
    else
    {
    	gsMC_Drive.uw32SpeedStimulatorCnt = 0;
    }
}


/* Local function */
static void Variable_Init()
{
	eBldcStateIndex = BLDC_INIT;
	unBldcFaultFlags.Byte = 0;  

	gsMC_Drive.uw16FTM_cntr = 0;
	gsMC_Drive.uw16FTM_cntr_last = 0;

	gsMC_Drive.uw16Comm_Period = 0;
	gsMC_Drive.uw16Comm_Cntr = 0;
	gsMC_Drive.uw16Speed_Act = 0;
	gsMC_Drive.uw16Speed_Act_Tmp = 0;
	gsMC_Drive.uw16Speed_Act_Tmp0 = 0;
	gsMC_Drive.uw16Speed_Act_Tmp1 = 0;
	gsMC_Drive.uw16Speed_Act_Tmp2 = 0;

	gsMC_Drive.w16Speed_Act_flt = 0;
	gsMC_Drive.w16Speed_ramp = 0;
	gsMC_Drive.w16Speed_err = 0;
	gsMC_Drive.w16Speed_req = 0;
	
	gsMC_Drive.uw16CoefVelocityPeriodScale = NUMERATOR_FOR_SPEED;
	gsMC_Drive.sSpeedPiParams.propGain = 10;
	gsMC_Drive.sSpeedPiParams.intGain = 1;
	gsMC_Drive.sSpeedPiParams.propGainShift = 0;
	gsMC_Drive.sSpeedPiParams.intGainShift = 0;
	gsMC_Drive.sSpeedPiParams.upperLimit = PWM_MODULO;
	gsMC_Drive.sSpeedPiParams.lowerLimit = 0;
	gsMC_Drive.sSpeedPiParams.integralPortionK_1_32.HL_W16.H = 0;
	gsMC_Drive.sSpeedPiParams.integralPortionK_1_32.HL_W16.L = 0;
	gsMC_Drive.sSpeedPiParams.integralPortionK_1_32.S32 = 0;
	gsMC_Drive.sSpeedPiParams.propk_1 = 0;
	
    
	gsMC_Drive.HALL_signal = 0;
	gsMC_Drive.HALL_vector = 0;
	gsMC_Drive.HALL_vector_pre = 0;
	gsMC_Drive.uw32SpeedStimulatorCnt = 0;
	gsMC_Drive.ACMP_DACValue = 0;
	uw16Duty = 0;
	uw16FaultReleaseTime = 2000;

}



void Commutation(uint8_t HALL_vector,byte Direction)
{
	if (Direction == 0)
	{
		switch(HALL_vector)
		{			
		  case (4)://AB :Q1PWM/Q2OFF/Q3OFF/Q4ON/Q5OFF/Q6OFF
			  {
			  	  SYS_SOPT8 = SYS_SOPT8_FTM2CH3OCV_MASK;
			  	  SYS_SOPT7 = SYS_SOPT7_FTM2CH1OC_MASK \
			  			  | SYS_SOPT7_FTM2CH2OC_MASK | SYS_SOPT7_FTM2CH3OC_MASK \
			  			  | SYS_SOPT7_FTM2CH4OC_MASK | SYS_SOPT7_FTM2CH5OC_MASK;
				  break;
			  }
		  case (5)://AC :Q1PWM/Q2OFF/Q3OFF/Q4OFF/Q5OFF/Q6ON
			  {
		  	  	  SYS_SOPT8 = SYS_SOPT8_FTM2CH5OCV_MASK;
		  	  	  SYS_SOPT7 = SYS_SOPT7_FTM2CH1OC_MASK \
		  	  			  | SYS_SOPT7_FTM2CH2OC_MASK | SYS_SOPT7_FTM2CH3OC_MASK \
		  	  			  | SYS_SOPT7_FTM2CH4OC_MASK | SYS_SOPT7_FTM2CH5OC_MASK;
				  break;
			  }
		  case (6)://BC :Q1OFF/Q2OFF/Q3PWM/Q4OFF/Q5OFF/Q6ON
			  {
	  	  	  	  SYS_SOPT8 = SYS_SOPT8_FTM2CH5OCV_MASK;
			  	  SYS_SOPT8 = SYS_SOPT8_FTM2CH5OCV_MASK;
			  	  SYS_SOPT7 = SYS_SOPT7_FTM2CH0OC_MASK | SYS_SOPT7_FTM2CH1OC_MASK \
			  			  | SYS_SOPT7_FTM2CH3OC_MASK \
			  			  | SYS_SOPT7_FTM2CH4OC_MASK | SYS_SOPT7_FTM2CH5OC_MASK;
				  break;
			  }
		  case (1)://BA :Q1OFF/Q2ON/Q3PWM/Q4OFF/Q5OFF/Q6OFF
			  {		  
		  	  	  SYS_SOPT8 = SYS_SOPT8_FTM2CH1OCV_MASK;
		  	  	  SYS_SOPT7 = SYS_SOPT7_FTM2CH0OC_MASK | SYS_SOPT7_FTM2CH1OC_MASK \
		  	  			  | SYS_SOPT7_FTM2CH3OC_MASK \
		  	  			  | SYS_SOPT7_FTM2CH4OC_MASK | SYS_SOPT7_FTM2CH5OC_MASK;
				  break;
			  }
		  case (2)://CA :Q1OFF/Q2ON/Q3OFF/Q4OFF/Q5PWM/Q6OFF
			  {
			  	  SYS_SOPT8 = SYS_SOPT8_FTM2CH1OCV_MASK;
			  	  SYS_SOPT7 = SYS_SOPT7_FTM2CH0OC_MASK | SYS_SOPT7_FTM2CH1OC_MASK \
			  			  | SYS_SOPT7_FTM2CH2OC_MASK | SYS_SOPT7_FTM2CH3OC_MASK \
			  			  | SYS_SOPT7_FTM2CH5OC_MASK;
				  break;
			  }
		  case (3)://CB :Q1OFF/Q2OFF/Q3OFF/Q4ON/Q5PWM/Q6OFF
			  {
			  	  SYS_SOPT8 = SYS_SOPT8_FTM2CH3OCV_MASK;
			  	  SYS_SOPT7 = SYS_SOPT7_FTM2CH0OC_MASK | SYS_SOPT7_FTM2CH1OC_MASK \
			  			  | SYS_SOPT7_FTM2CH2OC_MASK | SYS_SOPT7_FTM2CH3OC_MASK \
			  			  | SYS_SOPT7_FTM2CH5OC_MASK;
				  break;
			  }
		  default :	  break;
	}
	}
	else
	{
			switch(HALL_vector)
			{
			  case (1)://AB :Q1PWM/Q2OFF/Q3OFF/Q4ON/Q5OFF/Q6OFF
				  {
				  	  SYS_SOPT8 = SYS_SOPT8_FTM2CH3OCV_MASK;
				  	  SYS_SOPT7 = SYS_SOPT7_FTM2CH1OC_MASK \
				  			  | SYS_SOPT7_FTM2CH2OC_MASK | SYS_SOPT7_FTM2CH3OC_MASK \
				  			  | SYS_SOPT7_FTM2CH4OC_MASK | SYS_SOPT7_FTM2CH5OC_MASK;
					  break;
				  }
			  case (2)://AC :Q1PWM/Q2OFF/Q3OFF/Q4OFF/Q5OFF/Q6ON
				  {
			  	  	  SYS_SOPT8 = SYS_SOPT8_FTM2CH5OCV_MASK;
			  	  	  SYS_SOPT7 = SYS_SOPT7_FTM2CH1OC_MASK \
			  	  			  | SYS_SOPT7_FTM2CH2OC_MASK | SYS_SOPT7_FTM2CH3OC_MASK \
			  	  			  | SYS_SOPT7_FTM2CH4OC_MASK | SYS_SOPT7_FTM2CH5OC_MASK;
					  break;
				  }
			  case (3)://BC :Q1OFF/Q2OFF/Q3PWM/Q4OFF/Q5OFF/Q6ON
				  {
		  	  	  	  SYS_SOPT8 = SYS_SOPT8_FTM2CH5OCV_MASK;
				  	  SYS_SOPT8 = SYS_SOPT8_FTM2CH5OCV_MASK;
				  	  SYS_SOPT7 = SYS_SOPT7_FTM2CH0OC_MASK | SYS_SOPT7_FTM2CH1OC_MASK \
				  			  | SYS_SOPT7_FTM2CH3OC_MASK \
				  			  | SYS_SOPT7_FTM2CH4OC_MASK | SYS_SOPT7_FTM2CH5OC_MASK;
					  break;
				  }
			  case (4)://BA :Q1OFF/Q2ON/Q3PWM/Q4OFF/Q5OFF/Q6OFF
				  {		  
			  	  	  SYS_SOPT8 = SYS_SOPT8_FTM2CH1OCV_MASK;
			  	  	  SYS_SOPT7 = SYS_SOPT7_FTM2CH0OC_MASK | SYS_SOPT7_FTM2CH1OC_MASK \
			  	  			  | SYS_SOPT7_FTM2CH3OC_MASK \
			  	  			  | SYS_SOPT7_FTM2CH4OC_MASK | SYS_SOPT7_FTM2CH5OC_MASK;
					  break;
				  }
			  case (5)://CA :Q1OFF/Q2ON/Q3OFF/Q4OFF/Q5PWM/Q6OFF
				  {
				  	  SYS_SOPT8 = SYS_SOPT8_FTM2CH1OCV_MASK;
				  	  SYS_SOPT7 = SYS_SOPT7_FTM2CH0OC_MASK | SYS_SOPT7_FTM2CH1OC_MASK \
				  			  | SYS_SOPT7_FTM2CH2OC_MASK | SYS_SOPT7_FTM2CH3OC_MASK \
				  			  | SYS_SOPT7_FTM2CH5OC_MASK;
					  break;
				  }
			  case (6)://CB :Q1OFF/Q2OFF/Q3OFF/Q4ON/Q5PWM/Q6OFF
				  {
				  	  SYS_SOPT8 = SYS_SOPT8_FTM2CH3OCV_MASK;
				  	  SYS_SOPT7 = SYS_SOPT7_FTM2CH0OC_MASK | SYS_SOPT7_FTM2CH1OC_MASK \
				  			  | SYS_SOPT7_FTM2CH2OC_MASK | SYS_SOPT7_FTM2CH3OC_MASK \
				  			  | SYS_SOPT7_FTM2CH5OC_MASK;
					  break;
				  }
			  default :	  break;
			  
			}               
		
	}     	
}

void Speed_Measure(CTR_sDrive * s_Drive)
{
	if (s_Drive -> HALL_vector_pre == s_Drive -> HALL_vector)
	{
		s_Drive -> uw16Comm_Cntr++;
		if (s_Drive -> uw16Comm_Cntr > s_Drive -> uw16Comm_Period)
		{
			s_Drive -> uw16Comm_Period++;
		}
	}
	else
	{
		s_Drive -> uw16Comm_Cntr = 0;
		if(s_Drive -> uw16FTM_cntr > s_Drive -> uw16FTM_cntr_last)
		{
			s_Drive -> uw16Comm_Period = usub16(s_Drive -> uw16FTM_cntr,s_Drive -> uw16FTM_cntr_last);
		}
		else
		{
			s_Drive -> uw16Comm_Period = uadd16(s_Drive -> uw16FTM_cntr,usub16(65535,s_Drive -> uw16FTM_cntr_last));
		}
		s_Drive -> uw16FTM_cntr_last = s_Drive -> uw16FTM_cntr;
		
		if((subu16_u16to16(s_Drive -> HALL_vector ,s_Drive -> HALL_vector_pre)==1)||(subu16_u16to16(s_Drive -> HALL_vector ,s_Drive -> HALL_vector_pre)==-5))
		{
			s_Drive ->Direction =1;
		}
		else
		{
			s_Drive ->Direction =0;
		}
	}
	s_Drive -> HALL_vector_pre = s_Drive -> HALL_vector;	
	
	if (s_Drive -> uw16Comm_Period < 20 )
	{
		s_Drive -> uw16Comm_Period = 20;
	}
	if (s_Drive -> uw16Comm_Period > 255 )
	{
		s_Drive -> uw16Comm_Period = 255;
	}
	/* n = NUMERATOR_FOR_SPEED / uw16Comm_Period = (60*PWM_FREQUENCY_KHZ*1000.0/(6.0*uw16Comm_Period/PP)/N_MAX)*32768.0)*/
	s_Drive -> uw16Speed_Act = (unsigned int)(s_Drive -> uw16CoefVelocityPeriodScale/s_Drive -> uw16Comm_Period);
	if(s_Drive -> uw16Speed_Act < (signed int)(VELOCITY_MIN))
	{
		s_Drive -> uw16Speed_Act = 0;
	}
	s_Drive -> uw16Speed_Act_Tmp0 = s_Drive -> uw16Speed_Act << 5;//shift to increase the accuracy 
	s_Drive -> uw16Speed_Act_Tmp1 = umul16_8to16(s_Drive -> uw16Speed_Act_Tmp0,2); 
	s_Drive -> uw16Speed_Act_Tmp2 = umul16_8to16(s_Drive -> uw16Speed_Act_Tmp,254);
	s_Drive -> uw16Speed_Act_Tmp = uadd16(s_Drive -> uw16Speed_Act_Tmp1,s_Drive -> uw16Speed_Act_Tmp2);	
	if(s_Drive ->Direction)
	{
		s_Drive -> w16Speed_Act_flt = s_Drive -> uw16Speed_Act_Tmp >>5;//shift back
	}
	else
	{
		s_Drive -> w16Speed_Act_flt = -(signed int)(s_Drive -> uw16Speed_Act_Tmp >>5);//shift back
	}
}



void FaultDetect(CTR_sDrive * gsMC_Drive,BLDC_FAULT_FLAGS * unBLDCFaultFlags)
{
	unBLDCFaultFlags->Bits.OverVoltageFault =   gsMC_Drive->w16Vdc > (int16_t)(Voltage_MAX)? 1:0;
	unBLDCFaultFlags->Bits.UnderVoltageFault =   gsMC_Drive->w16Vdc < (int16_t)(Voltage_MIN)? 1:0;
	unBLDCFaultFlags->Bits.Soft_OCFault =   gsMC_Drive->w16Idc > (int16_t)(IDC_MAX)? 1:0;
}

void GetADCValue(CTR_sDrive * gsMC_Drive)
{
    gsMC_Drive->w16Vdc = ADC_GetResult(ADC_VDC_CHANNEL)<<3;
    
	if(gsMC_Drive->uw16IDCOffset_cntr <= 363) // Calibration IDC offset will last 363ms
	{
		gsMC_Drive->uw16IDCOffset_cntr++;
		if((gsMC_Drive->uw16IDCOffset_cntr >299)&&(gsMC_Drive->uw16IDCOffset_cntr <= 363))
		{    	
			gsMC_Drive->w32Idc_offsetTmp += ADC_GetResult(ADC_IDC_CHANNEL)<<3;	    	
		}
		else if(gsMC_Drive->uw16IDCOffset_cntr == 364)
		{
			gsMC_Drive->w16Idc_offset = (int16_t)(gsMC_Drive->w32Idc_offsetTmp>>6);
		}
	}
	else
	{
	    gsMC_Drive->w16Idc = sub16(ADC_GetResult(ADC_IDC_CHANNEL)<<3,gsMC_Drive->w16Idc_offset);
	}	
    	
}


/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: ADC_GetResult
   *
   * @brief start a conversion and get conversion result
   *        
   * @param  none 
   *
   * @return none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
unsigned short ADC_GetResult( unsigned char channel)
{
	ADC_SC1 = channel; // initial the ADC conversion,select ADC channel
	while( !ADC_SC1_COCO );
	return ADC_R;
}
