/* led.c */
#include "led.h"
#include "fsl_gpio.h"

typedef struct
{
    GPIO_Type * base;
    uint32_t    pin;
} gpio_pin_id_t;

const gpio_pin_id_t led_gpio_pins[LED_COUNT] =
{
    { GPIOD, 0 },
    { GPIOD, 1 },
    { GPIOD, 3 },
    { GPIOD, 2 },
    { GPIOD, 4 },
    { GPIOE, 20 },
    { GPIOE, 21 },
    { GPIOE, 29 },
    { GPIOE, 30 },
    { GPIOE, 24 },
    { GPIOD, 5  },
};

void led_init(void)
{
    /* enable clocks. */
    CLOCK_EnableClock(kCLOCK_PortD);
    CLOCK_EnableClock(kCLOCK_PortE);

    PORTE->PCR[20] = PORT_PCR_MUX(1);
    PORTE->PCR[21] = PORT_PCR_MUX(1);
    PORTE->PCR[29] = PORT_PCR_MUX(1);
    PORTE->PCR[30] = PORT_PCR_MUX(1);
    PORTE->PCR[24] = PORT_PCR_MUX(1);
    PORTD->PCR[0] = PORT_PCR_MUX(1);
    PORTD->PCR[1] = PORT_PCR_MUX(1);
    PORTD->PCR[2] = PORT_PCR_MUX(1);
    PORTD->PCR[3] = PORT_PCR_MUX(1);
    PORTD->PCR[4] = PORT_PCR_MUX(1);
    PORTD->PCR[5] = PORT_PCR_MUX(1);

    gpio_pin_config_t gpio_pin_config;
    gpio_pin_config.pinDirection =kGPIO_DigitalOutput;
    gpio_pin_config.outputLogic = 1u;
    for (uint32_t i = 0u; i < LED_COUNT; i++)
    {
        GPIO_PinInit(led_gpio_pins[i].base, led_gpio_pins[i].pin, &gpio_pin_config);
    }
}

void led_on(uint32_t mask)
{
    for (uint32_t i = 0u; i < LED_COUNT; i++)
    {
        if (0u != ((1u << i) & mask) )
        {
            GPIO_WritePinOutput(led_gpio_pins[i].base, led_gpio_pins[i].pin, 0u);
        }
    }
}

void led_off(uint32_t mask)
{
    for (uint32_t i = 0u; i < LED_COUNT; i++)
    {
        if (0u != ((1u << i) & mask) )
        {
            GPIO_WritePinOutput(led_gpio_pins[i].base, led_gpio_pins[i].pin, 1u);
        }
    }
}

#define LED_BLINK_NUM 8

typedef struct
{
    uint8_t is_on;
    uint8_t remain_blink_num;
} led_blink_handler;

led_blink_handler led_blink_handlers[LED_BLINK_NUM];

static void led_blink_init_pit_hardware(void);

void led_blink_init(void)
{
    for (uint32_t i = 0u; i < LED_BLINK_NUM; i++)
    {
        led_blink_handlers[i].is_on = false;
        led_blink_handlers[i].remain_blink_num = 0;
    }

    led_blink_init_pit_hardware();
}

static void led_blink_init_pit_hardware(void)
{
    /* pit. */
    CLOCK_EnableClock(kCLOCK_Pit0);

    /* unlock the pit and setup the period. */
    PIT->MCR &= ~PIT_MCR_MDIS_MASK; /* unlock the pit from reset mode. */
    PIT->CHANNEL[0].LDVAL = CLOCK_GetFreq(kCLOCK_BusClk)/ 5u;

    /* enable interrupt for pit. */
    NVIC_EnableIRQ(PIT_IRQn);
    PIT->CHANNEL[0].TCTRL = PIT_TCTRL_TIE_MASK /* enable interrupt. */
                          | PIT_TCTRL_TEN_MASK /* enable pit timer. */
                          ;
}

void led_blink_set_blink(uint32_t led_index, uint8_t blink_num)
{
    led_blink_handlers[led_index].remain_blink_num = blink_num;
}

void led_blink_periodical_hook(void)
{
    for (uint32_t i = 0u; i < LED_BLINK_NUM; i++)
    {
        if (led_blink_handlers[i].is_on)
        {
            led_blink_handlers[i].is_on = false;
            led_off(1u << i);
            if (led_blink_handlers[i].remain_blink_num > 0U)
            {
                led_blink_handlers[i].remain_blink_num --;
            }
        }
        else
        {
            if (led_blink_handlers[i].remain_blink_num > 0u)
            {
                led_blink_handlers[i].is_on = true;
                led_on(1u << i);
            }
        }
    }
}

/* isr for pit interrupt. */
void PIT_IRQHandler(void)
{
    if ( PIT->CHANNEL[0].TFLG & PIT_TFLG_TIF_MASK)
    {
        PIT->CHANNEL[0].TFLG = PIT_TFLG_TIF_MASK;
        led_blink_periodical_hook();
    }
}

/* EOF. */

