/* touch.c */

#include "touch.h"
#include "fsl_clock.h"
#include "fsl_port.h"

/* TSI转换器通道号 */
const uint8_t touch_channel_code[TOUCH_2D_ALL_COUNT] =
{
    7, 6,  0,  3,  2,    /* X. */
    8, 9, 10, 13, 14,    /* Y. */
    15                   /* Shield. */
};

volatile uint32_t touch_channel_index;
volatile int32_t  touch_channel_values[TOUCH_2D_ALL_COUNT]; /* 扩展了采样结果寄存器. */
volatile bool     touch_scan_queue_done = false;

/* 初始化引脚 */
void touch_init_pins(void)
{
    /* enable clocks. */
    CLOCK_EnableClock(kCLOCK_PortA);
    CLOCK_EnableClock(kCLOCK_PortB);
    CLOCK_EnableClock(kCLOCK_PortC);

    /* x. */
    PORT_SetPinMux(PORTA, 1u , kPORT_PinDisabledOrAnalog); /* R0. */
    PORT_SetPinMux(PORTA, 2u , kPORT_PinDisabledOrAnalog); /* R1. */
    PORT_SetPinMux(PORTB, 0u , kPORT_PinDisabledOrAnalog); /* R2. */
    PORT_SetPinMux(PORTB, 1u , kPORT_PinDisabledOrAnalog); /* R3. */
    PORT_SetPinMux(PORTB, 2u , kPORT_PinDisabledOrAnalog); /* R4. */

    /* y. */
    PORT_SetPinMux(PORTB, 3u , kPORT_PinDisabledOrAnalog); /* C0. */
    PORT_SetPinMux(PORTB, 16u, kPORT_PinDisabledOrAnalog); /* C1. */
    PORT_SetPinMux(PORTB, 17u, kPORT_PinDisabledOrAnalog); /* C2. */
    PORT_SetPinMux(PORTC,  0u, kPORT_PinDisabledOrAnalog); /* C3. */
    PORT_SetPinMux(PORTC,  1u, kPORT_PinDisabledOrAnalog); /* C4. */

    /* shield. */
    PORT_SetPinMux(PORTC, 2u , kPORT_PinDisabledOrAnalog); /* shield. */
}

/* 使用中断方式在后台扫描传感器队列. */
void touch_init(void)
{
    TSI_Type *base = TSI0;

    CLOCK_EnableClock(kCLOCK_Tsi0);

    base->GENCS = TSI_GENCS_ESOR(0)   /* enable end-of-scan interrupt. */
                | TSI_GENCS_MODE(0)   /* TSI analog modes setup and status bits. */
                | TSI_GENCS_REFCHRG(1)/* indicate the reference oscillator charge and discharge current value. */
                | TSI_GENCS_DVOLT(0)  /* 采样比较器的门限阈值. */
                | TSI_GENCS_EXTCHRG(1)/* indicate the electrode oscillator charge and discharge current value. */
                | TSI_GENCS_PS(3)     /* indicate the prescaler of the output of electrode oscillator. */
                | TSI_GENCS_NSCN(7)   /* indicate the scan number for each electrode. */
                | TSI_GENCS_TSIEN(1)  /* enable tsi converter. */
                | TSI_GENCS_TSIIEN(1) /* enable interrupts. */


                | TSI_GENCS_STPE(0)   /* disable tsi in low power modes. */
                | TSI_GENCS_STM(0)    /* software trigger. */
                ;
#if 0
    /* setup filter. */
    base->SINC  = TSI_SINC_CUTOFF(1)     /* 移除尾巴末尾n位. */
                | TSI_SINC_ORDER(0)      /* use 1 or 2 order SINC filter. */
                | TSI_SINC_DECIMATION(15) /* 为每个通道连续采样次数n倍 */
                ;
#endif

    /* disable the hardware threashold. */
    base->TSHD  = TSI_TSHD_THRESH(0xffff)
                | TSI_TSHD_THRESL(0)
                ;

    /* clear flags and enable interrupt. */
    base->GENCS |= TSI_GENCS_TSIIEN(1) /* enable interrupts. */
                |  TSI_GENCS_ESOR(1)   /* enable end-of-scan interrupt. */
                ;

    NVIC_EnableIRQ(TSI0_IRQn);
}

void touch_do_soft_trigger(uint32_t channel_index)
{
    TSI_Type *base = TSI0;

    base->DATA  = TSI_DATA_SWTS_MASK
                | TSI_DATA_TSICH(channel_index)
                ;
}

uint32_t touch_get_conv_value(void)
{
    TSI_Type *base = TSI0;
    return (TSI_DATA_TSICNT_MASK & base->DATA);
}

void touch_start(void)
{
    touch_scan_queue_done = false;
    touch_channel_index = 0u;
    touch_do_soft_trigger(touch_channel_code[touch_channel_index]);
}

/* ISR for TSI interrupt. */
void TSI0_IRQHandler(void)
{
    TSI_Type *base = TSI0;
    uint32_t flags = base->GENCS;
#if 1
    for (uint32_t i = 0u; i < 10u; i++)
    {
        asm("nop");
    }
#endif
    base->GENCS = flags; /* clear flags */

    if (flags & TSI_GENCS_EOSF_MASK)
    {
        if (touch_channel_index < (TOUCH_2D_ALL_COUNT))
        {
            touch_channel_values[touch_channel_index] = touch_get_conv_value();
            touch_channel_index++;
            touch_do_soft_trigger(touch_channel_code[touch_channel_index]);
        }
        else
        {
            touch_scan_queue_done = true;
        }
    }
}

void touch_wait_data_ready(int32_t *outputs)
{
    while (!touch_scan_queue_done)
    {}
    touch_scan_queue_done = false;

    for (uint32_t i = 0u; i < TOUCH_2D_ALL_COUNT; i++)
    {
        outputs[i] = touch_channel_values[i];
    }
}

/* lptmr to trigger the scan round periodically. */
void lptmr_init(void)
{
    /* enable clock. */
    //CLOCK_SetIpSrc(kCLOCK_Lptmr0, kCLOCK_IpSrcFircAsync); /* 实际使用的是FircDiv2, 24MHz. */
    CLOCK_EnableClock(kCLOCK_Lptmr0);

    /* setup controller. */
    LPTMR0->CSR = LPTMR_CSR_TCF_MASK; /* disable timer and clear flags. */
    LPTMR0->PSR = LPTMR_PSR_PRESCALE(0x2)
                | LPTMR_PSR_PBYP_MASK /* 不分频得到计数单元. */
                | LPTMR_PSR_PCS(1); /* PCS = 1, 使用LPO 1khz时钟. */
    LPTMR0->CMR = LPTMR_CMR_COMPARE(1000 / 25); /* 50Hz中断频率. */
    LPTMR0->CSR = LPTMR_CSR_TIE(1) /* enable interrupt. */
                | LPTMR_CSR_TFC(0) /* CNR is reset whenever TCF is set. */
                | LPTMR_CSR_TMS(0) /* Time Counter mode. */
                ;

    /* enable nvic. */
    NVIC_EnableIRQ(LPTMR0_IRQn);
}

void lptmr_start(void)
{
    LPTMR0->CSR |= LPTMR_CSR_TEN_MASK;
}

/* ISR for LPTMR0. */
void LPTMR0_IRQHandler(void)
{
    uint32_t flags = LPTMR0->CSR;

    LPTMR0->CSR = flags; /* clear flags. */

    touch_start(); /* start a round of scan. */
}

/* EOF. */

