/*
 * Copyright (c) 2007-2015 Freescale Semiconductor, Inc.
 * Copyright 2018-2019 NXP
 *
 * License: NXP LA_OPT_NXP_Software_License
 *
 * NXP Confidential. This software is owned or controlled by NXP and may
 * only be used strictly in accordance with the applicable license terms.
 * By expressly accepting such terms or by downloading, installing,
 * activating and/or otherwise using the software, you are agreeing that
 * you have read, and that you agree to comply with and are bound by,
 * such license terms.  If you do not agree to be bound by the applicable
 * license terms, then you may not retain, install, activate or otherwise
 * use the software.  This code may only be used in a microprocessor,
 * microcontroller, sensor or digital signal processor ("NXP Product")
 * supplied directly or indirectly from NXP.  See the full NXP Software
 * License Agreement in license/LA_OPT_NXP_Software_License.pdf
 *
 * FreeMASTER Communication Driver - Serial Communication Interface
 */
#include "freemaster.h"
#include "freemaster_private.h"

/* Numeric identifier to help pre-processor to identify whether our driver is used or not. */
#define FMSTR_SERIAL_MPC577xC_SCI_ID 1

#if (FMSTR_MK_IDSTR(FMSTR_SERIAL_DRV) == FMSTR_SERIAL_MPC577xC_SCI_ID)
#if !(FMSTR_DISABLE) 

#include "freemaster_serial.h"
#include "freemaster_mpc5777c_sci.h"

/***********************************
*  local variables
***********************************/

/* SCI base address */
#ifdef FMSTR_SCI_BASE
    static FMSTR_ADDR fmstr_sciBaseAddr = (FMSTR_ADDR)FMSTR_SCI_BASE;
#else
    static FMSTR_ADDR fmstr_sciBaseAddr = (FMSTR_ADDR)0;
#endif

/***********************************
*  local function prototypes
***********************************/

/* Interface function - Initialization of SCI driver adapter */
static FMSTR_BOOL _FMSTR_MPC577xC_Sci_Init(void);
static void _FMSTR_MPC577xC_Sci_EnableTransmit(FMSTR_BOOL enable);
static void _FMSTR_MPC577xC_Sci_EnableReceive(FMSTR_BOOL enable);
static void _FMSTR_MPC577xC_Sci_EnableTransmitInterrupt(FMSTR_BOOL enable);
static void _FMSTR_MPC577xC_Sci_EnableTransmitCompleteInterrupt(FMSTR_BOOL enable);
static void _FMSTR_MPC577xC_Sci_EnableReceiveInterrupt(FMSTR_BOOL enable);
static FMSTR_BOOL _FMSTR_MPC577xC_Sci_IsTransmitRegEmpty(void);
static FMSTR_BOOL _FMSTR_MPC577xC_Sci_IsReceiveRegFull(void);
static FMSTR_BOOL _FMSTR_MPC577xC_Sci_IsTransmitterActive(void);
static void _FMSTR_MPC577xC_Sci_PutChar(FMSTR_BCHR  ch);
static FMSTR_BCHR _FMSTR_MPC577xC_Sci_GetChar(void);
static void _FMSTR_MPC577xC_Sci_Flush(void);

/***********************************
*  global variables
***********************************/
/* Interface of this SCI driver */ 

const FMSTR_SERIAL_DRV_INTF FMSTR_SERIAL_MPC577xC_SCI =
{
    .Init                       = _FMSTR_MPC577xC_Sci_Init,
    .EnableTransmit             = _FMSTR_MPC577xC_Sci_EnableTransmit,
    .EnableReceive              = _FMSTR_MPC577xC_Sci_EnableReceive,
    .EnableTransmitInterrupt    = _FMSTR_MPC577xC_Sci_EnableTransmitInterrupt,
    .EnableTransmitCompleteInterrupt= _FMSTR_MPC577xC_Sci_EnableTransmitCompleteInterrupt,
    .EnableReceiveInterrupt     = _FMSTR_MPC577xC_Sci_EnableReceiveInterrupt,
    .IsTransmitRegEmpty         = _FMSTR_MPC577xC_Sci_IsTransmitRegEmpty,
    .IsReceiveRegFull           = _FMSTR_MPC577xC_Sci_IsReceiveRegFull,
    .IsTransmitterActive        = _FMSTR_MPC577xC_Sci_IsTransmitterActive,
    .PutChar                    = _FMSTR_MPC577xC_Sci_PutChar,
    .GetChar                    = _FMSTR_MPC577xC_Sci_GetChar,
    .Flush                      = _FMSTR_MPC577xC_Sci_Flush,
};

/****************************************************************************************
* General peripheral space access macros
*****************************************************************************************/

#define FMSTR_SETBIT(base, offset, bit)         (*(((volatile FMSTR_U16*)(base))+(offset/2)) |= (bit))
#define FMSTR_CLRBIT(base, offset, bit)         (*(((volatile FMSTR_U16*)(base))+(offset/2)) &= ~(bit))
#define FMSTR_TSTBIT(base, offset, bit)         (*(((volatile FMSTR_U16*)(base))+(offset/2)) & (bit))
#define FMSTR_SETREG(base, offset, value)       (*(((volatile FMSTR_U16*)(base))+(offset/2)) = (value))
#define FMSTR_GETREG(base, offset)              (*(((volatile FMSTR_U16*)(base))+(offset/2)))

/****************************************************************************************
* SCI module constants
*****************************************************************************************/

/* SCI module registers */
#define FMSTR_SCIBRR_OFFSET     0
#define FMSTR_SCISDR_OFFSET     6
#define FMSTR_SCICR1_OFFSET     2
//#define FMSTR_SCICR2_OFFSET     4
#define FMSTR_SCIIFSR1_OFFSET     8
#define FMSTR_SCIIFSR2_OFFSET     0xA


/* SCI Control Register bits */
#define FMSTR_SCICR1_LOOPS     0x8000
//#define FMSTR_SCICR1_SWAI      0x40
#define FMSTR_SCICR1_RSRC      0x2000
#define FMSTR_SCICR1_M         0x1000
#define FMSTR_SCICR1_WAKE      0x0800
//#define FMSTR_SCICR1_ILT       0x04
#define FMSTR_SCICR1_PE        0x0200
#define FMSTR_SCICR1_PT        0x0100
#define FMSTR_SCICR1_TIE       0x0080
#define FMSTR_SCICR1_TCIE      0x0040
#define FMSTR_SCICR1_RIE       0x0020
#define FMSTR_SCICR1_ILIE      0x0010
#define FMSTR_SCICR1_TE        0x0008
#define FMSTR_SCICR1_RE        0x0004
#define FMSTR_SCICR1_RWU       0x0002
#define FMSTR_SCICR1_SBK       0x0001

/* SCI Status registers bits */
#define FMSTR_SCIIFSR_TDRE       0x8000
#define FMSTR_SCIIFSR_TC         0x4000
#define FMSTR_SCIIFSR_RDRF       0x2000
#define FMSTR_SCIIFSR_IDLE       0x1000
#define FMSTR_SCIIFSR_OR         0x0800
#define FMSTR_SCIIFSR_NF         0x0400
#define FMSTR_SCIIFSR_FE         0x0200
#define FMSTR_SCIIFSR_PF         0x0100
//#define FMSTR_SCIIFSR2_BRK13     0x04
//#define FMSTR_SCIIFSR2_TXDIR     0x02
//#define FMSTR_SCIIFSR2_RAF       0x01

/**************************************************************************//*!
*
* @brief    SCI communication initialization
*
******************************************************************************/

static FMSTR_BOOL _FMSTR_MPC577xC_Sci_Init(void)
{
#if FMSTR_SERIAL_SINGLEWIRE
    #error Internal single wire mode is not supported.
    return FMSTR_FALSE;
#endif
    
    /* valid runtime module address must be assigned */
    if(fmstr_sciBaseAddr != 0)
        return FMSTR_TRUE;
    else 
        return FMSTR_FALSE;
}

/**************************************************************************//*!
*
* @brief    Enable/Disable SCI transmitter
*
******************************************************************************/

static void _FMSTR_MPC577xC_Sci_EnableTransmit(FMSTR_BOOL enable)
{
    if(enable)
    {
        /* Enable transmitter */
        FMSTR_SETBIT(fmstr_sciBaseAddr, FMSTR_SCICR1_OFFSET, FMSTR_SCICR1_TE);
    }
    else
    {
        /* Disable transmitter */
        FMSTR_CLRBIT(fmstr_sciBaseAddr, FMSTR_SCICR1_OFFSET, FMSTR_SCICR1_TE);
    }
}

/**************************************************************************//*!
*
* @brief    Enable/Disable SCI receiver
*
******************************************************************************/

static void _FMSTR_MPC577xC_Sci_EnableReceive(FMSTR_BOOL enable)
{
    if(enable)
    {
        /* Enable receiver (enables single-wire connection) */
        FMSTR_SETBIT(fmstr_sciBaseAddr, FMSTR_SCICR1_OFFSET, FMSTR_SCICR1_RE);
    } 
    else
    {
        /* Disable receiver */
        FMSTR_CLRBIT(fmstr_sciBaseAddr, FMSTR_SCICR1_OFFSET, FMSTR_SCICR1_RE);
    }
}

/**************************************************************************//*!
*
* @brief    Enable/Disable interrupt from transmit register empty event
*
******************************************************************************/

static void _FMSTR_MPC577xC_Sci_EnableTransmitInterrupt(FMSTR_BOOL enable)
{
    if(enable)
    {
        /* Enable interrupt */
        FMSTR_SETBIT(fmstr_sciBaseAddr, FMSTR_SCICR1_OFFSET, FMSTR_SCICR1_TIE);
    }
    else
    {
        /* Disable interrupt */
        FMSTR_CLRBIT(fmstr_sciBaseAddr, FMSTR_SCICR1_OFFSET, FMSTR_SCICR1_TIE);
    }
}

/**************************************************************************//*!
*
* @brief    Enable/Disable interrupt when transmission is complete
*
******************************************************************************/

static void _FMSTR_MPC577xC_Sci_EnableTransmitCompleteInterrupt(FMSTR_BOOL enable)
{
    if(enable)
    {
        /* Enable interrupt */
        FMSTR_SETBIT(fmstr_sciBaseAddr, FMSTR_SCICR1_OFFSET, FMSTR_SCICR1_TCIE);
    }
    else
    {
        /* Disable interrupt */
        FMSTR_CLRBIT(fmstr_sciBaseAddr, FMSTR_SCICR1_OFFSET, FMSTR_SCICR1_TCIE);
    }
}

/**************************************************************************//*!
*
* @brief    Enable/Disable interrupt from receive register full event
*
******************************************************************************/

static void _FMSTR_MPC577xC_Sci_EnableReceiveInterrupt(FMSTR_BOOL enable)
{
    if(enable)
    {
        /* Enable interrupt */
        FMSTR_SETBIT(fmstr_sciBaseAddr, FMSTR_SCICR1_OFFSET, FMSTR_SCICR1_RIE);
    }
    else
    {
        /* Disable interrupt */
        FMSTR_CLRBIT(fmstr_sciBaseAddr, FMSTR_SCICR1_OFFSET, FMSTR_SCICR1_RIE);
    }
}

/**************************************************************************//*!
*
* @brief    Returns TRUE if the transmit register is empty, and it's possible to put next char
*
******************************************************************************/

static FMSTR_BOOL _FMSTR_MPC577xC_Sci_IsTransmitRegEmpty(void)
{
    return (FMSTR_BOOL) FMSTR_TSTBIT(fmstr_sciBaseAddr, FMSTR_SCIIFSR1_OFFSET, FMSTR_SCIIFSR_TDRE);
}

/**************************************************************************//*!
*
* @brief    Returns TRUE if the receive register is full, and it's possible to get received char
*
******************************************************************************/

static FMSTR_BOOL _FMSTR_MPC577xC_Sci_IsReceiveRegFull(void)
{
    return (FMSTR_BOOL) FMSTR_TSTBIT(fmstr_sciBaseAddr, FMSTR_SCIIFSR1_OFFSET, FMSTR_SCIIFSR_RDRF);
}

/**************************************************************************//*!
*
* @brief    Returns TRUE if the transmitter is still active 
*
******************************************************************************/

static FMSTR_BOOL _FMSTR_MPC577xC_Sci_IsTransmitterActive(void)
{
    /* 0 - Transmission in progress, 1 - No transmission in progress */
    return (!(FMSTR_TSTBIT(fmstr_sciBaseAddr, FMSTR_SCIIFSR1_OFFSET, FMSTR_SCIIFSR_TC)));
}

/**************************************************************************//*!
*
* @brief    The function puts the char for transmit
*
******************************************************************************/

static void _FMSTR_MPC577xC_Sci_PutChar(FMSTR_BCHR  ch)
{
    FMSTR_SETREG(fmstr_sciBaseAddr, FMSTR_SCISDR_OFFSET, ch);
    FMSTR_SETBIT(fmstr_sciBaseAddr, FMSTR_SCIIFSR1_OFFSET, FMSTR_SCIIFSR_TC);
}


/**************************************************************************//*!
*
* @brief    The function gets the received char
*
******************************************************************************/
static FMSTR_BCHR _FMSTR_MPC577xC_Sci_GetChar(void)
{
	FMSTR_BCHR c=0;
	c = FMSTR_GETREG(fmstr_sciBaseAddr, FMSTR_SCISDR_OFFSET);
	FMSTR_SETBIT(fmstr_sciBaseAddr, FMSTR_SCIIFSR1_OFFSET, FMSTR_SCIIFSR_RDRF);
    return c;
}

/**************************************************************************//*!
*
* @brief    The function sends buffered data
*
******************************************************************************/

static void _FMSTR_MPC577xC_Sci_Flush(void)
{
}

/**************************************************************************//*!
*
* @brief    Assign FreeMASTER communication module base address
*
******************************************************************************/

void FMSTR_SerialSetBaseAddress(FMSTR_ADDR base)
{
    fmstr_sciBaseAddr = base;
}

/**************************************************************************//*!
*
* @brief    Process FreeMASTER serial interrupt (call this function from SCI ISR)
*
******************************************************************************/

void FMSTR_SerialIsr()
{
    /* process incomming or just transmitted byte */
    #if (FMSTR_LONG_INTR) || (FMSTR_SHORT_INTR)
        FMSTR_ProcessSerial();
    #endif
}

#else /* !(FMSTR_DISABLE) */

/* Empty API functions when FMSTR_DISABLE is set */
void FMSTR_SerialSetBaseAddress(FMSTR_ADDR base)
{
    FMSTR_UNUSED(base);
}

void FMSTR_SerialIsr()
{
}

#endif /* !(FMSTR_DISABLE) */ 
#endif /* (FMSTR_MK_IDSTR(FMSTR_SERIAL_DRV) == FMSTR_SERIAL_S12Z_SCI_ID) */
