#include "fsl_usart.h"
#include "fsl_common.h"
#include "spi_to_dual_usart_bridge.h"

/*******************************************************************************
 * Variables
 ******************************************************************************/
//save data read from USART temporarily 
static uint8_t tempUSARTData = 0;
//USART channel number
static int8_t  usartChannelNumber = -1;

/*******************************************************************************
 * Code
 ******************************************************************************/
/*!
 * brief  get USART channel number
 * param  void
 * return int8_t->USART channel number
 */
int8_t getUSARTChannelNum(void)
{
	return usartChannelNumber;
}

/*!
 * brief  USART0 interrupt service routine
 * param  void
 * return void
 */
void USART0_IRQHandler(void)
{
	if(USART_GetStatusFlags(USART0) & kUSART_RxReady)
	{//USART0 has received data
		//read data from USART0 and clear status register
		tempUSARTData = USART_ReadByte(USART0);
		//set USART channel number to 0
		usartChannelNumber = 0;
		//save data to rx fifo in bridge
		saveUSARTDataToRxFifo(tempUSARTData);
	}
}
 
/*!
 * brief  USART1 interrupt service routine
 * param  void
 * return void
 */
void USART1_IRQHandler(void)
{
	if(USART_GetStatusFlags(USART1) & kUSART_RxReady)
	{//USART1 has received data
		//read data from USART1 and clear status register
		tempUSARTData = USART_ReadByte(USART1);
		//set USART channel number to 1
		usartChannelNumber = 1;
		//save data to rx fifo in bridge
		saveUSARTDataToRxFifo(tempUSARTData);
	}
}
 
/*!
 * brief  USART initialize
 * param  void
 * return void
 */
void BOARD_UsartInit(void)
{
	//usart configuration structure
	usart_config_t config;
	//main clock frequency
	uint32_t mainClockFreq;
	
	//get the default configuration of usart
	USART_GetDefaultConfig(&config);
	//enable usart receive
  config.enableRx     = true;
	//enable usart send
  config.enableTx     = true;
	//Set baud rate as 115200
  config.baudRate_Bps = 115200;
  
	//get main clock frequency
	mainClockFreq = CLOCK_GetFreq(kCLOCK_MainClk);
  //initialize the USART0 with configuration
	USART_Init(USART0, &config, mainClockFreq);
	//initialize the USART1 with configuration
  USART_Init(USART1, &config, mainClockFreq);
	
	//enable USART0 interrupt in NVIC
	EnableIRQ(USART0_IRQn);
	//enable USART0 interrupt in USART
	USART_EnableInterrupts(USART0, kUSART_RxReadyInterruptEnable);
	//enable USART1 interrupt in NVIC
	EnableIRQ(USART1_IRQn);
	//enable USART1 interrupt in USART
	USART_EnableInterrupts(USART1, kUSART_RxReadyInterruptEnable);
}

/*!
 * brief  USART send data saved in buffer
 * param  USART_Type *base->USART peripheral base address
 * param  uint8_t* p_dataBuf->pointer to buffer for data to be sent
 * param  uint32_t dataLengh->data length to be sent
 * return void
 */
void BOARD_UsartSendData(USART_Type *base, uint8_t* p_dataBuf, uint32_t dataLengh)
{
	if((base != NULL) && (p_dataBuf != NULL) && (dataLengh != 0))
	{//UART peripheral and data buffer are available and data length is more than 0
		//send data through usart
		USART_WriteBlocking(base,p_dataBuf,dataLengh);
	}
}
