/*
 * The Clear BSD License
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer below) provided
 * that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of the copyright holder nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * NO EXPRESS OR IMPLIED LICENSES TO ANY PARTY'S PATENT RIGHTS ARE GRANTED BY THIS LICENSE.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "board.h"
#include "fsl_debug_console.h"
#include "fsl_spifi_dma.h"

#include "pin_mux.h"
#include "spiflash.h"
#include <stdbool.h>
/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define EXAMPLE_SPIFI SPIFI0 //Address 0x10000000 - 0x11000000

#define SPI_FLASH_BAUDRATE (96000000)
#define SPI_FLASH_DMA DMA0
#define SPIFI_CHANNEL 18

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
static dma_handle_t s_DmaHandle;
static spifi_dma_handle_t handle;
static volatile bool finished = false;
static SPI_FLASH_MODE spi_flash_curmode;


static spifi_command_t spifi_command[COMMAND_NUM] = {
    {PAGE_SIZE, false, kSPIFI_DataInput, 1, kSPIFI_CommandDataQuad, kSPIFI_CommandOpcodeAddrThreeBytes, 0x6B},
//    {PAGE_SIZE, false, kSPIFI_DataOutput, 0, kSPIFI_CommandDataQuad, kSPIFI_CommandOpcodeAddrThreeBytes, 0x32},
    {PAGE_SIZE, false, kSPIFI_DataOutput, 0, kSPIFI_CommandOpcodeSerial, kSPIFI_CommandOpcodeAddrThreeBytes, 0x38},
    {4, false, kSPIFI_DataInput, 0, kSPIFI_CommandAllSerial, kSPIFI_CommandOpcodeOnly, 0x05}, //Get stauitus
    {0, false, kSPIFI_DataOutput, 0, kSPIFI_CommandAllSerial, kSPIFI_CommandOpcodeAddrThreeBytes, 0x20}, //4KB sector erase
    {0, false, kSPIFI_DataOutput, 0, kSPIFI_CommandAllSerial, kSPIFI_CommandOpcodeAddrThreeBytes, 0x52}, //32KB sector erase
    {0, false, kSPIFI_DataOutput, 0, kSPIFI_CommandAllSerial, kSPIFI_CommandOpcodeAddrThreeBytes, 0xD8}, //64KB sector erase
    {0, false, kSPIFI_DataOutput, 0, kSPIFI_CommandAllSerial, kSPIFI_CommandOpcodeOnly, 0x06}, //Write Enable
    {4, false, kSPIFI_DataOutput, 0, kSPIFI_CommandAllSerial, kSPIFI_CommandOpcodeOnly, 0x01}
	}; //Write register


/*******************************************************************************
 * Code
 ******************************************************************************/
static void callback(SPIFI_Type *base, spifi_dma_handle_t *handle, status_t status, void *userData)
{
    finished = true;
}

static void check_if_finish()
{
    uint32_t val = 0;
    /* Check WIP bit */
    do
    {
        SPIFI_SetCommand(EXAMPLE_SPIFI, &spifi_command[GET_STATUS]);
        while ((EXAMPLE_SPIFI->STAT & SPIFI_STAT_INTRQ_MASK) == 0U)
        {
        }
        val = SPIFI_ReadData(EXAMPLE_SPIFI);
    } while (val & 0x1);
}

#if defined(QUAD_MODE_VAL)
void spi_flash_enable_quad_mode()
{
    /* Write enable */
    SPIFI_SetCommand(EXAMPLE_SPIFI, &spifi_command[WRITE_ENABLE]);

    /* Set write register command */
    SPIFI_SetCommand(EXAMPLE_SPIFI, &spifi_command[WRITE_REGISTER]);

    SPIFI_WriteData(EXAMPLE_SPIFI, QUAD_MODE_VAL);

    check_if_finish();
}
#endif
/*
    Init SPI FLASH using DMA
    spiclksrc: SPIFI Clock source
*/
int8_t spi_flash_init(clock_attach_id_t spiclksrc)
{
    spifi_config_t config = {0};
    uint32_t sourceClockFreq;

    DMA_Init(SPI_FLASH_DMA);

    DMA_EnableChannel(SPI_FLASH_DMA, SPIFI_CHANNEL);
    DMA_CreateHandle(&s_DmaHandle, SPI_FLASH_DMA, SPIFI_CHANNEL);

    /* Set SPIFI clock source */
    CLOCK_AttachClk(spiclksrc);
    sourceClockFreq = CLOCK_GetSpifiClkFreq();

    /* Set the clock divider */
    CLOCK_SetClkDiv(kCLOCK_DivSpifiClk, sourceClockFreq / SPI_FLASH_BAUDRATE, false);

    /* Initialize SPIFI */
    SPIFI_GetDefaultConfig(&config);
    SPIFI_Init(EXAMPLE_SPIFI, &config);
    SPIFI_TransferRxCreateHandleDMA(EXAMPLE_SPIFI, &handle, callback, NULL, &s_DmaHandle);

	spi_flash_curmode = SPI_FLASH_MODE_UNKNOWN;

    return 1;
}

int8_t spi_flash_setmode(SPI_FLASH_MODE mode)
{
	if (spi_flash_curmode == SPI_FLASH_MODE_UNKNOWN) {
		/* Setup memory command */
		SPIFI_SetMemoryCommand(EXAMPLE_SPIFI, &spifi_command[READ]);
		spi_flash_curmode = SPI_FLASH_MODE_MEMORY;
	}

    if (spi_flash_curmode == mode) {
        return 1;
	} else {
        /* Reset the SPIFI to switch to command mode */
        SPIFI_ResetCommand(EXAMPLE_SPIFI);
        /* From SPI_FLASH_MODE_MEMORY to SPI_FLASH_MODE_COMMAND */
		if (mode == SPI_FLASH_MODE_COMMAND) {
			 /* Write enable */
			SPIFI_SetCommand(EXAMPLE_SPIFI, &spifi_command[WRITE_ENABLE]);
		} else {
            SPIFI_SetMemoryCommand(EXAMPLE_SPIFI, &spifi_command[READ]);
		}
		spi_flash_curmode = mode;
	}
	return 1;
}

int8_t spi_flash_erase_sector(uint32_t sectornum, SPI_FLASH_ERASE_SECTOR_OPTION option)
{

	SPIFI_SetCommand(EXAMPLE_SPIFI, &spifi_command[WRITE_ENABLE]);//added by nn
    /* Set address */
	switch(option) {
    case SPI_FLASH_ERASE_SECTOR_4KB:
        /* Erase sector */		
		SPIFI_SetCommandAddress(EXAMPLE_SPIFI, sectornum * SECTOR_4KB_SIZE);
        SPIFI_SetCommand(EXAMPLE_SPIFI, &spifi_command[ERASE_4KB_SECTOR]);
		break;
    case SPI_FLASH_ERASE_SECTOR_32KB:		
		SPIFI_SetCommandAddress(EXAMPLE_SPIFI, sectornum * SECTOR_32KB_SIZE);
        SPIFI_SetCommand(EXAMPLE_SPIFI, &spifi_command[ERASE_32KB_SECTOR]);
		break;
    case SPI_FLASH_ERASE_SECTOR_64KB:		
		SPIFI_SetCommandAddress(EXAMPLE_SPIFI, sectornum * SECTOR_SIZE);
        SPIFI_SetCommand(EXAMPLE_SPIFI, &spifi_command[ERASE_64KB_SECTOR]);
		break;
	default:
		return 0;
    }
    /* Check if finished */
    check_if_finish();

    return 1;
}

int8_t spi_flash_program_page(uint32_t pagenum, uint8_t *pbuffer)
{
    spifi_transfer_t xfer = {0};

    SPIFI_SetCommand(EXAMPLE_SPIFI, &spifi_command[WRITE_ENABLE]);
    SPIFI_SetCommandAddress(EXAMPLE_SPIFI, pagenum * PAGE_SIZE);
    SPIFI_SetCommand(EXAMPLE_SPIFI, &spifi_command[PROGRAM_PAGE]);
    xfer.data = pbuffer;
    xfer.dataSize = PAGE_SIZE;
    SPIFI_TransferSendDMA(EXAMPLE_SPIFI, &handle, &xfer);
    finished = false;
    while (!finished)
    {
    }

    check_if_finish();

    return 1;
}
void spifi_mem_read(uint32_t addr, uint8_t *data, uint32_t size)
{
	uint32_t i;
	uint8_t *val;

    /* Reset to memory command mode */
    SPIFI_ResetCommand(EXAMPLE_SPIFI);

    SPIFI_SetCommand(EXAMPLE_SPIFI, &spifi_command[READ]);

    for (i = 0; i < size; i++)
    {
    	val = (uint8_t *)(addr + i);
    	*(data + i) = *val;
    }
}
