/*******************************************************************************
*
* E M B E D D E D   W I Z A R D   P R O J E C T
*
*                                                Copyright (c) TARA Systems GmbH
*                                    written by Paul Banach and Manfred Schweyer
*
********************************************************************************
*
* This software is delivered "as is" and shows the usage of other software
* components. It is provided as an example software which is intended to be
* modified and extended according to particular requirements.
*
* TARA Systems hereby disclaims all warranties and conditions with regard to the
* software, including all implied warranties and conditions of merchantability
* and non-infringement of any third party IPR or other rights which may result
* from the use or the inability to use the software.
*
********************************************************************************
*
* DESCRIPTION:
*   This file is part of the interface (glue layer) between an Embedded Wizard
*   generated UI application and the board support package (BSP) of a dedicated
*   target.
*   This template provides access to some LEDs and buttons of the board.
*
*******************************************************************************/

#include "board.h"
#include "fsl_gpio.h"
#include "fsl_pint.h"
#include "fsl_inputmux.h"

#include "ewconfig.h"
#include "ew_bsp_inout.h"

static void setLed( int aIndex, int aOn );

static TButtonCallback        UserButtonCallback = NULL;

/*******************************************************************************
* FUNCTION:
*   pint_intr_callback
*
* DESCRIPTION:
*   The GPIO interrupt callback function.
*
* ARGUMENTS:
*   GPIO_Pin - the gpio pin.
*
* RETURN VALUE:
*   None
*
*******************************************************************************/
void pint_intr_callback( pint_pin_int_t pintr, uint32_t pmatch_status )
{
  int pinState = 1;

  if ( GPIO_PinRead( GPIO, BOARD_SW5_GPIO_PORT, BOARD_SW5_GPIO_PIN ))
    pinState = 0;

  if ( UserButtonCallback )
    UserButtonCallback( pinState );
}


/*******************************************************************************
* FUNCTION:
*   EwBspInOutInitButton
*
* DESCRIPTION:
*   Configures one hardware button of the board used for demo applications.
*
* ARGUMENTS:
*   aButtonCallback - The button callback.
*
* RETURN VALUE:
*   None
*
*******************************************************************************/
void EwBspInOutInitButton( TButtonCallback aButtonCallback )
{
  CLOCK_EnableClock( kCLOCK_Gpio1 );  // user button pin

  /* Connect trigger sources to PINT */
  INPUTMUX_Init( INPUTMUX );
  INPUTMUX_AttachSignal( INPUTMUX, kPINT_PinInt0, kINPUTMUX_GpioPort1Pin1ToPintsel );

  /* Turnoff clock to inputmux to save power. Clock is only needed to make changes */
  INPUTMUX_Deinit( INPUTMUX );

  /* Initialize PINT */
  PINT_Init( PINT );

  /* Setup Pin Interrupt 0 for rising edge */
  PINT_PinInterruptConfig( PINT, kPINT_PinInt0, kPINT_PinIntEnableBothEdges, pint_intr_callback );

  /* Enable callbacks for PINT */
  PINT_EnableCallback( PINT );

  UserButtonCallback = aButtonCallback;
}


/*******************************************************************************
* FUNCTION:
*   EwBspInOutInitLed
*
* DESCRIPTION:
*   Configures one LED of the board used for demo applications.
*
* ARGUMENTS:
*   None
*
* RETURN VALUE:
*   None
*
*******************************************************************************/
void EwBspInOutInitLed( void )
{
  /* Define the init structure for the output LED pin*/
  gpio_pin_config_t led_config =
  {
    kGPIO_DigitalOutput, 0,
  };

  CLOCK_EnableClock( kCLOCK_Gpio2 );  // LED2,3
  CLOCK_EnableClock( kCLOCK_Gpio3 );  // LED1

  /* Init output LED GPIO. */
  GPIO_PinInit( GPIO, BOARD_LED1_GPIO_PORT, BOARD_LED1_GPIO_PIN, &led_config );
  GPIO_PinInit( GPIO, BOARD_LED2_GPIO_PORT, BOARD_LED2_GPIO_PIN, &led_config );
  GPIO_PinInit( GPIO, BOARD_LED3_GPIO_PORT, BOARD_LED3_GPIO_PIN, &led_config );

  setLed( 1, 0 );
  setLed( 2, 0 );
  setLed( 3, 0 );
}


/*******************************************************************************
* FUNCTION:
*   EwBspInOutLedOn
*
* DESCRIPTION:
*   Switch LED on (used for demo applications).
*
* ARGUMENTS:
*   None
*
* RETURN VALUE:
*   None
*
*******************************************************************************/
void EwBspInOutLedOn( void )
{
  setLed( 1, 1 );
}


/*******************************************************************************
* FUNCTION:
*   EwBspInOutLedOff
*
* DESCRIPTION:
*   Switch LED off (used for demo applications).
*
* ARGUMENTS:
*   None
*
* RETURN VALUE:
*   None
*
*******************************************************************************/
void EwBspInOutLedOff( void )
{
  setLed( 1, 0 );
}


/*******************************************************************************
* FUNCTION:
*   setLed
*
* DESCRIPTION:
*   Switch the given LED on or off.
*
* ARGUMENTS:
*   None
*
* RETURN VALUE:
*   None
*
*******************************************************************************/
static void setLed( int aIndex, int aOn )
{
  uint32_t port;
  uint32_t pin;

  switch( aIndex )
  {
    case 1:
    default:
      port = BOARD_LED1_GPIO_PORT;
      pin  = BOARD_LED1_GPIO_PIN;
      break;

    case 2:
      port = BOARD_LED2_GPIO_PORT;
      pin  = BOARD_LED2_GPIO_PIN;
      break;

    case 3:
      port = BOARD_LED3_GPIO_PORT;
      pin  = BOARD_LED3_GPIO_PIN;
      break;
  }

  GPIO_PinWrite( GPIO, port, pin, aOn ? 0 : 1 );
}

/* mli */
