/*******************************************************************************
* NXP Semiconductors
* (c) Copyright 2016 NXP Semiconductors
* ALL RIGHTS RESERVED.
********************************************************************************
Services performed by NXP in this matter are performed AS IS and without any 
warranty. CUSTOMER retains the final decision relative to the total design
and functionality of the end product. NXP neither guarantees nor will be held
liable by CUSTOMER for the success of this project.
NXP DISCLAIMS ALL WARRANTIES, EXPRESSED, IMPLIED OR STATUTORY INCLUDING,
BUT NOT LIMITED TO, IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR
A PARTICULAR PURPOSE ON ANY HARDWARE, SOFTWARE ORE ADVISE SUPPLIED 
TO THE PROJECT BY NXP, AND OR NAY PRODUCT RESULTING FROM NXP SERVICES. 
IN NO EVENT SHALL NXP BE LIABLE FOR INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING
OUT OF THIS AGREEMENT. 
CUSTOMER agrees to hold NXP harmless against any and all claims demands 
or actions by anyone on account of any damage, or injury, whether commercial,
contractual, or tortuous, rising directly or indirectly as a result 
of the advise or assistance supplied CUSTOMER in connection with product, 
services or goods supplied under this Agreement.
********************************************************************************
* File:             main.c
* Owner:            David Tosenovjan
* Version:          0.2
* Date:             Feb-17-2021
* Classification:   General Business Information        
********************************************************************************
* Detailed Description:
* Purpose of the example is to show how to generate Multi-bit or Single-bit ECC
* error in internal FLASH (user must choose it in the option at the end of main
* function).
* ECC error is injected by reading of pre-defined patterns in UTEST area at
* addresses 0x00400040 and 0x00400060. 
* When corrupted data is accessed the IVOR1 exception handler is called in case
* of multi-bit ECC error (IVOR1 exception occurs) and FCCU_Alarm_Interrupt
* handler is called in case of single-bit ECC error (FCCU interrupt occurs).
* Both function calls MEMU handler.
* The example displays notices in the terminal window (connector J19 on
* MPC57xx_Motherboard)(19200-8-no parity-1 stop bit-no flow control on eSCI_A).
* No other external connection is required.
* ------------------------------------------------------------------------------
* Test HW:         MPC57xx_Motherboard + MPC5744P-144DC
* MCU:             PPC5744PFMLQ8,0N15P,QQAA1515N, Rev2.1B
* Fsys:            200 MHz PLL with 40 MHz crystal reference
* Debugger:        Lauterbach Trace32
* Target:          internal_FLASH, RAM
* Terminal:        19200-8-no parity-1 stop bit-no flow control
* EVB connection:  default
********************************************************************************
Revision History:
Ver  Date         Author            Description of Changes
0.0  Jan-09-2017  David Tosenovjan  Initial version
0.1  Dec-21-2017  David Tosenovjan  Minor edits
0.2  Feb-17-2021  David Tosenovjan  MEMU_handler, invalidate entry by w0c
*******************************************************************************/

#include <ppc_ghs.h>

#include "uart.h"
#include "stdio.h"
#include "MPC5744P.h"
#include "INTC_Init.h"
#include "MPC5744P_MEMU_macros.h"


/*******************************************************************************
* External objects
*******************************************************************************/

/*******************************************************************************
* Global variables
*******************************************************************************/
extern const uint32_t IntcIsrVectorTable[];
extern void xcptn_xmpl(void);


/*******************************************************************************
* Constants and macros
*******************************************************************************/
#define VLE_IS_ON 1

#define DRUN_MODE   0x3

/* FCCU keys */
#define FCCU_NCFK_KEY 0xAB3498FE

/* CTRLK register OP-keys for FCCU configuration */
#define KEY_OP1  0x913756AF
#define KEY_OP2  0x825A132B
#define KEY_OP16 0x7ACB32F0
#define KEY_OP31 0x29AF8752


/*******************************************************************************
* Local types
*******************************************************************************/

/*******************************************************************************
* Local function prototypes
*******************************************************************************/
void Sys_Init(void);
void FCCU_Init(void);
void ClearNCF(void);
void FCCU_Alarm_Interrupt(void);

void MEMU_handler(void);
void Machine_check_handler(void);
static void Increment_MCSRR0_to_next_instruction(void);
static void ClearMCSR(void);
static void ClearMSR_EE(void);
static void SetMSR_ME(void);

void Inject_2bit_FLASH_ECC_error_by_UTEST_read(void);
void Inject_1bit_FLASH_ECC_error_by_UTEST_read(void);


/*******************************************************************************
* Local variables
*******************************************************************************/
uint32_t NCF_status[4];

/* debugging variables */
vint32_t IVOR1_handler_pass_count = 0;
vint32_t noncritical_fault_count = 0;


/*******************************************************************************
* Local functions
*******************************************************************************/

/*******************************************************************************
Function Name : Sys_Init
Engineer      : Peter Vlna
Date          : Dec-19-2014
Parameters    : 
Modifies      : 
Returns       : 
Notes         : Clock settings 200MHz PLL
Issues        : 
*******************************************************************************/
void Sys_Init(void)
{
    //Clk Out
    MC_CGM.AC6_SC.R = 0x02000000;  //Enable PLL0 - clkout0 //MC_CGM_AC6_SC
    MC_CGM.AC6_DC0.B.DE = 1;       //Enable divider for SYSCLK0
    MC_CGM.AC6_DC0.B.DIV = 0;      //Set divider for SYSCLK0

    //Configure RunPeripheralConfiguration regist   ers in ME_RUN_PC0
    MC_ME.RUN_PC[0].B.DRUN = 1;    //Enable DRUN mode for all peripherals
    
    //Enable external oscilator
    MC_ME.DRUN_MC.B.XOSCON = 1;
    
    //AUX Clock Selector 3 setup
    MC_CGM.AC3_SC.B.SELCTL =0x01;  //connect (8..40MHz) XTALL to the PLL0 input
    MC_ME.DRUN_MC.B.PLL0ON = 1;    //Enable PLL0 for DRUN mode

    // Set PLL0 to 200MHz
    PLLDIG.PLL0CR.B.CLKCFG = 1;             //Bypass mode PLL0 on
    // RFDPHI1 = 10, RFDPHI = 2, PREDIV = 2, MFD = 14
    PLLDIG.PLL0DV.R = 0x50000000 |0x00020000 |0x00002000 |0x0014 ; 
    
    // Set PPL0 as system clock
    MC_ME.DRUN_MC.B.SYSCLK = 0x2;
    //  Enable system clock divider /4
    MC_CGM.SC_DC0.B.DIV = 0x3;
    
    //Mode transition to apply the PLL0 setup and set Normal mode with PLL
    MC_ME.MCTL.R = 0x30005AF0;      //DRUN Mode & Key 
    MC_ME.MCTL.R = 0x3000A50F;      //DRUN Mode & Key

    while(!MC_ME.GS.B.S_PLL0);      //ME_GS Wait for PLL stabilization.
    while(MC_ME.GS.B.S_MTRANS);     //Waiting for end of transaction
    // ME_GS Check DRUN mode has successfully been entered
    while(MC_ME.GS.B.S_CURRENT_MODE != DRUN_MODE);  
}


/*******************************************************************************
Function Name : FCCU_Init
Engineer      : David Tosenovjan
Date          : Mar-05-2016
Parameters    : 
Modifies      : 
Returns       : 
Notes         :                             
Issues        : 
*******************************************************************************/
void FCCU_Init(void)
{
    /* clear possible faults*/
    ClearNCF();

    /* Unlock configuration */ 
    FCCU.TRANS_LOCK.R = 0xBC; 
    /* provide Config state key */ 
    FCCU.CTRLK.R = 0x913756AF; //key for OP1 
    /* enter config state - OP1 */ 
    FCCU.CTRL.R = 0x1; //set OP1 - set up FCCU into the CONFIG mode 
    /* wait for successful state transition */ 
    while (FCCU.CTRL.B.OPS != 0x3); //operation status succesful

    /* Non-critical fault enable for all MEMU sources */
    FCCU.NCF_TOE[0].R = 0xFFFFFFFF; //ALARM Timeout Enable 
    FCCU.NCF_E[0].R = 0x00FF8000; // NCF[15]-NCF[23]
    FCCU.IRQ_ALARM_EN[0].R = 0x00FF8000;
    
    /* set up the NORMAL mode of FCCU */ 
    FCCU.CTRLK.R = 0x825A132B; //key for OP2 
    FCCU.CTRL.R = 0x2; //set the OP2 - set up FCCU into the NORMAL mode 
    while (FCCU.CTRL.B.OPS != 0x3); //operational status succesful 
}


/*******************************************************************************
Function Name : ClearNCF
Engineer      : David Tosenovjan
Date          : Mar-05-2016
Parameters    : 
Modifies      : 
Returns       : 
Notes         : clears all FCCU faults
Issues        : 
*******************************************************************************/
void ClearNCF(void)
{
    uint32_t i,b[4];
    for(i=0;i<4;i++)
    {
        FCCU.NCFK.R = FCCU_NCFK_KEY;
        FCCU.NCF_S[i].R = 0xFFFFFFFF;
        while(FCCU.CTRL.B.OPS != 0x3)
        {
        
        };              /* wait for the completion of the operation */
        b[i]=FCCU.NCF_S[i].R;
    }
}


/*******************************************************************************
Function Name : FCCU_Alarm_Interrupt
Engineer      : David Tosenovjan
Date          : Mar-05-2016
Parameters    : 
Modifies      : 
Returns       : 
Notes         : 
Issues        : for simplicity it clears all faults!
*******************************************************************************/
void FCCU_Alarm_Interrupt(void)
{
   uint32_t i;
   
   MEMU_handler();

   FCCU.CTRL.R = 10; /* OP10 - Read the NCF status registers */
   while(FCCU.CTRL.B.OPS != 0x3)
   {         
   };      /* wait for the completion of the operation */
    
   /* load these regs to the variable field */
   for(i=0;i<4;i++)
   {
       NCF_status[i] = FCCU.NCF_S[i].R;
   }
   
   noncritical_fault_count++;
   ClearNCF();    
}


/*******************************************************************************
Function Name : MEMU_handler
Engineer      : David Tosenovjan
Date          : Jan-09-2017
Parameters    : 
Modifies      : 
Returns       : 
Notes         : 
*******************************************************************************/
void MEMU_handler(void)
{
    uint32_t MEMU_ERR_FLAG_value = 0;
    uint32_t i;
    
    /* examine source of ECC error */
    /* for the purpose of this example only FLASH errors are being examined */

    MEMU_ERR_FLAG_value = MEMU.ERR_FLAG.R;
    
    if (MEMU_ERR_FLAG_value & MEMU_ERR_FLAG_FLASH_all)    
    {
        printf("FLASH ECC error found\r\n");
        if (MEMU_ERR_FLAG_value & MEMU_ERR_FLAG_F_UCE)    
        {
            if (1 == MEMU.FLASH_UNCERR_STS.B.VLD)
            {
                printf("It is non-correctable error at address ");
                printf("0x%08X\r\n",MEMU.FLASH_UNCERR_ADDR.R);

                /* remove the reson of ECC error (application specific) */
                printf("Application specific measures should be taken\r\n");

                // invalidate the entry (w0c)
                MEMU.FLASH_UNCERR_STS.R = 0x00000000;		
            }    
        }

        if (MEMU_ERR_FLAG_value & MEMU_ERR_FLAG_F_CE)    
        {
            for(i=0;i<20;i++)
            { 
                if (1 == MEMU.FLASH_CERR[i].STS.B.VLD)
                {
                    printf("It is correctable error at address ");
                    printf("0x%08X\r\n",MEMU.FLASH_CERR[i].ADDR.R);

                    /* there is no need to correct single bit errors */
                    printf("Single bit errors ignored\r\n");

		    // invalidate the entry (w0c)
                    MEMU.FLASH_CERR[i].STS.R = 0x00000000;
                }
            }
            
        }

        if (MEMU_ERR_FLAG_value & MEMU_ERR_FLAG_FLASH_overflows)    
        {
            printf("Overflow!");
            while(1)
            {
                asm("nop");
            }
        }
      
    }

    /* check other possible sources */
    else if (MEMU_ERR_FLAG_value & MEMU_ERR_FLAG_PERRAM_all)    
    {
        printf("Peripheral RAM ECC error found\r\n");
    }

    else if (MEMU_ERR_FLAG_value & MEMU_ERR_FLAG_SYSRAM_all)    
    {
        printf("System RAM ECC error found\r\n");
    }
    
    /* clear all MEMU error flag flags to stop FCCU fault indication */
    MEMU.ERR_FLAG.R = 0xFFFFFFFF;
}


/*******************************************************************************
Function Name : Machine_check_handler
Engineer      : David Tosenovjan
Date          : Mar-05-2016
Parameters    : 
Modifies      : 
Returns       : 
Notes         : Function handles Machine_check_handler
*******************************************************************************/
//__interrupt
void Machine_check_handler(void)
{
    
    MEMU_handler(); 
    
    Increment_MCSRR0_to_next_instruction();
    
    ClearMCSR();
    
    IVOR1_handler_pass_count++;
    
}


/*******************************************************************************
Function Name : Increment_MCSRR0_to_next_instruction
Engineer      : David Tosenovjan
Date          : Sep-16-2015
Parameters    : 
Modifies      : 
Returns       : 
Notes         : - based on algorithm described in AN4648
                - applicable for IVOR1 (stores address to MCSRR0)                              
Issues        : 
*******************************************************************************/
#if VLE_IS_ON // for VLE code 
static asm void Increment_MCSRR0_to_next_instruction(void)
{
    
    /* MCSRR0->r5 */
    mfmcsrr0 r5
    
    /* determine opcode @ MCSRR0 */
    se_lhz r4,0(r5)
    
    /* check bit 31,28 only*/
    e_andi. r3,r4,0x9000
    e_cmpli 0x0,r3,0x1000 
    
    e_bne __machine_check_adjust_for_16bit_opcode
    
        /* 0xx1 => 32 bit*/
        se_addi r5,2
    
    __machine_check_adjust_for_16bit_opcode:
    
        /* all others just 16 bit long*/
        se_addi r5,2 
        
        /* save adjusted return address*/
        mtmcsrr0 r5 
    
}
#else //BookE
static asm void Increment_MCSRR0_to_next_instruction(void)
{
    
    /* MCSRR0->r5 */
    mfmcsrr0 r5
    
    /* for BookE, all instructions are 32 bit long */
    se_addi r5,4
        
    /* save adjusted return address*/
    mtmcsrr0 r5 

}
#endif


/*******************************************************************************
Function Name : ClearMCSR
Engineer      : David Tosenovjan
Date          : Sep-16-2015
Parameters    : 
Modifies      : 
Returns       : 
Notes         : When MSR[ME] is set, error flags must be cleared to prevent 
                machine check exception to recall.                                 
Issues        : 
*******************************************************************************/
static asm void ClearMCSR(void)
{
    /* load mask */   
    e_lis r4, 0xFFFF
    se_subi  r4,0x0001
    
    /* Clear MCSR */
    mtmcsr r4
}


/*******************************************************************************
Function Name : ClearMSR_EE
Engineer      : David Tosenovjan
Date          : Sep-16-2015
Parameters    : 
Modifies      : 
Returns       : 
Notes         : Negates the MSR[EE].                                   
Issues        : 
*******************************************************************************/
static asm void ClearMSR_EE(void)
{
    /* read spr MSR */
    mfmsr r3  
    
    /* load mask to negate the EE bit */   
    //lis r4, 0xFFFF
    //addi r4, r4, 0x7FFF
    e_lis    r4,0xffff
    e_add16i r4,r4,0x7FFF
        
    
    /* clear EE bit */
    //and r3, r3, r4
    se_and   r3,r4
    
    /* write back to MSR */
    mtmsr r3  
}


/*******************************************************************************
Function Name : SetMSR_ME
Engineer      : David Tosenovjan
Date          : Sep-16-2015
Parameters    : 
Modifies      : 
Returns       : 
Notes         : Asserts the MSR[ME].                                   
Issues        : 
*******************************************************************************/
static asm void SetMSR_ME(void)
{ 
    /* read spr MSR */
    mfmsr r3
    
    /* load mask to assert the ME bit */   
    //lis r4, 0x0000
    //addi r4, r4, 0x1000
    e_lis    r4,0x0000
    e_add16i r4,r4,0x1000
    
    /* set ME bit */
    //or r3, r3, r4
    se_or    r3,r4
    
    /* write back to MSR */
    mtmsr r3 
}


/*******************************************************************************
Function Name : 
Engineer      : David Tosenovjan
Date          : 
Parameters    : 
Modifies      : 
Returns       : 
Notes         :                             
Issues        : 
*******************************************************************************/
void Inject_2bit_FLASH_ECC_error_by_UTEST_read(void)
{
    register uint32_t test_read = 0;
    
    // ECC is checked and non-correctable error found
    test_read = *(unsigned int*)0x00400060;
}    


/*******************************************************************************
Function Name : 
Engineer      : David Tosenovjan
Date          : 
Parameters    : 
Modifies      : 
Returns       : 
Notes         :                             
Issues        : 
*******************************************************************************/
void Inject_1bit_FLASH_ECC_error_by_UTEST_read(void)
{ 
    register uint32_t test_read = 0;

    /* Enable single bit ECC error reporting in flash controller */

    // Enable UTest mode
    C55FMC.UT0.R = 0xF9F99999;
    // Enable single bit error correction
    C55FMC.UT0.B.SBCE = 1;
    // Finish the UTest mode by writing UT0[UTE] with 0.
    C55FMC.UT0.B.UTE = 0;
    
    // ECC is checked and non-correctable error found
    test_read = *(unsigned int*)0x00400040;
}


/*******************************************************************************
* Global functions
*******************************************************************************/

/*******************************************************************************
Function Name : main
Engineer      : David Tosenovjan
Date          : Jan-09-2017
Parameters    : 
Modifies      : 
Returns       : 
Notes         : 
*******************************************************************************/
void main (void)
{
    Sys_Init();

    printf("ECC error injection example\r\n");
    
    INTC_Init();
    FCCU_Init();


    SetMSR_ME();
    //ClearMSR_EE();

    /**************************************************************************/
    /*                                                                        */
    /* Choose which ECC error is supposed to be injected !                    */
    /*                                                                        */
    /**************************************************************************/
    #if 1
        Inject_2bit_FLASH_ECC_error_by_UTEST_read();
    #else
        Inject_1bit_FLASH_ECC_error_by_UTEST_read();
    #endif
    
  while(1);
}//main
