/*******************************************************************************
* NXP Semiconductors
* (c) Copyright 2016 NXP Semiconductors
* ALL RIGHTS RESERVED.
********************************************************************************
Services performed by NXP in this matter are performed AS IS and without any 
warranty. CUSTOMER retains the final decision relative to the total design
and functionality of the end product. NXP neither guarantees nor will be held
liable by CUSTOMER for the success of this project.
NXP DISCLAIMS ALL WARRANTIES, EXPRESSED, IMPLIED OR STATUTORY INCLUDING,
BUT NOT LIMITED TO, IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR
A PARTICULAR PURPOSE ON ANY HARDWARE, SOFTWARE ORE ADVISE SUPPLIED 
TO THE PROJECT BY NXP, AND OR NAY PRODUCT RESULTING FROM NXP SERVICES. 
IN NO EVENT SHALL NXP BE LIABLE FOR INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING
OUT OF THIS AGREEMENT. 
CUSTOMER agrees to hold NXP harmless against any and all claims demands 
or actions by anyone on account of any damage, or injury, whether commercial,
contractual, or tortuous, rising directly or indirectly as a result 
of the advise or assistance supplied CUSTOMER in connection with product, 
services or goods supplied under this Agreement.
********************************************************************************
* File:             uart.c
* Owner:            David Tosenovjan
* Version:          1.1
* Date:             Mar-01-2016
* Classification:   General Business Information
* Brief:            terminal IO
********************************************************************************
* Detailed Description: 
* 
* Implements GHS ind_io library function calls to do printf() on eSCI_A
* Functions "read" and "write" defined in this module replace the ones from the
* library so finally content of printf() is sent to eSCI_A.
*
* ------------------------------------------------------------------------------
* Test HW:    MPC57xx_Motherboard + MPC5744P-144DC
* MCU:        PPC5744PFMLQ8,0N15P,QQAA1515N, Rev2.1B    
* Terminal:   19200-8-no parity-1 stop bit-no flow control on LINFlexD_0
* Fsys:       100MHz
*                    
********************************************************************************
Revision History:
Ver  Date         Author            Description of Changes
1.0  Dec-21-2015  Martin Kovar      Initial Version
1.1  Mar-01-2016  David Tosenovjan  dapted to GHS environment, minor edits
*******************************************************************************/

#include "MPC5744P.h"
#include "uart.h"

/*******************************************************************************
* External objects
*******************************************************************************/

/*******************************************************************************
* Global variables
*******************************************************************************/
int UART_initialized = 0;

/*******************************************************************************
* Constants and macros
*******************************************************************************/

/*******************************************************************************
* Local types
*******************************************************************************/

/*******************************************************************************
* Local function prototypes
*******************************************************************************/

/*******************************************************************************
* Local variables
*******************************************************************************/

/*******************************************************************************
* Local functions
*******************************************************************************/

/*******************************************************************************
Function Name : Init_LINFlexD_0
Engineer      : David Tosenovjan
Date          : 
Parameters    : 
Modifies      : 
Returns       : 
Notes         : initializes MPC5644P's Init_LINFlexD_0 module for UART mode
Issues        : expecting system clock is 100MHz
*******************************************************************************/
void LINFlexD_0_Init (void)
{
    /* enter INIT mode */
    LINFlexD_0.LINCR1.R = 0x0081; /* SLEEP=0, INIT=1 */
    

    /* wait for the INIT mode */
    while (0x1000 != (LINFlexD_0.LINSR.R & 0xF000)) {}
        
    /* configure pads */
    SIUL2.MSCR[18].B.SSS = 1;    // Pad PB2: Source signal is LIN0_TXD
    SIUL2.MSCR[18].B.OBE = 1;    // Pad PB2: Output Buffer Enable
    SIUL2.MSCR[18].B.SRC = 3;    // Pad PB2: Maximum slew rate
    SIUL2.IMCR[165].B.SSS = 1;   // LIN0_RXD: connected to pad PB3
    SIUL2.MSCR[19].B.IBE = 1;    // Pad PB3: Enable pad for input - LIN0_RXD

    /* configure for UART mode */

    // set the UART bit first to be able to write the other bits
    LINFlexD_0.UARTCR.R = 0x0001; 
    
    // 8bit data, no parity, Tx and Rx enabled, UART mode
    LINFlexD_0.UARTCR.R = 0x0033;  // Transmit buffer size = 1 (TDFL = 0)
                                   // Receive buffer size = 1 (RDFL = 0)
    
    /* configure baudrate 19200 */
    /* assuming 64 MHz peripheral set 1 clock (fsys below)*/
    /* LFDIV = fsys / (16 * desired baudrate)
       LINIBRR = integer part of LFDIV
       LINFBRR = 16 * fractional part of LFDIV (after decimal point)
       
       for instance:
       LFDIV = 100e6/(16*19200) = 325.52
       LINIBRR = 325
       LINFBRR = 16*0.52 = 8.32
    */

    LINFlexD_0.LINFBRR.R = 8;
    LINFlexD_0.LINIBRR.R = 325;
    
    /* enter NORMAL mode */
    LINFlexD_0.LINCR1.R = 0x0080; /* INIT=0 */
}


/*******************************************************************************
Function Name : TransmitData
Engineer      : David Tosenovjan
Date          : 
Parameters    : pBuf - input string. won't be modified by the function
              : cnt  - input number of characters to be transmitted
Modifies      : 
Returns       : 
Notes         : Tx data on LINFlexD_0. polled mode.
Issues        :  
*******************************************************************************/
void TransmitData(const char * const pBuf, const unsigned int cnt)
{
    uint8_t j = 0; // Dummy variable
    
    for (j=0; j< cnt; j++) 
    {  // Loop for character string  
        LINFlexD_0.BDRL.B.DATA0 = *(pBuf+j);
          //write character to transmit buffer
        while (1 != LINFlexD_0.UARTSR.B.DTFTFF) {}
          // Wait for data transmission completed flag
        LINFlexD_0.UARTSR.R = 0x0002;
          // clear the DTF flag and not the other flags
    }

    
}

/*******************************************************************************
Function Name : ReceiveData
Engineer      : David Tosenovjan
Date          : Apr-14-2011
Parameters    : pBuf - address of a char where the received char is written to
                       the address (pBuf) doesn't change in the function
Modifies      : 
Returns       : 
Notes         : Rx data on LINFlexD_0. polled mode.
Issues        :  
*******************************************************************************/
void ReceiveData(char * const pBuf)
{
    
    int32_t rx_data;
    
    /* Wait for data reception completed flag */
    while (1 != LINFlexD_0.UARTSR.B.DRFRFE) {}  
        
    /* Wait for Release Message Buffer */
    while (1 != LINFlexD_0.UARTSR.B.RMB) {}  
    
    /* get the data */
    
    rx_data = LINFlexD_0.BDRM.R; // read whole register due to erratum e4897PS
    
    *pBuf = (uint8_t)rx_data; // take 
    
    /* clear the DRF and RMB flags by writing 1 to them */
    LINFlexD_0.UARTSR.R = 0x0204;

}


/*******************************************************************************
* Global functions
*******************************************************************************/


/*
Methods called by GHS ind_io libraries to perform console IO:
*/


int read (int fno, void *buf, int size)
{
    int count;
    
    if (UART_initialized == 0)
    {
        LINFlexD_0_Init();
        UART_initialized = 1;
    }

    for (count = 0; count < size; count++)
    {
        ReceiveData( (char *)buf + count );
    }

    return size;
}


int write (int fno, const void *buf, int size)
{ 
    if (UART_initialized == 0)
    {
        LINFlexD_0_Init();
        UART_initialized = 1;
    }

    TransmitData ((const char *)buf,size);
    return size; 
}



