
/*
 * Copyright (c) 2013 - 2015, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "fsl_debug_console.h"
#include "fsl_mecc.h"

#define MECC1_OCRAM1_START_ADDR 0x20240000  // OCRAM1 512KB
#define MECC1_OCRAM1_END_ADDR   0x202BFFFF  
#define MECC2_OCRAM2_START_ADDR 0x202C0000  // OCRAM2 512KB
#define MECC2_OCRAM2_END_ADDR   0x2033FFFF  

typedef struct 
{
    uint32_t low;
    uint32_t high;
    uint8_t  ecc;
}ecc_erro_injection_t;

const ecc_erro_injection_t ecc_error_injection_single_bit = 
{
    // Set any bit to be 1 would inject singel bit ecc error
    0x00000001,
    0x00000000,
    0x00
};
const ecc_erro_injection_t ecc_error_injection_multi_bit = 
{
    // Set any more than 1 bit to be 1 would inject multi bit ecc error
    0x00000003,
    0x00000000,
    0x00
};
const ecc_erro_injection_t ecc_error_injection_disable = 
{
    // Set all bit to be 0 would disable error injection
    0x00000000,
    0x00000000,
    0x00
};

/*
    [CM7 and CM4]: OCRAM1 and OCRAM2 ECC    
    Bit 840[2], 950[0] should be fused for MECC
*/
void MECC_enable(void)
{
    // Already done by ROM
    // MECC1->PIPE_ECC_EN |= MECC_PIPE_ECC_EN_ECC_EN_MASK;
    // MECC2->PIPE_ECC_EN |= MECC_PIPE_ECC_EN_ECC_EN_MASK;
    
    PRINTF("MECCn->PIPE_ECC_EN bit 4 as ECC_EN should be set to be 1 by ROM.\r\n");
    PRINTF("MECC1->PIPE_ECC_EN: %x \r\n", MECC1->PIPE_ECC_EN);
    PRINTF("MECC2->PIPE_ECC_EN: %x \r\n", MECC2->PIPE_ECC_EN);
}


void MECC_app_init(void)
{
    mecc_config_t config;
    MECC_GetDefaultConfig(&config);
    config.enableMecc         = true;
    config.Ocram1StartAddress = MECC1_OCRAM1_START_ADDR;
    config.Ocram1EndAddress   = MECC1_OCRAM1_END_ADDR;
    MECC_Init(MECC1, &config);
    EnableIRQ(MECC1_INT_IRQn);
    EnableIRQ(MECC1_FATAL_INT_IRQn);
    
    MECC_EnableInterrupts(MECC1, kMECC_SingleError0InterruptEnable |
                                 kMECC_SingleError1InterruptEnable |
                                 kMECC_SingleError2InterruptEnable |
                                 kMECC_SingleError3InterruptEnable);
    
    MECC_EnableInterrupts(MECC1, kMECC_MultiError0InterruptEnable |
                                 kMECC_MultiError1InterruptEnable |
                                 kMECC_MultiError2InterruptEnable |
                                 kMECC_MultiError3InterruptEnable);
    
    config.Ocram1StartAddress = MECC2_OCRAM2_START_ADDR;
    config.Ocram1EndAddress   = MECC2_OCRAM2_END_ADDR;
    MECC_Init(MECC2, &config);
    EnableIRQ(MECC2_INT_IRQn);
    EnableIRQ(MECC2_FATAL_INT_IRQn);
    
    MECC_EnableInterrupts(MECC2, kMECC_SingleError0InterruptEnable |
                                 kMECC_SingleError1InterruptEnable |
                                 kMECC_SingleError2InterruptEnable |
                                 kMECC_SingleError3InterruptEnable);
    
    MECC_EnableInterrupts(MECC2, kMECC_MultiError0InterruptEnable |
                                 kMECC_MultiError1InterruptEnable |
                                 kMECC_MultiError2InterruptEnable |
                                 kMECC_MultiError3InterruptEnable);
}

static uint32_t addr_to_bank(uint32_t address)
{
    uint32_t bank;
    uint32_t mod;
    address = address & 0xff; // accelerate mod computation
    mod = address % 0x20;

    if(mod < 0x8)      bank = kMECC_OcramBank0;
    else if(mod<0x10)  bank = kMECC_OcramBank1;
    else if(mod<0x18)  bank = kMECC_OcramBank2;
    else               bank = kMECC_OcramBank3;
    
    return bank;
}

void MECC_error_injection(MECC_Type * mecc,
                          uint64_t  * address,
                          uint64_t    data,
                          const ecc_erro_injection_t * ecc_erro_injection)
{
    uint32_t bank;
    uint32_t high;
    uint32_t low;
    
    low  = data;
    high = data>>32;
    PRINTF("[Error injection]\r\n");
    PRINTF("address = 0x%x\r\n", address);
    PRINTF("data = 0x%x%x\r\n", high, low);
    
    bank = addr_to_bank( (uint32_t)address );
    
    // Enable error injection
    MECC_ErrorInjection(mecc, 
                        ecc_erro_injection->low, 
                        ecc_erro_injection->high, 
                        ecc_erro_injection->ecc, 
                        bank); 

    // Inject wrong data
    *address = data;
    asm(" dsb");
    
    // After error injection, disable error injection feature.
    MECC_ErrorInjection(mecc,
                        ecc_error_injection_disable.low,
                        ecc_error_injection_disable.high,
                        ecc_error_injection_disable.ecc,
                        bank);
    
    PRINTF("\r\n");
}
void MECC_touch(volatile uint64_t * address)
{
    // Here, read an address with ECC feature enabled.
    // If here is ECC error, IRQ would be triggered.
    // This code use 64 bit read, 32 bit read can also be used here.
    volatile uint64_t r;
    uint32_t high;
    uint32_t low;
    
    PRINTF("[Touch]\r\n");
    PRINTF("Address = 0x%x. \r\n", address);
    PRINTF("Begin:\r\n");
    r = *address;
    low = r;
    high = r>>32;
    PRINTF("Data = 0x%x_%x. \r\n", high, low);
    PRINTF("End.\r\n");
}

static void example_mecc1_single_bit_ecc_error(void)
{
    // Single bit injection and trigger
    MECC_error_injection(MECC1,
                         (uint64_t *)0x20240020,           // Address
                         0x11223344aabbccddul,             // Data
                         &ecc_error_injection_single_bit); // Single bit error injection
    MECC_touch((volatile uint64_t *)0x20240020);
}
static void example_mecc1_multi_bit_ecc_error(void)
{
    // Multi bit injection and trigger
    MECC_error_injection(MECC1,
                         (uint64_t *)0x20240028,           // Address
                         0x11223344aabbccddul,             // Data
                         &ecc_error_injection_multi_bit);  // Multi bit error injection
    MECC_touch((volatile uint64_t *)0x20240028);
}
static void example_mecc2_single_bit_ecc_error(void)
{
    // Single bit injection and trigger
    MECC_error_injection(MECC2,
                         (uint64_t *)0x202c0020,           // Address
                         0x11223344aabbccddul,             // Data
                         &ecc_error_injection_single_bit); // Single bit error injection
    MECC_touch((volatile uint64_t *)0x202c0020);
}
static void example_mecc2_multi_bit_ecc_error(void)
{
    // Multi bit injection and trigger
    MECC_error_injection(MECC2,
                         (uint64_t *)0x202c0028,           // Address
                         0x11223344aabbccddul,             // Data
                         &ecc_error_injection_multi_bit);  // Multi bit error injection
    MECC_touch((volatile uint64_t *)0x202c0028);
}


void MECC_example(void)
{
    PRINTF("MECC_example \r\n");
    MECC_enable();
    MECC_app_init();
    PRINTF("MECC2->ERR_STATUS: %x \r\n",  MECC2->ERR_STATUS);
    PRINTF("MECC2->ERR_STAT_EN: %x \r\n", MECC2->ERR_STAT_EN);
    PRINTF("MECC_example \r\n");

    PRINTF("\r\n\r\n");
    PRINTF("ecc1_single_bit_ecc, press any key to continue. \r\n"); GETCHAR();
    example_mecc1_single_bit_ecc_error();

    PRINTF("\r\n\r\n");    
    PRINTF("ecc1_multi_bit_ecc, press any key to continue. \r\n"); GETCHAR();
    example_mecc1_multi_bit_ecc_error();
    
    PRINTF("\r\n\r\n");  
    PRINTF("ecc2_single_bit_ecc, press any key to continue. \r\n"); GETCHAR();
    example_mecc2_single_bit_ecc_error();
    
    PRINTF("\r\n\r\n");  
    PRINTF("ecc2_multi_bit_ecc, press any key to continue. \r\n"); GETCHAR();
    example_mecc2_multi_bit_ecc_error();
    
    
    PRINTF("Demo end.\r\n\r\n");  
}

void MECC1_INT_IRQHandler(void)
{
    uint32_t intStatus;
    intStatus = MECC_GetStatusFlags(MECC1);
    MECC_ClearStatusFlags(MECC1, intStatus);
    PRINTF("-----IRQ-----\r\n");
    PRINTF("MECC1_INT_IRQHandler, status = 0x%x \r\n", intStatus);
    PRINTF("Single bit ECC error correction event happens.\r\n");
    PRINTF("-----Exit-----\r\n");
    SDK_ISR_EXIT_BARRIER;   
}
void MECC1_FATAL_INT_IRQHandler(void)
{
    uint32_t intStatus;
    intStatus = MECC_GetStatusFlags(MECC1);
    MECC_ClearStatusFlags(MECC1, intStatus);
    PRINTF("-----IRQ-----\r\n");
    PRINTF("MECC1_FATAL_INT_IRQHandler, status = 0x%x \r\n", intStatus);
    PRINTF("Multi bit ECC error detection event happens.\r\n");
    PRINTF("-----Exit-----\r\n");
    SDK_ISR_EXIT_BARRIER;
}
void MECC2_INT_IRQHandler(void)
{
    uint32_t intStatus;
    intStatus = MECC_GetStatusFlags(MECC2);
    MECC_ClearStatusFlags(MECC2, intStatus);
    PRINTF("-----IRQ-----\r\n");
    PRINTF("MECC2_INT_IRQHandler, status = 0x%x \r\n", intStatus);
    PRINTF("Single bit ECC error correction event happens.\r\n");
    PRINTF("-----Exit-----\r\n");
    SDK_ISR_EXIT_BARRIER;
}
void MECC2_FATAL_INT_IRQHandler(void)
{
    uint32_t intStatus;
    intStatus = MECC_GetStatusFlags(MECC2);
    MECC_ClearStatusFlags(MECC2, intStatus);
    PRINTF("-----IRQ-----\r\n");
    PRINTF("MECC2_FATAL_INT_IRQHandler, status = 0x%x \r\n", intStatus);
    PRINTF("Multi bit ECC error detection event happens.\r\n");
    PRINTF("-----Exit-----\r\n");
    SDK_ISR_EXIT_BARRIER;
}
