/*
 * Copyright 2020 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "FreeRTOS.h"
#include "task.h"

#include "fsl_debug_console.h"
#include "littlevgl_support.h"
#include "pin_mux.h"
#include "board.h"
#include "lvgl.h"
#include "gui_guider.h"
#include "events_init.h"
#include "custom.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/

/*******************************************************************************
 * Function Declarations
 ******************************************************************************/
void gpio_task_cb(lv_task_t task);

/*******************************************************************************
 * Variables
 ******************************************************************************/
static volatile bool s_lvgl_initialized = false;
lv_ui guider_ui;
uint32_t mySemaphore;

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
void BOARD_USER_BUTTON_IRQ_HANDLER(void)
{
	/* clear the interrupt status */
	GPIO_PortClearInterruptFlags(BOARD_USER_BUTTON_GPIO, 1U << BOARD_USER_BUTTON_GPIO_PIN);
	mySemaphore = 1;
	/* Insert LVGL code. */
	SDK_ISR_EXIT_BARRIER;
}

void gpio_task_cb(lv_task_t task)
{
    if(mySemaphore) {
        mySemaphore = 0;
        lv_event_send(guider_ui.screen_plus, LV_EVENT_CLICKED, NULL);
    }
}


static void AppTask(void *param)
{
	gpio_pin_config_t sw_config = {
	    kGPIO_DigitalInput,
	    0,
	    kGPIO_IntRisingEdge,
	};
	lv_task_t * gpio_task;

    lv_port_pre_init();
    lv_init();
    lv_port_disp_init();
    lv_port_indev_init();

    mySemaphore = 0;

    EnableIRQ(BOARD_USER_BUTTON_IRQ);
    GPIO_PinInit(BOARD_USER_BUTTON_GPIO, BOARD_USER_BUTTON_GPIO_PIN, &sw_config);

    /* Enable GPIO pin interrupt */
    GPIO_PortEnableInterrupts(BOARD_USER_BUTTON_GPIO, 1U << BOARD_USER_BUTTON_GPIO_PIN);

    s_lvgl_initialized = true;

    setup_ui(&guider_ui);
    events_init(&guider_ui);
    custom_init(&guider_ui);

    gpio_task = lv_task_create((lv_task_cb_t)gpio_task_cb, 200, LV_TASK_PRIO_HIGH, NULL);

    for (;;)
    {
        lv_task_handler();
        vTaskDelay(5);
    }
}

/*******************************************************************************
 * Code
 ******************************************************************************/
/*!
 * @brief Main function
 */
int main(void)
{
	/*******************************************************************
	 * AN13217 Software -
	 *
	 * This application is a modified version of the buttonCounterDemo
	 * provided in the GUI Guider deliverable.  The modifications made
	 * are as follows:
	 *
	 * 1) Pressing the minus sign should toggle the USER_LED, D18.
	 *
	 * 2) Pressing the user pushbutton, USER_BUTTON (SW8), will
	 *    increment the count and update the counter display on the
	 *    screen.
	 *
	 * Supported boards:
	 * IMXRT1050-EVKB
	 *
	 */
    BaseType_t stat;

    /* Init board hardware. */
    /* Set the eLCDIF read_qos priority high, to make sure eLCDIF
     * can fetch data in time when PXP is used.
     */
    *((volatile uint32_t *)0x41044100) = 5;

    BOARD_ConfigMPU();
    BOARD_InitPins();
    BOARD_InitI2C1Pins();
    BOARD_InitSemcPins();
    BOARD_BootClockRUN();
    BOARD_InitDebugConsole();

    USER_LED_INIT(LOGIC_LED_OFF);
    USER_LED_ON();



    stat = xTaskCreate(AppTask, "littlevgl", configMINIMAL_STACK_SIZE + 800, NULL, tskIDLE_PRIORITY + 2, NULL);

    if (pdPASS != stat)
    {
        PRINTF("Failed to create littlevgl task");
        while (1)
            ;
    }

    vTaskStartScheduler();

    for (;;)
    {
    } /* should never get here */
}

/*!
 * @brief Malloc failed hook.
 */
void vApplicationMallocFailedHook(void)
{
    for (;;)
        ;
}

/*!
 * @brief FreeRTOS tick hook.
 */
void vApplicationTickHook(void)
{
    if (s_lvgl_initialized)
    {
        lv_tick_inc(1);
    }
}

/*!
 * @brief Stack overflow hook.
 */
void vApplicationStackOverflowHook(TaskHandle_t xTask, signed char *pcTaskName)
{
    (void)pcTaskName;
    (void)xTask;

    for (;;)
        ;
}
