/*
 * Copyright 2017-2020 NXP
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms.  By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms.  If you do not agree to
 * be bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#ifndef AUDIO_INTERNAL_H_
#define AUDIO_INTERNAL_H_

#define PHLOG_MODULE  "audio"
#include "log.h"
#include "state.h"
#include "error.h"
#include "audio.h"
#include "board.h"

/** TX DMA transfer size based on audio_GetTxSampleSizeBytes() */
#define TX_AUDIO_DMA_TRANSFER_SIZE(x)  (AUDIO_SERVICE_SYSTEM_SAMPLING_RATE * x * TX_AUDIO_CHANNELS)
/** RX DMA transfer size*/
#define RX_AUDIO_DMA_TRANSFER_SIZE  (AUDIO_SERVICE_SYSTEM_SAMPLING_RATE * AUDIO_RX_SAMPLE_SIZE_BYTES * RX_AUDIO_CHANNELS)

#define AUDIO_DMA_INTERRUPT_PRIORITY  (1u)
#define AUDIO_DMA_INTERRUPT_SOURCE    (DMA0_IRQn)

typedef struct _audio_service_ctxt {
    audio_ctrl_config_t userConfig;
    state_t currentState;
    bool isBusy;
} audio_service_ctxt_t;

extern audio_service_ctxt_t g_AudioServiceCtxt;

/**
 * @brief Initializes the ring buffer for Tx
 */
void audio_RingbufferTxInit(void);

/**
 * @brief Initializes the ring buffer for Rx
 */
void audio_RingbufferRxInit(void);

/**
 * @brief Sets the audio interfcae word length for the Tx channel and allows to choose if the KL
 * acts as I2S master or slave.
 *
 * @param audioIfWordLength Audio interface word length, use @ref audioIfWordLength_t.
 * @param masterSlave select if KL should be I2S master or slave.
 * @param audioFormat audio format of tx
 * @return kERROR_Ok if Setting is success, kERROR_Fail if setting failed.
 */
error_t audio_SetTxI2sConfig(audioIfWordLength_t audioIfWordLength, sai_master_slave_t masterSlave, audioCtrlFormat_t audioFormat);

/**
 * @brief Sets the audio inteface word length for the Rx channel and allows to choose if the KL
 * acts as I2S master or slave.
 *
 * @param audioIfWordLength Audio interface word length, use @ref audioIfWordLength_t.
 * @param masterSlave select if KL should be I2S master or slave.
 * @return kERROR_Ok if Setting is success, kERROR_Fail if setting failed.
 */
error_t audio_SetRxI2sConfig(audioIfWordLength_t audioIfWordLength, sai_master_slave_t masterSlave);

/**
 * @brief returns the audio Tx sample size in bytes.
 *
 * @return audio Tx sample size in bytes.
 */
uint8_t audio_GetTxSampleSizeBytes(void);

#endif /* AUDIO_INTERNAL_H_ */
