/*
 * Copyright 2019-2020 NXP
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms.  By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms.  If you do not agree to
 * be bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/**
 * @defgroup KL_MODULES_AUDIOSERVICE_RX_API RX API
 * @ingroup KL_MODULES_AUDIOSERVICE
 *
 * The audio RX (receiver) service is responsible for receiving audio data over I2S.
 *
 * For this, it initializes the I2S and DMA interfaces using the KL ports and registers.
 * Audio packets are transferred from the I2S block to the KL over DMA and then read from a ring buffer.
 *
 * Check the @ref KL_MODULES_AUDIOSERVICE_RX_SEQ_DIG "sequence diagram" for more info.
 *
 * @{
 */

#ifndef AUDIO_RX_H_
#define AUDIO_RX_H_

/* ---------------------------------------------------------------------------- */
/* Include files                                                                */
/* ---------------------------------------------------------------------------- */

#include <stdint.h>


/* -------------------------------------------------------------------------
 * Public functions
 * ------------------------------------------------------------------------- */

/**
 * Reads the data of the input ring buffer.
 * @note The Receiver starts itself as soon as enough samples were buffered.
 *
 * @param pOutBuffer buffer that is going to be read.
 * @param NbSamples Number of samples to read.
 */
void audio_ReadBuffer(uint8_t *pOutBuffer, uint16_t NbSamples);

/**
 * Returns the current available space in the rx buffer.
 *
 * @return rx available space.
 */
uint32_t audio_RxAvailableSpace(void);

/**
 * Returns the current rx buffer size.
 *
 * @return current buffer size.
 */
uint32_t audio_GetRxBufferSize(void);

/**
 * Initialization of Receiver part of audio
 *  Configures I2S, Inits I2S, Inits Audio Formats, Inits SAI,
 *  Inits DMA
 */
void audio_InitRx(void);

/**
 * Start receiving audio samples from I2S
 *  Starts DMA, Starts SAI
 */
void audio_StartRx(void);

/**
 * Stops the data reception.
 */
void audio_StopRx(void);

/**
 * Resets the receiver
 */
void audio_ResetRx(void);

/** @} */

/**
 * @{
 *
 * @defgroup KL_MODULES_AUDIOSERVICE_RX_SEQ_DIG RX sequence diagrams
 * @ingroup KL_MODULES_AUDIOSERVICE
 * @par Sequence diagram of Audio RX Service
 *
 * @msc
 * msc=omegapple;
 * hscale=auto;
 * text.wrap=no;
 * RX [label="Audio RX"];
 * RB [label="Audio RingBuffer"];
 * DMA [label="DMA Interface"];
 * SAI [label="SAI"];
 * |||;
 *
 * ---: I2S pins initialized;
 * ->RX: Initialize RX [url="@ref audio_InitRx"];
 * RX->SAI: Initialize RX SAI;
 * RX->SAI: Enable RX I2S;
 * RX->DMA: Configure RX DMA;
 * <<RX;
 *
 * ...: ;
 * ...: ;
 * RX--SAI: Start reading samples regularly [line.corner = note];
 * ->RX: Read samples [URL = "@ref audio_ReadBuffer"];
 * RX->RB: Read samples from ring buffer;
 * box RX..SAI: if (!started && bufferHalfFilled)
 * {
 *   RX->DMA: Start RX DMA;
 * };
 * <<RX;
 * ...: ;
 * RX<<DMA: DMA RX CallBack;
 * RX->RB: Update write pointer;
 * RX->DMA: Prepare next DMA transfer;
 *
 * @endmsc
 */

/** @} */

#endif /* AUDIO_RX_H_ */
