/*
 * Copyright 2017-2020 NXP
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms.  By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms.  If you do not agree to
 * be bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#ifndef DMA_INTERFACE_H_
#define DMA_INTERFACE_H_

#include "fsl_dma.h"

/**
 * @defgroup KL_MODULES_AUDIOSERVICE_DMA dma_interface
 * @ingroup KL_MODULES_AUDIOSERVICE
 * dma_interface is an abstraction layer for the DMA driver.
 * DMA registers and memory banks are accessed here.
 * This is a sub module of @ref KL_MODULES_AudioStreaming audio
 * @{
 */

/**
 * @brief Configure The DMA, Tx/Rx
 *
 * @param handle DMA handle to configure
 * @param DmaChannel DMA channel to configure
 * @param source DMA mux source
 * @return void.
 */
void dma_ConfigureDMA(dma_handle_t *handle, uint8_t DmaChannel, uint32_t source);

/**
 * @brief Configure the Link Tx/Rx DMA
 *
 * @param handle DMA handle to configure
 * @param DmaChannel DMA channel to configure
 * @param callback pointer to DMA call back function
 * @return void.
 */
void dma_ConfigureLinkDMA(dma_handle_t *handle, uint8_t DmaChannel, dma_callback callback);

/**
 * @brief returns the value of SAR-source address register
 *
 * @param base Pointer to the DMA registers
 * @param channel DMA channel to disable
 * @return bcrRegister address.
 */
uint32_t dma_GetSourceAddress(DMA_Type *base, uint8_t channel);

/**
 * @brief returns the value of DAR-Destination address register
 *
 * @param base Pointer to the DMA registers
 * @param channel DMA channel to disable
 * @return bcrRegister address.
 */
uint32_t dma_GetDestinationAddress(DMA_Type *base, uint8_t channel);

/**
 * @brief Returns the BCR-Byte Count Register address
 *
 * @param base Pointer to the DMA registers
 * @param channel DMA channel to disable
 * @return bcrRegister address.
 */
uint32_t dma_GetBcrRegisterAddress(DMA_Type *base, uint8_t channel);

/**
 * @brief Disables the DMA-Hardware request
 *
 * @param base Pointer to the DMA registers
 * @param channel DMA channel to disable
 * @param Status true/false
 * @return void.
 */
void dma_DisableHwRequest(DMA_Type *base, uint8_t channel, bool Status);

/** @} */

#endif /* DMA_INTERFACE_H_ */
