/* ---------------------------------------------------------------------------- */
/* Copyright 2020 NXP.                                                          */
/*                                                                              */
/* This software is owned or controlled by NXP and may only be used strictly    */
/* in accordance with the applicable license terms. By expressly accepting such */
/* terms or by downloading, installing, activating and/or otherwise using the   */
/* software, you are agreeing that you have read, and that you agree to comply  */
/* with and are bound by, such license terms. If you do not agree to be bound   */
/* by the applicable license terms, then you may not retain, install, activate  */
/* or otherwise use the software.                                               */
/* ---------------------------------------------------------------------------- */

#ifndef LOG_H_
#define LOG_H_

/**
 * @addtogroup KL_MODULES_DEBUG_LOGGING_API
 *
 * @anchor debug_logging
 *
 * This is a wrapper around the standard logging provided by the KL SDK (KSDK).\n
 * This implies whatever format specifiers that are supported by the KSDK, will be supported by these macro's as well. To use this debug logging the user has to compile with:
 * - @anchor debug_flag DEBUG
 * - SDK_DEBUGCONSOLE=1
 * - PHLOG_MODULE\n
 *   This is the string prefix (f.e. "spi_module") that will be used in debug logs to identify the module printing the message.
 * .
 *
 * @{
 */

#include "fsl_debug_console.h"

#ifndef PHLOG_MODULE
    #error "this file should not be included on it's own ..."
#endif

#ifdef DEBUG
/**
 * @ref PDBG helper Macro
 */
    #define PMDBG(__mod, __msg, ...)  PRINTF("[" __mod "] " __msg "\n", ## __VA_ARGS__)

/**
 * @ref PERR helper Macro
 */
    #define PMERR(__mod, __msg, ...)  PRINTF("[" __mod "] error: " __msg "\n", ## __VA_ARGS__)

    #include <stdio.h>
/**
 * Helper macro to print out an array, uses @ref PDBG to print.
 */
    #define PHEXDUMP(__prefix, __l, __d)  { \
        char __hexdump[(__l) * 3]; \
        __hexdump[0] = '\0'; \
        for (__typeof__(__l)__it = 0; __it < (__l); __it++) { \
            snprintf(__hexdump, sizeof(__hexdump), "%s%02x ", __hexdump, (__d)[__it]); \
        } \
        PDBG(__prefix " %d bytes [ %s]", __l, __hexdump); \
}
#else
/**
 * @ref PDBG helper Macro when compiled out
 */
    #define PMDBG(__mod, __msg, ...)
/**
 * @ref PERR helper Macro when compiled out
 */
    #define PMERR(__mod, __msg, ...)
/**
 * @ref PHEXDUMP when compiled out
 */
    #define PHEXDUMP(__prefix, __l, __d)
#endif

/**
 * @ref PINF helper Macro
 */
#define PMINF(__mod, __msg, ...)  PRINTF("[" __mod "] " __msg "\n", ## __VA_ARGS__)

/**
 * Macro to print general debug info. This is compiled out if the @ref debug_flag "debug flag" is **NOT** defined.
 */
#define PDBG(__m, ...)  PMDBG(PHLOG_MODULE, __m, ## __VA_ARGS__)

/**
 * Macro to print debug info that will **NOT** be compiled out
 * if the @ref debug_flag "debug flag" is not defined.
 */
#define PINF(__m, ...)  PMINF(PHLOG_MODULE, __m, ## __VA_ARGS__)

/**
 * Macro to print error info. This is compiled out if the @ref debug_flag "debug flag" is **NOT** defined.
 */
#define PERR(__m, ...)  PMERR(PHLOG_MODULE, __m, ## __VA_ARGS__)

/** @} */

#endif /* LOG_H_ */
