/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2020 NXP
 * All rights reserved.
 *
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*******************************************************************************
 * Includes
 ******************************************************************************/
#include "lwip/opt.h"

#include "udp_op.h"

#include "pin_mux.h"
#include "clock_config.h"
#include "board.h"
#include "fsl_phy.h"

#include "fsl_phyrtl8211f.h"
#include "fsl_enet_mdio.h"
#include "fsl_gpio.h"
#include "fsl_iomuxc.h"
#include "fsl_enet.h"
#include "fsl_rdc.h"
#include "fsl_mu.h"
#include "fsl_debug_console.h"
#include "lpm.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define RDC_DISABLE_A53_ACCESS 0xFC
#define RDC_DISABLE_M7_ACCESS  0xF3

/* Address of PHY interface. */
#define EXAMPLE_PHY_ADDRESS   BOARD_ENET0_PHY_ADDRESS
/* PHY operations. */
#define EXAMPLE_PHY_OPS       phyrtl8211f_ops
/* MDIO operations. */
#define EXAMPLE_MDIO_OPS enet_ops

/* ENET clock frequency. */
#define EXAMPLE_CLOCK_FREQ CLOCK_GetEnetAxiFreq()

/*! @brief Stack size of the temporary lwIP initialization thread. */
#define INIT_THREAD_STACKSIZE 1024

/*! @brief Priority of the temporary lwIP initialization thread. */
#define INIT_THREAD_PRIO DEFAULT_THREAD_PRIO

/* Task priorities. */
#define main_task_PRIORITY      (2)

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/

static mdio_handle_t mdioHandle = {.ops = &EXAMPLE_MDIO_OPS};
static phy_handle_t phyHandle   = {.phyAddr = EXAMPLE_PHY_ADDRESS, .mdioHandle = &mdioHandle, .ops = &EXAMPLE_PHY_OPS};

static SemaphoreHandle_t xRmtcoreSuspSema = NULL;

/*******************************************************************************
 * Code
 ******************************************************************************/
/*
 * MU Interrrupt RPMsg handler
 */
int32_t MU1_M7_IRQHandler(void)
{
    if (kMU_GenInt0Flag & MU_GetStatusFlags(MUB))
    {
        BaseType_t reschedule = 0;
        MU_ClearStatusFlags(MUB, kMU_GenInt0Flag);
        xSemaphoreGiveFromISR(xRmtcoreSuspSema, &reschedule);
        portYIELD_FROM_ISR(reschedule);
    }

#if (defined __CORTEX_M) && ((__CORTEX_M == 4U) || (__CORTEX_M == 7U))
    __DSB();
#endif

    return 0;
}

static void BOARD_InitModuleClock(void)
{
    if (CLOCK_IsRootEnabled(kCLOCK_RootEnetAxi))
    {
        if (CLOCK_GetRootMux(kCLOCK_RootEnetAxi) != kCLOCK_EnetAxiRootmuxSysPll2Div4)
        {
            PRINTF("Change enet axi clock root to SysPll2Div4\r\n");
            CLOCK_DisableRoot(kCLOCK_RootEnetAxi);
            /* SYS_PLL2(1000M) / 3 = 250MHz */
            CLOCK_SetRootMux(kCLOCK_RootEnetAxi, kCLOCK_EnetAxiRootmuxSysPll2Div4);

            CLOCK_EnableRoot(kCLOCK_RootEnetAxi);
        }
    }
    else
    {
        PRINTF("Enable enet axi clock root to SysPll2Div4\r\n");
        CLOCK_SetRootMux(kCLOCK_RootEnetAxi, kCLOCK_EnetAxiRootmuxSysPll2Div4);
        CLOCK_EnableRoot(kCLOCK_RootEnetAxi);
    }

    if (CLOCK_IsRootEnabled(kCLOCK_RootEnetRef))
    {
        if (CLOCK_GetRootMux(kCLOCK_RootEnetRef) != kCLOCK_EnetRefRootmuxSysPll2Div8)
        {
            PRINTF("Change enet ref clock root to SysPll2Div8\r\n");
            CLOCK_DisableRoot(kCLOCK_RootEnetRef);
            /* SYS_PLL2(1000M) / 8 = 125MHz */
            CLOCK_SetRootMux(kCLOCK_RootEnetRef, kCLOCK_EnetRefRootmuxSysPll2Div8);
            CLOCK_EnableRoot(kCLOCK_RootEnetRef);
        }
    }
    else
    {
        PRINTF("Enable enet ref clock root to SysPll2Div8\r\n");
        CLOCK_SetRootMux(kCLOCK_RootEnetRef, kCLOCK_EnetRefRootmuxSysPll2Div8);
        CLOCK_EnableRoot(kCLOCK_RootEnetRef);
    }

   if (CLOCK_IsRootEnabled(kCLOCK_RootEnetPhy))
    {
        if (CLOCK_GetRootMux(kCLOCK_RootEnetPhy) != kCLOCK_EnetRefRootmuxOsc24M)
        {
            PRINTF("Change enet phy clock root to Osc24M\r\n");
            CLOCK_DisableRoot(kCLOCK_RootEnetPhy);
            /* 24M */
            CLOCK_SetRootMux(kCLOCK_RootEnetPhy, kCLOCK_EnetRefRootmuxOsc24M);
            CLOCK_EnableRoot(kCLOCK_RootEnetPhy);
        }
    }
    else
    {
        /* 24M */
        PRINTF("Enable enet phy clock root to Osc24M\r\n");
        CLOCK_SetRootMux(kCLOCK_RootEnetPhy, kCLOCK_EnetRefRootmuxOsc24M);
        CLOCK_EnableRoot(kCLOCK_RootEnetPhy);
    }

    /* Enable enet clocks */
    CLOCK_EnableClock(kCLOCK_Sim_enet);
    CLOCK_EnableClock(kCLOCK_Enet1);
}

static void IOMUXC_SelectENETClock(void)
{
    /* bit22: RGMII iomuxc_gpr_enet_clk_dir
     * bit13: RMII GPR_ENET_TX_CLK_SEL(internal or OSC) */
    IOMUXC_GPR->GPR1 |= IOMUXC_GPR_GPR1_IOMUXC_GPR_ENET1_RGMII_EN_MASK; 
}

void BOARD_ENETFlexibleConfigure(enet_config_t *config)
{
    config->miiMode = kENET_RgmiiMode;
}

static void BOARD_ResetPhy(void)
{
    gpio_pin_config_t gpio_config = {kGPIO_DigitalOutput, 0, kGPIO_NoIntmode};

    GPIO_PinInit(GPIO4, 2, &gpio_config);
    /* For a complete PHY reset of RTL8211FDI-CG, this pin must be asserted low for at least 10ms. And
     * wait for a further 30ms(for internal circuits settling time) before accessing the PHY register */
    GPIO_WritePinOutput(GPIO4, 2, 0);
    SDK_DelayAtLeastUs(10000, CLOCK_GetFreq(kCLOCK_CoreM7Clk));
    GPIO_WritePinOutput(GPIO4, 2, 1);
    SDK_DelayAtLeastUs(30000, CLOCK_GetFreq(kCLOCK_CoreM7Clk));
}

static void BOARD_InitEnet(void)
{
    BOARD_InitModuleClock();

    mdioHandle.resource.csrClock_Hz = EXAMPLE_CLOCK_FREQ;

    EnableIRQ(ENET1_MAC0_Rx_Tx_Done0_IRQn);
    EnableIRQ(ENET1_MAC0_Rx_Tx_Done1_IRQn);
}

static void BOARD_SetENetRdcForMCore(void)
{
    rdc_periph_access_config_t periphConfig;

    RDC_GetDefaultPeriphAccessConfig(&periphConfig);
    /* Do not allow the A53 domain(domain0) to access the following peripherals */
    periphConfig.policy = RDC_DISABLE_A53_ACCESS;
    periphConfig.periph = kRDC_Periph_ENET1;
    RDC_SetPeriphAccessConfig(RDC, &periphConfig);
    periphConfig.periph = kRDC_Periph_GPIO4;
    RDC_SetPeriphAccessConfig(RDC, &periphConfig);
}

static void BOARD_SetENetRdcForACore(void)
{
    rdc_periph_access_config_t periphConfig;

    periphConfig.policy = RDC_DISABLE_M7_ACCESS;
    periphConfig.periph = kRDC_Periph_ENET1;
    RDC_SetPeriphAccessConfig(RDC, &periphConfig);
    periphConfig.periph = kRDC_Periph_GPIO4;
    RDC_SetPeriphAccessConfig(RDC, &periphConfig);
}

static void BOARD_PeripheralRdcSetting(void)
{
    rdc_periph_access_config_t periphConfig;

    RDC_GetDefaultPeriphAccessConfig(&periphConfig);
    /* Do not allow the A53 domain(domain0) to access the following peripherals */
    periphConfig.policy = RDC_DISABLE_A53_ACCESS;
    periphConfig.periph = kRDC_Periph_UART4;
    RDC_SetPeriphAccessConfig(RDC, &periphConfig);
    periphConfig.periph = kRDC_Periph_GPT1;
    RDC_SetPeriphAccessConfig(RDC, &periphConfig);
}

static void Board_WaitForPMChange(mu_power_mode_t target_mode)
{
    mu_power_mode_t mode;
    do {
        mode = MU_GetOtherCorePowerMode(MUB);
    } while (target_mode != mode);
}

static int32_t Board_WakeupACore(void)
{
    status_t ret;

    ret = MU_TriggerInterrupts(MUB, kMU_GenInt0InterruptTrigger);
    Board_WaitForPMChange(kMU_PowerModeStop);

    return ret;
}

static int32_t Board_WaitForACoreSuspend(void)
{
    if (xSemaphoreTake(xRmtcoreSuspSema, portMAX_DELAY) == pdTRUE)
    {
        Board_WaitForPMChange(kMU_PowerModeRun);
        return 0;
    }
    else
    {
        return -1;
    }
}

static void Board_InitMU(void)
{
    MU_Init(MUB);
    MU_EnableInterrupts(MUB, (uint32_t)kMU_GenInt0InterruptEnable);
    NVIC_SetPriority(MU1_M7_IRQn, configLIBRARY_MAX_SYSCALL_INTERRUPT_PRIORITY + 1);
    EnableIRQ(MU1_M7_IRQn);
}

static void main_task(void *pvParameters)
{
    /* Start udp */
    udp_op_init();

    do
    {
        /* Wait for A core to suspend */
        PRINTF("Waiting for A core suspend Sema...\r\n");
        if (!Board_WaitForACoreSuspend())
        {
            BOARD_InitEnet();

            //BOARD_SetENetRdcForMCore();

            BOARD_ResetPhy();

            /* Start udp */
            udp_op_init_netif(&phyHandle);

            udp_op_thread(NULL);

            udp_op_deinit_netif();

            /* udp processing task will exits when wakeup packet is received */
            /* Release enet */
            //BOARD_SetENetRdcForACore();

            /* Wakeup A core */
            Board_WakeupACore();
        }
    } while (1);
}

/*!
 * @brief Main function
 */
int main(void)
{
#if defined(FSL_FEATURE_SOC_LPC_ENET_COUNT) && (FSL_FEATURE_SOC_LPC_ENET_COUNT > 0)
    static mem_range_t non_dma_memory[] = NON_DMA_MEMORY_ARRAY;
#endif /* FSL_FEATURE_SOC_LPC_ENET_COUNT */

    /* Init board hardware. */
    /* M7 has its local cache and enabled by default,
     * need to set smart subsystems (0x28000000 ~ 0x3FFFFFFF)
     * non-cacheable before accessing this address region */
    BOARD_InitMemory();

    /* Board specific RDC settings */
    BOARD_RdcInit();
    BOARD_PeripheralRdcSetting();

    BOARD_InitPins();
    BOARD_BootClockRUN();
    BOARD_InitDebugConsole();
    //BOARD_InitModuleClock();

    LPM_Init();

    PRINTF("\r\n************************************************\r\n");
    PRINTF(" LWIP M7 Demo using UDP protocol\r\n");
    PRINTF("    Build Time: %s--%s \r\n", __DATE__, __TIME__);
    PRINTF("************************************************\r\n");

    BOARD_InitEnet1GPins();
    IOMUXC_SelectENETClock();

    Board_InitMU();

    xRmtcoreSuspSema = xSemaphoreCreateBinary();

    /* Create main task */
    //if (!sys_thread_new("main_task", main_task, NULL, DEFAULT_THREAD_STACKSIZE, DEFAULT_THREAD_PRIO))
    //if (xTaskCreate(main_task, "main_task", configMINIMAL_STACK_SIZE + 100, NULL, main_task_PRIORITY, NULL) != pdPASS)
    if (xTaskCreate(main_task, "main_task", DEFAULT_THREAD_STACKSIZE, NULL, main_task_PRIORITY, NULL) != pdPASS)
    {
        PRINTF("Task creation failed!.\r\n");
        while (1)
            ;
    }

    vTaskStartScheduler();

    for (;;)
        ;

    /* Will not get here unless a task calls vTaskEndScheduler ()*/
    return 0;
}

void vApplicationMallocFailedHook(void)
{
    PRINTF("Malloc Failed!!!\r\n");
}
