/*
 * Copyright (c) 2001-2003 Swedish Institute of Computer Science.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
 * SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * This file is part of the lwIP TCP/IP stack.
 *
 * Author: Adam Dunkels <adam@sics.se>
 *
 */

#include "udp_op.h"
#include "lwip/opt.h"

#if LWIP_NETCONN

#include "lwip/api.h"
#include "lwip/sys.h"
#include "lwip/netifapi.h"
#include "lwip/tcpip.h"
#include "netif/ethernet.h"
#include "enet_ethernetif.h"
#include "board.h"
#include "fsl_enet.h"

/* ENET instance select. */
#ifndef EXAMPLE_NETIF_INIT_FN
/*! @brief Network interface initialization function. */
#define EXAMPLE_NETIF_INIT_FN ethernetif0_init
#endif /* EXAMPLE_NETIF_INIT_FN */

static struct netif netif;
static char rx_data_buf[PKT_DATA_BUF_SZ] = { 0 };
static char tx_data_buf[PKT_DATA_BUF_SZ] = { 0 };

static void udp_op_netif_remove_callback(struct netif *netif)
{
    ENET_Deinit(ENET1);

    /* If enet clock is not enabled, A core will get stuck after wakeup */
    CLOCK_EnableClock(s_enetClock[ENET_GetInstance(ENET1)]);
}

void udp_op_thread(void *arg)
{
    struct netconn *conn = NULL;
    struct netbuf *rx_netbuf = NULL;
    struct netbuf *tx_netbuf = NULL;
    err_t err;
    LWIP_UNUSED_ARG(arg);
    char *resp_str = NULL;
    uint32_t got_wakeup_pkt = 0;

#if LWIP_IPV6
    conn = netconn_new(NETCONN_UDP_IPV6);
    LWIP_ERROR("udpecho: invalid conn", (conn != NULL), return;);
    netconn_bind(conn, IP6_ADDR_ANY, configUDP_OP_SERVER_PORT);
#else /* LWIP_IPV6 */
    conn = netconn_new(NETCONN_UDP);
    LWIP_ERROR("udpecho: invalid conn", (conn != NULL), return;);
    netconn_bind(conn, IP_ADDR_ANY, configUDP_OP_SERVER_PORT);
#endif /* LWIP_IPV6 */

    while (1)
    {
        err = netconn_recv(conn, &rx_netbuf);
        if (err == ERR_OK)
        {
            uint16_t rx_data_len = rx_netbuf->p->tot_len;
            /*  no need netconn_connect here, since the netbuf contains the address */
            if(netbuf_copy(rx_netbuf, rx_data_buf, PKT_DATA_BUF_SZ) != rx_data_len)
            {
                LWIP_DEBUGF(LWIP_DBG_ON, ("netbuf_copy failed\n"));
            }
            else
            {
                uint32_t resp_data_len = 0;

                /* Check if it is a heartbeat packet */
                if ((rx_data_len == strlen(PKT_DATA_HEARTBEAT)) &&
                    (!memcmp(rx_data_buf, PKT_DATA_HEARTBEAT, rx_data_len)))
                {
                    resp_str = PKT_DATA_HEARTBEAT_RESP;
                }
                /* Check if it is a wakeup packet */
                else if ((rx_data_len == strlen(PKT_DATA_WAKEUP)) &&
                    (!memcmp(rx_data_buf, PKT_DATA_WAKEUP, rx_data_len)))
                {
                    resp_str = PKT_DATA_WAKEUP_RESP;
                    got_wakeup_pkt = 1;
                }

                if (resp_str)
                {
                    resp_data_len = strlen(resp_str);
                    /* Got heartbeat packet, send response back */
                    memcpy(tx_data_buf, resp_str, resp_data_len);
                    tx_data_buf[resp_data_len] = '\0';

                    tx_netbuf = netbuf_new();
                    if (NULL == tx_netbuf)
                    {
                        LWIP_DEBUGF(LWIP_DBG_ON, ("netbuf_new failed\n"));
                        continue;
                    }
                    if (netbuf_ref(tx_netbuf, tx_data_buf, resp_data_len))
                    {
                        LWIP_DEBUGF(LWIP_DBG_ON, ("netbuf_ref failed\n"));
                        continue;
                    }
                    netbuf_set_fromaddr(tx_netbuf, netbuf_fromaddr(rx_netbuf));
                    netbuf_fromport(tx_netbuf) = netbuf_fromport(rx_netbuf);
                    err = netconn_send(conn, tx_netbuf);
                    if(err != ERR_OK)
                    {
                        LWIP_DEBUGF(LWIP_DBG_ON, ("netconn_send failed: %d\n", (int)err));
                        continue;
                    }
                    else
                    {
                        PRINTF("Send response packet back!\r\n");
                    }
                    if (tx_netbuf)
                    {
                        netbuf_free(tx_netbuf);
                        netbuf_delete(tx_netbuf);
                        tx_netbuf = NULL;
                    }
                }    
                else
                {
                    PRINTF("Ignore unknown packet!\r\n");
                }
            }
        }

        if (rx_netbuf)
        {
            netbuf_delete(rx_netbuf);
            rx_netbuf = NULL;
        }

        /* Got wakeup packet, need to release enet control and wakeup A core */
        if (1 == got_wakeup_pkt)
        {
            break;
        }
    }

    if (conn)
    {
        netconn_delete(conn);
        conn = NULL;
    }
}
/*-----------------------------------------------------------------------------------*/
void udp_op_process_thr(void)
{
    sys_thread_new("udp_op_thread", udp_op_thread, NULL, DEFAULT_THREAD_STACKSIZE, DEFAULT_THREAD_PRIO);
}

void udp_op_deinit_netif()
{
    netifapi_netif_set_down(&netif);
    netifapi_netif_remove(&netif);
}

void udp_op_init_netif(phy_handle_t *h_phy)
{
    ip4_addr_t netif_ipaddr, netif_netmask, netif_gw;
    ethernetif_config_t enet_config = {
        .phyHandle  = h_phy,
        .macAddress = configMAC_ADDR,
#if defined(FSL_FEATURE_SOC_LPC_ENET_COUNT) && (FSL_FEATURE_SOC_LPC_ENET_COUNT > 0)
        .non_dma_memory = non_dma_memory,
#endif /* FSL_FEATURE_SOC_LPC_ENET_COUNT */
    };

    ip4_addr_set_u32(&netif_ipaddr,  configUDP_OP_SERVER_IP);
    ip4_addr_set_u32(&netif_netmask, configUDP_OP_SERVER_NET_MASK);
    ip4_addr_set_u32(&netif_gw,      configUDP_OP_GATEWAY_IP);

    netifapi_netif_add(&netif, &netif_ipaddr, &netif_netmask, &netif_gw, &enet_config, EXAMPLE_NETIF_INIT_FN,
                       tcpip_input);
    netifapi_netif_set_default(&netif);
    netifapi_netif_set_up(&netif);
    /* In remove callback, we'll disable enet interrupts */
    /* Calling to netif_set_remove_callback need to lock core first */
    sys_lock_tcpip_core();
    netif_set_remove_callback(&netif, udp_op_netif_remove_callback);
    sys_unlock_tcpip_core();

    PRINTF("\r\n************************************************\r\n");
    PRINTF(" IPv4 Address     : %u.%u.%u.%u\r\n", ((u8_t *)&netif_ipaddr)[0], ((u8_t *)&netif_ipaddr)[1],
           ((u8_t *)&netif_ipaddr)[2], ((u8_t *)&netif_ipaddr)[3]);
    PRINTF(" IPv4 Subnet mask : %u.%u.%u.%u\r\n", ((u8_t *)&netif_netmask)[0], ((u8_t *)&netif_netmask)[1],
           ((u8_t *)&netif_netmask)[2], ((u8_t *)&netif_netmask)[3]);
    PRINTF(" IPv4 Gateway     : %u.%u.%u.%u\r\n", ((u8_t *)&netif_gw)[0], ((u8_t *)&netif_gw)[1],
           ((u8_t *)&netif_gw)[2], ((u8_t *)&netif_gw)[3]);
    PRINTF("************************************************\r\n");
}

void udp_op_init()
{
    tcpip_init(NULL, NULL);
}

#endif /* LWIP_NETCONN */
