/*
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

/* FreeRTOS kernel includes. */
#include "FreeRTOS.h"
#include "task.h"
#include "queue.h"
#include "timers.h"
#include "semphr.h"

/* Freescale includes. */
#include "fsl_device_registers.h"
#include "fsl_debug_console.h"
#include "board.h"

#include "pin_mux.h"
#include "clock_config.h"

#include "fsl_rdc.h"
#include "fsl_gpio.h"
#include "fsl_gpc.h"
#include "fsl_gpt.h"
#include "fsl_iomuxc.h"
#include "fsl_mu.h"

#include "lpm.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define TEST_FREQ_SWITCH        (0)
#define TEST_REINIT_SYSPLL      (0)

#define USE_WAKEUP_TIMER        (1)
#define USE_WAKEUP_GPIO         (0)

#define APP_PowerUpSlot (5U)
#define APP_PowerDnSlot (6U)

#define RDC_DISABLE_A53_ACCESS 0xFC
#define RDC_DISABLE_M7_ACCESS 0xF3

#define TIME_DELAY_SLEEP      5000

extern uint32_t SystemCoreClock;

static SemaphoreHandle_t xWuSemaphore = NULL;

static portBASE_TYPE xHigherPriorityTaskWoken; 

/* Task priorities. */
#define menu_task_PRIORITY      (configMAX_PRIORITIES - 2)

#if TEST_FREQ_SWITCH
int32_t m4_chg_clk_flag = 0;
int32_t m4_vdelay_test = 0;

extern void vPortSetupTimerInterrupt( void );

static SemaphoreHandle_t xDelayTestSemaphore = NULL;
static SemaphoreHandle_t xTimeoutSemaphore = NULL;

#define tickless_task_PRIORITY   (configMAX_PRIORITIES - 1)
#endif

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
static void menu_task(void *pvParameters);
#if TEST_FREQ_SWITCH
static void Tickless_task(void *pvParameters);
#endif

/*******************************************************************************
 * Code
 ******************************************************************************/
#if USE_WAKEUP_TIMER
static void wakeupTimerCallback(TimerHandle_t xTimer)
{
    xHigherPriorityTaskWoken = pdFALSE;

    xSemaphoreGiveFromISR(xWuSemaphore, &xHigherPriorityTaskWoken);

    portYIELD_FROM_ISR(xHigherPriorityTaskWoken);
}
#endif

#if USE_WAKEUP_GPIO
void GPIO1_Combined_0_15_IRQHandler(void){
    //uint32_t iFlag;
    //iFlag = GPIO_PortGetInterruptFlags(GPIO1);
    //PRINTF("GPIO IRQ triggered 0x%x\r\n", iFlag);
    GPIO_PortClearInterruptFlags(GPIO1, 1U << 13);
#if defined __CORTEX_M && (__CORTEX_M == 4U || __CORTEX_M == 7U)
    __DSB();
#endif
    xSemaphoreGiveFromISR(xWuSemaphore, &xHigherPriorityTaskWoken);

    portYIELD_FROM_ISR(xHigherPriorityTaskWoken);
}
#endif

void Peripheral_RdcSetting(void)
{
    rdc_periph_access_config_t periphConfig;

    RDC_GetDefaultPeriphAccessConfig(&periphConfig);
    /* Do not allow the A53 domain(domain0) to access the following peripherals. */
    periphConfig.policy = RDC_DISABLE_A53_ACCESS;
    periphConfig.periph = kRDC_Periph_UART4;
    RDC_SetPeriphAccessConfig(RDC, &periphConfig);
    //periphConfig.periph = kRDC_Periph_SNVS_HP;
    //RDC_SetPeriphAccessConfig(RDC, &periphConfig);
#if USE_WAKEUP_GPIO
    periphConfig.periph = kRDC_Periph_GPIO1;
    RDC_SetPeriphAccessConfig(RDC, &periphConfig);
#endif
#if USE_WAKEUP_TIMER
    periphConfig.periph = kRDC_Periph_GPT1;
    RDC_SetPeriphAccessConfig(RDC, &periphConfig);
#endif
    /* Do not allow the m4 domain(domain1) to access SAI3.
     * The purpose is to avoid system hang when A core to access SAI3 once M7 enters STOP mode.
     */
    //periphConfig.policy = RDC_DISABLE_M7_ACCESS;
    //periphConfig.periph = kRDC_Periph_SAI3;
    //RDC_SetPeriphAccessConfig(RDC, &periphConfig);
}

#if TEST_FREQ_SWITCH
void LPM_SysTickHandler( void )
{
    /* The SysTick runs at the lowest interrupt priority, so when this interrupt
    executes all interrupts must be unmasked.  There is therefore no need to
    save and then restore the interrupt mask value as its value is already
    known. */
    portDISABLE_INTERRUPTS();
    {
        if (1 == m4_chg_clk_flag) {
            // Change to 24M
            LPM_MCORE_ChangeM7Clock(LPM_M7_LOW_FREQ);
            SystemCoreClockUpdate();
            vPortSetupTimerInterrupt();
            m4_chg_clk_flag = 0;
        } else if (2 == m4_chg_clk_flag) {
            // Change to 400M
            LPM_MCORE_ChangeM7Clock(LPM_M7_HIGH_FREQ);
            SystemCoreClockUpdate();
            vPortSetupTimerInterrupt();
            m4_chg_clk_flag = 0;
        }

        /* Increment the RTOS tick. */
        if( xTaskIncrementTick() != pdFALSE )
        {
            /* A context switch is required.  Context switching is performed in
            the PendSV interrupt.  Pend the PendSV interrupt. */
            portNVIC_INT_CTRL_REG = portNVIC_PENDSVSET_BIT;
        }
    }
	portENABLE_INTERRUPTS();
}
#endif

static uint8_t show_menu(void)
{
    PRINTF("\r\n==== LOW POWER DEMO Menu ====\n\r\n");
    SystemCoreClockUpdate();
    PRINTF("Core Freq: %d\r\n", SystemCoreClock);
    PRINTF("\r\nSelect the desired operation \n\r\n");
    PRINTF("Press 1 for enter: WAIT     - Wait mode\r\n");
    PRINTF("Press 2 for enter: STOP     - Stop mode\r\n");
    PRINTF("Press 3 for wakeup A core by MU GIR INT\r\n");
    PRINTF("Press 4 for wakeup A core by MU TF INT\r\n");
#if TEST_FREQ_SWITCH
    PRINTF("Press 5 for switch M7 clock to 24M\r\n");
    PRINTF("Press 6 for switch M7 clock back to 400M\r\n");
    PRINTF("Press 7 to test vTaskDelay for 20s\r\n");
#endif
#if TEST_REINIT_SYSPLL
    PRINTF("Press 8 for reinit syspll1\r\n");
    PRINTF("Press 9 for disable syspll1\r\n");
#endif
    PRINTF("\r\nWaiting for power mode select..\r\n\r\n");

    /* Wait for user response */
    return GETCHAR();
}

#if USE_WAKEUP_TIMER
static void APP_SetWakeupConfig(uint8_t wakeup_sec)
{
    TimerHandle_t wuTimerHandle = NULL;

    /* Create the software timer. */
    wuTimerHandle = xTimerCreate("WakeUpTimer",          /* Text name. */
                                 wakeup_sec * 1000,      /* Timer period. */
                                 pdFALSE,                /* Disable auto reload. */
                                 0,                      /* ID is not used. */
                                 wakeupTimerCallback);   /* The callback function. */
    /* Start timer. */
    xTimerStart(wuTimerHandle, 0);
}

static int32_t APP_GetWakeupTimeout(void)
{
    uint8_t timeout;

    while (1)
    {
        PRINTF("Select the wake up timeout in seconds.\r\n");
        PRINTF("The allowed range is 1s ~ 9s.\r\n");
        PRINTF("Enter 0 will keep in low power mode without wakeup.\r\n");
        PRINTF("Eg. enter 5 to wake up in 5 seconds.\r\n");
        PRINTF("\r\nWaiting for input timeout value...\r\n\r\n");

        timeout = GETCHAR();
        PRINTF("%c\r\n", timeout);
        if ((timeout >= '0') && (timeout <= '9'))
        {
            return timeout - '0';
        }
        else
        {
            return -1;
        }
    }
}

static int32_t APP_GetWakeupConfig(void)
{
    uint8_t wakeupTimeout = 0;

    /* Wakeup source is LPIT, user should input wakeup timeout value. */
    wakeupTimeout = APP_GetWakeupTimeout();
    if (wakeupTimeout > 0)
    {
        PRINTF("Will wakeup in %d seconds.\r\n", wakeupTimeout);
    }
    else if (0 == wakeupTimeout)
    {
        PRINTF("Will stay without wakeup forever.\r\n");
    }
    else
    {
        PRINTF("Wrong value!\r\n");
    }

    return wakeupTimeout;
}
#endif

#if USE_WAKEUP_GPIO
static void APP_InitWakeupGPIO(void)
{
    gpio_pin_config_t gpioConfig = {kGPIO_DigitalInput, 0, kGPIO_IntHighLevel};
    IOMUXC_SetPinMux(IOMUXC_GPIO1_IO13_GPIO1_IO13, 0U);
    IOMUXC_SetPinConfig(IOMUXC_GPIO1_IO13_GPIO1_IO13, IOMUXC_SW_PAD_CTL_PAD_DSE(6U) | IOMUXC_SW_PAD_CTL_PAD_FSEL(2U) | IOMUXC_SW_PAD_CTL_PAD_ODE_MASK | IOMUXC_SW_PAD_CTL_PAD_HYS_MASK); 
    GPIO_PinInit(GPIO1, 13, &gpioConfig);
    GPIO_PortClearInterruptFlags(GPIO1, 1U << 13);
    GPIO_PortEnableInterrupts(GPIO1, 1U << 13);
    EnableIRQ(GPIO1_Combined_0_15_IRQn);
    GPC_EnableIRQ(BOARD_GPC_BASEADDR, GPIO1_Combined_0_15_IRQn);
}
#endif

/*!
 * @brief Main function
 */
int main(void)
{
    int32_t i = 0;

    /* Init board hardware. */
    /* M7 has its local cache and enabled by default,
     * need to set smart subsystems (0x28000000 ~ 0x3FFFFFFF)
     * non-cacheable before accessing this address region */
    BOARD_InitMemory();

    /* Board specific RDC settings */
    BOARD_RdcInit();
    Peripheral_RdcSetting();
    BOARD_InitPins();
    BOARD_BootClockRUN();
    BOARD_InitDebugConsole();

    MU_Init(MUB);

    //LPM_Init();

    /*
     * In order to wakeup M7 from LPM, all PLLCTRLs need to be set to "NeededRun"
     */
    for (i = 0; i < 39; i++)
    {
        CCM->PLL_CTRL[i].PLL_CTRL = kCLOCK_ClockNeededRun;
    }

    PRINTF("\r\n####################  LOW POWER DEMO ####################\n\r\n");
    PRINTF("    Build Time: %s--%s \r\n", __DATE__, __TIME__);
    PRINTF("    Core Freq: %d\r\n", SystemCoreClock);

    xWuSemaphore = xSemaphoreCreateBinary();

#if TEST_FREQ_SWITCH
    xDelayTestSemaphore = xSemaphoreCreateBinary();
    xTimeoutSemaphore = xSemaphoreCreateBinary();

    /*Create tickless task*/
    if (xTaskCreate(Tickless_task, "Tickless_task", configMINIMAL_STACK_SIZE + 100, NULL, tickless_task_PRIORITY,
                    NULL) != pdPASS)
    {
        PRINTF("Task creation failed!.\r\n");
        while (1)
            ;
    }
#endif

    if (xTaskCreate(menu_task, "menu_task", configMINIMAL_STACK_SIZE + 100, NULL, menu_task_PRIORITY, NULL) !=
        pdPASS)
    {
        PRINTF("Task creation failed!.\r\n");
        while (1)
            ;
    }
    vTaskStartScheduler();
    for (;;)
        ;
}

#if TEST_FREQ_SWITCH
/* Tickless Task */
static void Tickless_task(void *pvParameters)
{
    static TickType_t init_tick_cnt = 0;

    for (;;)
    {
        if (xSemaphoreTake(xDelayTestSemaphore, portMAX_DELAY) == pdTRUE)
        {
            PRINTF("Start vTaskDelay Test\r\n");
            init_tick_cnt = xTaskGetTickCount();
            for (;;)
            {
                if (m4_vdelay_test) {
                    TickType_t tick_cnt = xTaskGetTickCount();
                    PRINTF("%d\r\n", tick_cnt - init_tick_cnt);
                    vTaskDelay(TIME_DELAY_SLEEP);
                    if ((tick_cnt - init_tick_cnt) > 4 * TIME_DELAY_SLEEP) {
                        m4_vdelay_test = 0;
                    }
                } else {
                    xSemaphoreGive(xTimeoutSemaphore);
                    break;
                }
            }
        }
    }
}
#endif

/*!
 * @brief Task responsible for printing of "Hello world." message.
 */
static void menu_task(void *pvParameters)
{
    while (1)
    {
        uint8_t ch_input;

        ch_input = show_menu();
        PRINTF("Input: %c\r\n", ch_input);
        switch (ch_input) {
        case '1':
            {
                // Wait
                LPM_MCORE_SetLpmMode(LPM_M7_STATE_WAIT);
#if USE_WAKEUP_TIMER
                uint8_t wakeup_timeout = 0;

                wakeup_timeout = APP_GetWakeupConfig();
                if (wakeup_timeout > 0)
                {
                    APP_SetWakeupConfig(wakeup_timeout);
                }
#endif

#if USE_WAKEUP_GPIO
                APP_InitWakeupGPIO();
#endif

                if (xSemaphoreTake(xWuSemaphore, portMAX_DELAY) == pdTRUE)
                    ;

                LPM_MCORE_SetLpmMode(LPM_M7_STATE_STOP);
            }
            break;
        case '2':
            {
                // Stop
                LPM_MCORE_SetLpmMode(LPM_M7_STATE_STOP);
#if USE_WAKEUP_TIMER
                uint8_t wakeup_timeout = 0;

                wakeup_timeout = APP_GetWakeupConfig();
                if (wakeup_timeout > 0)
                {
                    APP_SetWakeupConfig(wakeup_timeout);
                }
#endif

#if USE_WAKEUP_GPIO
                APP_InitWakeupGPIO();
#endif

                if (xSemaphoreTake(xWuSemaphore, portMAX_DELAY) == pdTRUE)
                    ;
            }
            break;
        case '3':
            {
                // Wakeup by GIR
                MU_TriggerInterrupts(MUB, kMU_GenInt0InterruptTrigger);
                PRINTF("Set mu interrupt kMU_GenInt0InterruptTrigger\r\n");
            }
            break;
        case '4':
            {
                // Wakeup by TF
                MU_SendMsg(MUB, 1, 2);
                PRINTF("set mu interrupt MU_SendMsg\r\n");
            }
            break;
#if TEST_FREQ_SWITCH
        case '5':
            // Switch M7 clock to 24M
            // Clock will be updated in xPortSysTickHandler
            m4_chg_clk_flag = 1;
            while (m4_chg_clk_flag)
                ;
            PRINTF("Clock changed to %d\r\n", SystemCoreClock);
            break;
        case '6':
            // Switch M7 clock back to 400M
            // Clock will be updated in xPortSysTickHandler
            m4_chg_clk_flag = 2;
            while (m4_chg_clk_flag)
                ;
            PRINTF("Clock changed to %d\r\n", SystemCoreClock);
            break;
        case '7':
            // Test vDelay
            {
                TaskHandle_t h_tickless_task;
                /*Create tickless task*/
                if (xTaskCreate(Tickless_task, "Tickless_task", configMINIMAL_STACK_SIZE + 100, NULL, tickless_task_PRIORITY,
                                &h_tickless_task) != pdPASS)
                {
                    PRINTF("Task creation failed!.\r\n");
                    while (1)
                        ;
                }
                PRINTF("\r\nTick count :\r\n");
                m4_vdelay_test = 1;
                xSemaphoreGive(xDelayTestSemaphore);
                if (xSemaphoreTake(xTimeoutSemaphore, portMAX_DELAY) == pdTRUE)
                {
                    vTaskDelete(h_tickless_task);
                }
            }
            break;
#endif

#if TEST_REINIT_SYSPLL
        case '8':
            if (CLOCK_IsPllBypassed(CCM_ANALOG, kCLOCK_SysPll1InternalPll1BypassCtrl))
            {
                /* A core enter suspend, syspll1 is disabled, enable it for test */
                PRINTF("Enable sys pll1\r\n");
                BOARD_EnableSysPLL1(true);
            }
            else
            {
                PRINTF("sys pll1 not bypassed, no need to enable again\r\n");
            }
            break;
        case '9':
            if (!CLOCK_IsPllBypassed(CCM_ANALOG, kCLOCK_SysPll1InternalPll1BypassCtrl))
            {
                PRINTF("Disable sys pll1\r\n");
                BOARD_EnableSysPLL1(false);
            }
            else
            {
                PRINTF("sys pll1 not enabled, no need to disable again\r\n");
            }
            break;
#endif
        default:
            break;
        }
    }
}
