/*
 * Copyright (c) 2018-2021 Arm Limited. All rights reserved.
 * Copyright 2019-2022 NXP. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef __FLASH_LAYOUT_H__
#define __FLASH_LAYOUT_H__

/* To use the flash memory, change the settings of the jumpers listed in the "LPC55S36-EVK Board User Manual", Table 21 from default settings to octal flash. */
//#define NS_QSPI_FLASH  // NS in QSPI Flash.
//#define PS_QSPI_FLASH  // PS in QSPI Flash.

/* Flash layout on LPC55S36 (without BL2):
 *
 * 0x0000_0000 Primary image area(224 kB):
 *    0x0000_0000 Secure     image (96 kB) 
 *    0x0001_8000 Non-secure image (128 kB)
 * Reserved area:
 * 0xXXXX_XXXX Secure Binary tail Area (4 KB), if SB is used.
 * 0xXXXX_XXXX Internal Trusted Storage Area (3 KB)
 * 0xXXXX_XXXX OTP area (2 KB)
 * 0xXXXX_XXXX Protected Storage Area (3 KB)
 * 0xXXXX_XXXX Unused
 */

/* This header file is included from linker scatter file as well, where only a
 * limited C constructs are allowed. Therefore it is not possible to include
 * here the platform_base_address.h to access flash related defines. To resolve
 * this some of the values are redefined here with different names, these are
 * marked with comment.
 */

#define FLASH_PARTITION_SIZE                     (224 * 1024)    /* On-chip Flash partition: 224 kB*/

/* Size of a Secure and of a Non-secure image */
#ifdef NS_QSPI_FLASH /* NS in QSPI Flash, S in On-chip Flash  */
	#define FLASH_S_PARTITION_SIZE               (FLASH_PARTITION_SIZE) /* S partition: 224 kB*/
	#define FLASH_NS_PARTITION_SIZE              (256 * 1024)           /* NS partition: 256 kB*/
#else /* S and NS in On-chip Flash */

#ifndef FLASH_S_PARTITION_SIZE
	#define FLASH_S_PARTITION_SIZE               (96 * 1024)    /* S partition: 96 kB, NS partition: 128 kB. Must be aligned to 32kB*/
#endif
    #define FLASH_NS_PARTITION_SIZE              (FLASH_PARTITION_SIZE - FLASH_S_PARTITION_SIZE)    /* NS partition*/

#endif

/* Sector size of the embedded flash hardware (erase/program) */
#define FLASH_AREA_IMAGE_SECTOR_SIZE        (512)           /* 512 B. Flash memory program/erase operations have a page granularity. */

/* FLASH size */
#define FLASH_TOTAL_SIZE                    (0x00040000-(24*FLASH_AREA_IMAGE_SECTOR_SIZE)) /* 244 KB. The last 24 pages (12KB) are reserved for PFR on the 256KB flash. PRINCE covers maximum 240KB.*/

/* Sector size of the QSPI flash hardware */
#define QSPI_FLASH_AREA_IMAGE_SECTOR_SIZE   (0x1000)        /* 4 KB */
#define QSPI_FLASH_AREA_PROGRAM_SIZE        (256)
#define QSPI_FLASH_TOTAL_SIZE               (0x4000000)     /* 64 MB */

#ifdef SB_FILE /* Use signed Secure Binary (SB) image */
#define FLASH_SB_TAIL   0x1000 /* 4 KB, Signed image, reserved for Certificate block. */
#else
#define FLASH_SB_TAIL   0x0    /* 0 KB, Plain Image with or without CRC. */
#endif

/* Secure + Non-secure image primary slot */
#define FLASH_AREA_0_OFFSET             (0x0)
#define FLASH_AREA_0_SIZE               (FLASH_S_PARTITION_SIZE + \
                                        FLASH_NS_PARTITION_SIZE + \
                                        FLASH_SB_TAIL)

/* Internal Trusted Storage (ITS) Service definitions */
#define FLASH_ITS_AREA_OFFSET           (FLASH_AREA_0_OFFSET + FLASH_AREA_0_SIZE)
#define FLASH_ITS_AREA_SIZE             (0xC00) /* 3 KB */

/* OTP_definitions */
#define FLASH_OTP_NV_COUNTERS_AREA_OFFSET (FLASH_ITS_AREA_OFFSET + \
                                           FLASH_ITS_AREA_SIZE)
#define FLASH_OTP_NV_COUNTERS_AREA_SIZE   (FLASH_AREA_IMAGE_SECTOR_SIZE * 4)
#define FLASH_OTP_NV_COUNTERS_SECTOR_SIZE FLASH_AREA_IMAGE_SECTOR_SIZE

/* Offset and size definition in flash area used by assemble.py */
#define SECURE_IMAGE_OFFSET             (0x0)
#define SECURE_IMAGE_MAX_SIZE           FLASH_S_PARTITION_SIZE

#define NON_SECURE_IMAGE_OFFSET         (SECURE_IMAGE_OFFSET + \
                                         SECURE_IMAGE_MAX_SIZE)
#define NON_SECURE_IMAGE_MAX_SIZE       FLASH_NS_PARTITION_SIZE

/* Smallest flash programmable unit in bytes */
#define TFM_HAL_FLASH_PROGRAM_UNIT          FLASH_AREA_IMAGE_SECTOR_SIZE

/* Protected Storage (PS) Service definitions
 */
#ifdef PS_QSPI_FLASH /* PS in QSPI Flash */
    #define FLASH_PS_AREA_SIZE              (0x8000) /* 32 KB */
    
    #define TFM_HAL_PS_FLASH_DRIVER         Driver_QSPI_FLASH0

    /* Base address of dedicated flash area for PS */
    #define TFM_HAL_PS_FLASH_AREA_ADDR      0x0

    /* Number of physical erase sectors per logical FS block */
    #define TFM_HAL_PS_SECTORS_PER_BLOCK    1

    /* Smallest flash programmable unit in bytes */
    #define TFM_HAL_PS_PROGRAM_UNIT         QSPI_FLASH_AREA_PROGRAM_SIZE
    #define PS_FLASH_NAND_BUF_SIZE          (QSPI_FLASH_AREA_IMAGE_SECTOR_SIZE * \
                                             TFM_HAL_PS_SECTORS_PER_BLOCK)
#else  /* PS in On-chip Flash */
    #define FLASH_PS_AREA_SIZE              (0xC00) /* 3 KB */

    #define TFM_HAL_PS_FLASH_DRIVER         Driver_EFLASH0

    /* Base address of dedicated flash area for PS */
    #define TFM_HAL_PS_FLASH_AREA_ADDR      (FLASH_OTP_NV_COUNTERS_AREA_OFFSET + \
                                            FLASH_OTP_NV_COUNTERS_AREA_SIZE)

    /* Number of physical erase sectors per logical FS block */
    #define TFM_HAL_PS_SECTORS_PER_BLOCK    3

    /* Smallest flash programmable unit in bytes */
    #define TFM_HAL_PS_PROGRAM_UNIT         FLASH_AREA_IMAGE_SECTOR_SIZE
    #define PS_FLASH_NAND_BUF_SIZE          (FLASH_AREA_IMAGE_SECTOR_SIZE * \
                                            TFM_HAL_PS_SECTORS_PER_BLOCK)
#endif

/* Size of dedicated flash area for PS */
#define TFM_HAL_PS_FLASH_AREA_SIZE      FLASH_PS_AREA_SIZE
#define PS_RAM_FS_SIZE                  TFM_HAL_PS_FLASH_AREA_SIZE

/* Internal Trusted Storage (ITS) Service definitions
 * Note: Further documentation of these definitions can be found in the
 * TF-M ITS Integration Guide. The ITS should be in the internal flash, but is
 * allocated in the external flash just for development platforms that don't
 * have internal flash available.
 */
#define TFM_HAL_ITS_FLASH_DRIVER        Driver_EFLASH0

/* In this target the CMSIS driver requires only the offset from the base
 * address instead of the full memory address.
 */
/* Base address of dedicated flash area for ITS */
#define TFM_HAL_ITS_FLASH_AREA_ADDR     FLASH_ITS_AREA_OFFSET
/* Size of dedicated flash area for ITS */
#define TFM_HAL_ITS_FLASH_AREA_SIZE     FLASH_ITS_AREA_SIZE
#define ITS_RAM_FS_SIZE                 TFM_HAL_ITS_FLASH_AREA_SIZE
/* Number of physical erase sectors per logical FS block */
#define TFM_HAL_ITS_SECTORS_PER_BLOCK   3

/* Smallest flash programmable unit in bytes */
#define TFM_HAL_ITS_PROGRAM_UNIT        FLASH_AREA_IMAGE_SECTOR_SIZE
#define ITS_FLASH_NAND_BUF_SIZE         (FLASH_AREA_IMAGE_SECTOR_SIZE * \
                                        TFM_HAL_ITS_SECTORS_PER_BLOCK)

/* OTP / NV counter definitions */
#define TFM_OTP_NV_COUNTERS_AREA_SIZE   (FLASH_OTP_NV_COUNTERS_AREA_SIZE / 2)
#define TFM_OTP_NV_COUNTERS_AREA_ADDR   FLASH_OTP_NV_COUNTERS_AREA_OFFSET
#define TFM_OTP_NV_COUNTERS_SECTOR_SIZE FLASH_OTP_NV_COUNTERS_SECTOR_SIZE
#define TFM_OTP_NV_COUNTERS_BACKUP_AREA_ADDR (TFM_OTP_NV_COUNTERS_AREA_ADDR + \
                                              TFM_OTP_NV_COUNTERS_AREA_SIZE)

/* Use Flash memory to store Code data */
#define S_ROM_ALIAS_BASE    (0x10000000)
#ifdef NS_QSPI_FLASH /* NS in QSPI Flash */
    #define NS_ROM_ALIAS_BASE   (0x08000000)
#else /* NS in On-chip Flash */
    #define NS_ROM_ALIAS_BASE   (0x00000000)
#endif

/* Use SRAM[0-4] memory to store RW data */
#define S_RAM_ALIAS_BASE    (0x30000000)
#define NS_RAM_ALIAS_BASE   (0x20000000)

#define RESERVED_RAM_SIZE   (0x00004000)  /* Reserved SRAM0(16KB): 8KB Retention RAM. 8KB For PKC */

#define TOTAL_ROM_SIZE 		FLASH_TOTAL_SIZE
#define TOTAL_RAM_SIZE      (0x0001C000 - RESERVED_RAM_SIZE)     /* 112 KB RAM for data minus reserverd (also, without 16KB SRAM X for code)*/

#endif /* __FLASH_LAYOUT_H__ */