/*--------------------------------------------------------------------------*/
/* Copyright 2022-2023 NXP                                                  */
/*                                                                          */
/* NXP Confidential. This software is owned or controlled by NXP and may    */
/* only be used strictly in accordance with the applicable license terms.   */
/* By expressly accepting such terms or by downloading, installing,         */
/* activating and/or otherwise using the software, you are agreeing that    */
/* you have read, and that you agree to comply with and are bound by, such  */
/* license terms. If you do not agree to be bound by the applicable license */
/* terms, then you may not retain, install, activate or otherwise use the   */
/* software.                                                                */
/*--------------------------------------------------------------------------*/

#ifndef MCUXCLCORE_ANALYSIS_H_
#define MCUXCLCORE_ANALYSIS_H_

#define MCUXCLCORE_ANALYSIS_STR(a) #a
#define MCUXCLCORE_ANALYSIS_EMPTY()
#define MCUXCLCORE_ANALYSIS_DEFER(id) id MCUXCLCORE_ANALYSIS_EMPTY()
#define MCUXCLCORE_ANALYSIS_EXPAND(...) __VA_ARGS__

#define MCUXCLCORE_ANALYSIS_PRAGMA(x) _Pragma(#x)

/* Example of common patterns, with either just predefined rationale, or a combination of discards. */
#define MCUXCLCORE_ANALYSIS_START_PATTERN_HW_READ() \
  MCUXCLCORE_ANALYSIS_START_SUPPRESS_DISCARD_VOLATILE("Read from a HW peripheral")
#define MCUXCLCORE_ANALYSIS_STOP_PATTERN_HW_READ() \
  MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_DISCARD_VOLATILE()

#define MCUXCLCORE_ANALYSIS_START_PATTERN_HW_WRITE() \
  MCUXCLCORE_ANALYSIS_START_SUPPRESS_DISCARD_VOLATILE("Write to a HW peripheral")
  /*MCUXCLCORE_ANALYSIS_START_SUPPRESS_CAST_TO_OBJECT("Write to a HW peripheral")*/
#define MCUXCLCORE_ANALYSIS_STOP_PATTERN_HW_WRITE() \
  /*MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_CAST_TO_OBJECT()*/ \
  MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_DISCARD_VOLATILE()

#define MCUXCLCORE_ANALYSIS_START_PATTERN_DESCRIPTIVE_IDENTIFIER() \
  MCUXCLCORE_ANALYSIS_START_SUPPRESS_AMBIGUOUS_IDENTIFIER("Identifiers longer than 31 characters are allowed for more descriptive naming")
#define MCUXCLCORE_ANALYSIS_STOP_PATTERN_DESCRIPTIVE_IDENTIFIER() \
  MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_AMBIGUOUS_IDENTIFIER()

#define MCUXCLCORE_ANALYSIS_START_PATTERN_REINTERPRET_MEMORY_OF_OPAQUE_TYPES() \
  MCUXCLCORE_ANALYSIS_START_SUPPRESS_REINTERPRET_MEMORY("explicit pointer casts reinterpreting opaque types of workarea-like buffer objects are allowed.")
#define MCUXCLCORE_ANALYSIS_STOP_PATTERN_REINTERPRET_MEMORY() \
  MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_REINTERPRET_MEMORY()
/* Rule 11.3: applies to casts between ctx structs
* e.g. cast from mcuxClAead_Context_t to more specific type mcuxClAeadModes_Context_t
*/
#define MCUXCLCORE_ANALYSIS_START_CAST_TO_MORE_SPECIFIC_TYPE() \
  MCUXCLCORE_ANALYSIS_START_SUPPRESS_REINTERPRET_MEMORY("Cast to a more specific type is allowed")
#define MCUXCLCORE_ANALYSIS_STOP_CAST_TO_MORE_SPECIFIC_TYPE() \
  MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_REINTERPRET_MEMORY()

#define MCUXCLCORE_ANALYSIS_START_PATTERN_EXTERNAL_HEADER() \
  MCUXCLCORE_ANALYSIS_START_SUPPRESS_C11_EXTENSION("External header outside our control") \
  MCUXCLCORE_ANALYSIS_START_SUPPRESS_UNDEFINED_VALUE("External header outside our control") \
  MCUXCLCORE_ANALYSIS_START_SUPPRESS_RESERVED_MACRO_IDENTIFIER("External header outside our control") \
  MCUXCLCORE_ANALYSIS_START_SUPPRESS_PADDED_TO_ALIGNMENT_BOUNDARY("External header outside our control")
#define MCUXCLCORE_ANALYSIS_STOP_PATTERN_EXTERNAL_HEADER() \
  MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_PADDED_TO_ALIGNMENT_BOUNDARY() \
  MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_RESERVED_MACRO_IDENTIFIER() \
  MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_UNDEFINED_VALUE() \
  MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_C11_EXTENSION()

#define MCUXCLCORE_ANALYSIS_START_PATTERN_OBJ_SIZES() \
  MCUXCLCORE_ANALYSIS_START_SUPPRESS_UNUSED_VARIABLE("Variables used to determine object sizes") \
  MCUXCLCORE_ANALYSIS_START_SUPPRESS_MISSING_VARIABLE_DECLARATION("Variables used to determine object sizes")
#define MCUXCLCORE_ANALYSIS_STOP_PATTERN_OBJ_SIZES() \
  MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_MISSING_VARIABLE_DECLARATION() \
  MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_UNUSED_VARIABLE()

#define MCUXCLCORE_ANALYSIS_START_PATTERN_FUP() \
  MCUXCLCORE_ANALYSIS_START_SUPPRESS_MISSING_VARIABLE_DECLARATION("External declarations are generated by the FUP processing tool")
#define MCUXCLCORE_ANALYSIS_STOP_PATTERN_FUP() \
  MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_MISSING_VARIABLE_DECLARATION()

/* Example of basic violation suppression */
#define MCUXCLCORE_ANALYSIS_START_SUPPRESS_DEAD_CODE(rationale) \
  MCUXCLCORE_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_2_1, rationale)
#define MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_DEAD_CODE() \
  MCUXCLCORE_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_2_1)

#define MCUXCLCORE_ANALYSIS_START_SUPPRESS_TEXT_IN_COMMENTS(rationale) \
  MCUXCLCORE_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_3_1, rationale)
#define MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_TEXT_IN_COMMENTS() \
  MCUXCLCORE_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_3_1)

#define MCUXCLCORE_ANALYSIS_START_SUPPRESS_POINTER_CASTING(rationale) \
  MCUXCLCORE_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_11_3, rationale)
#define MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_POINTER_CASTING() \
  MCUXCLCORE_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_11_3)

#define MCUXCLCORE_ANALYSIS_START_SUPPRESS_DISCARD_VOLATILE(rationale) \
  MCUXCLCORE_ANALYSIS_CLANG_START_SUPPRESS_WARNING(-Wcast-qual, rationale) \
  MCUXCLCORE_ANALYSIS_GHS_START_SUPPRESS_WARNING(1836, rationale) \
  MCUXCLCORE_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_11_8, rationale)
#define MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_DISCARD_VOLATILE() \
  MCUXCLCORE_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_11_8) \
  MCUXCLCORE_ANALYSIS_GHS_STOP_SUPPRESS_WARNING(1836) \
  MCUXCLCORE_ANALYSIS_CLANG_STOP_SUPPRESS_WARNING(-Wcast-qual)

#define MCUXCLCORE_ANALYSIS_START_SUPPRESS_OUT_OF_BOUNDS_ACCESS(rationale) \
  MCUXCLCORE_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_18_1, rationale)
#define MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_OUT_OF_BOUNDS_ACCESS() \
  MCUXCLCORE_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_18_1) \

#define MCUXCLCORE_ANALYSIS_START_SUPPRESS_DISCARD_CONST(rationale) \
  MCUXCLCORE_ANALYSIS_CLANG_START_SUPPRESS_WARNING(-Wcast-qual, rationale) \
  MCUXCLCORE_ANALYSIS_GHS_START_SUPPRESS_WARNING(1836, rationale) \
  MCUXCLCORE_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_11_8, rationale)
#define MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_DISCARD_CONST() \
  MCUXCLCORE_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_11_8) \
  MCUXCLCORE_ANALYSIS_GHS_STOP_SUPPRESS_WARNING(1836) \
  MCUXCLCORE_ANALYSIS_CLANG_STOP_SUPPRESS_WARNING(-Wcast-qual)

#define MCUXCLCORE_ANALYSIS_START_SUPPRESS_DISCARD_CONST_QUALIFIER(rationale) \
  MCUXCLCORE_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_11_8, rationale)
#define MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_DISCARD_CONST_QUALIFIER() \
  MCUXCLCORE_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_11_8)

#define MCUXCLCORE_ANALYSIS_START_SUPPRESS_REINTERPRET_MEMORY_BETWEEN_INAPT_ESSENTIAL_TYPES(rationale) \
  MCUXCLCORE_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_11_3, rationale)
#define MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_REINTERPRET_MEMORY_BETWEEN_INAPT_ESSENTIAL_TYPES() \
  MCUXCLCORE_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_11_3)

#define MCUXCLCORE_ANALYSIS_START_SUPPRESS_REINTERPRET_MEMORY(rationale) \
  MCUXCLCORE_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_11_3, rationale)
#define MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_REINTERPRET_MEMORY() \
  MCUXCLCORE_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_11_3)

#define MCUXCLCORE_ANALYSIS_START_SUPPRESS_TYPECAST_INTEGER_TO_POINTER(rationale) \
  MCUXCLCORE_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_11_4, rationale) \
  MCUXCLCORE_ANALYSIS_COVERITY_START_DEVIATE(CERT_INT36_C, rationale) \
  MCUXCLCORE_ANALYSIS_COVERITY_START_DEVIATE(CERT_MSC15_C, rationale)
#define MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_TYPECAST_INTEGER_TO_POINTER() \
  MCUXCLCORE_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_MSC15_C) \
  MCUXCLCORE_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_INT36_C) \
  MCUXCLCORE_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_11_4)

#define MCUXCLCORE_ANALYSIS_START_SUPPRESS_AMBIGUOUS_IDENTIFIER(rationale) \
  MCUXCLCORE_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_5_1, rationale)
#define MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_AMBIGUOUS_IDENTIFIER() \
  MCUXCLCORE_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_5_1)

#define MCUXCLCORE_ANALYSIS_START_SUPPRESS_REINTERPRET_STRUCT(rationale) \
  MCUXCLCORE_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_11_1, rationale)
#define MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_REINTERPRET_STRUCT() \
  MCUXCLCORE_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_11_1)

#define MCUXCLCORE_ANALYSIS_START_SUPPRESS_DECLARED_BUT_NEVER_REFERENCED(rationale) \
  MCUXCLCORE_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_2_2, rationale)
#define MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_DECLARED_BUT_NEVER_REFERENCED() \
  MCUXCLCORE_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_2_2)

#define MCUXCLCORE_ANALYSIS_START_SUPPRESS_BOOLEAN_TYPE_FOR_CONDITIONAL_EXPRESSION() \
  MCUXCLCORE_ANALYSIS_COVERITY_START_FALSE_POSITIVE(MISRA_C_2012_Rule_14_4, "Conditional expression does have a boolean type.")
#define MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_BOOLEAN_TYPE_FOR_CONDITIONAL_EXPRESSION() \
  MCUXCLCORE_ANALYSIS_COVERITY_STOP_FALSE_POSITIVE(MISRA_C_2012_Rule_14_4)

#define MCUXCLCORE_ANALYSIS_START_SUPPRESS_C11_EXTENSION(rationale) \
  MCUXCLCORE_ANALYSIS_CLANG_START_SUPPRESS_WARNING(-Wc11-extensions, rationale)
#define MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_C11_EXTENSION() \
  MCUXCLCORE_ANALYSIS_CLANG_STOP_SUPPRESS_WARNING(-Wc11-extensions)

#define MCUXCLCORE_ANALYSIS_START_SUPPRESS_UNDEFINED_VALUE(rationale) \
  MCUXCLCORE_ANALYSIS_CLANG_START_SUPPRESS_WARNING(-Wundef, rationale)
#define MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_UNDEFINED_VALUE() \
  MCUXCLCORE_ANALYSIS_CLANG_STOP_SUPPRESS_WARNING(-Wundef)

#define MCUXCLCORE_ANALYSIS_START_SUPPRESS_RESERVED_IDENTIFIER(rationale) \
  MCUXCLCORE_ANALYSIS_CLANG_START_SUPPRESS_WARNING(-Wreserved-identifier, rationale)
#define MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_RESERVED_IDENTIFIER() \
  MCUXCLCORE_ANALYSIS_CLANG_STOP_SUPPRESS_WARNING(-Wreserved-identifier)

#define MCUXCLCORE_ANALYSIS_START_SUPPRESS_RESERVED_MACRO_IDENTIFIER(rationale) \
  MCUXCLCORE_ANALYSIS_CLANG_START_SUPPRESS_WARNING(-Wreserved-macro-identifier, rationale)
#define MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_RESERVED_MACRO_IDENTIFIER() \
  MCUXCLCORE_ANALYSIS_CLANG_STOP_SUPPRESS_WARNING(-Wreserved-macro-identifier)

#define MCUXCLCORE_ANALYSIS_START_SUPPRESS_MISSING_VARIABLE_DECLARATION(rationale) \
  MCUXCLCORE_ANALYSIS_CLANG_START_SUPPRESS_WARNING(-Wmissing-variable-declarations, rationale)
#define MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_MISSING_VARIABLE_DECLARATION() \
  MCUXCLCORE_ANALYSIS_CLANG_STOP_SUPPRESS_WARNING(-Wmissing-variable-declarations)

#define MCUXCLCORE_ANALYSIS_START_SUPPRESS_UNUSED_VARIABLE(rationale) \
  MCUXCLCORE_ANALYSIS_CLANG_START_SUPPRESS_WARNING(-Wunused-variable, rationale)
#define MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_UNUSED_VARIABLE() \
  MCUXCLCORE_ANALYSIS_CLANG_STOP_SUPPRESS_WARNING(-Wunused-variable)

#define MCUXCLCORE_ANALYSIS_START_SUPPRESS_PADDED_TO_ALIGNMENT_BOUNDARY(rationale) \
  MCUXCLCORE_ANALYSIS_CLANG_START_SUPPRESS_WARNING(-Wpadded, rationale)
#define MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_PADDED_TO_ALIGNMENT_BOUNDARY() \
  MCUXCLCORE_ANALYSIS_CLANG_STOP_SUPPRESS_WARNING(-Wpadded)

#define MCUXCLCORE_ANALYSIS_START_SUPPRESS_HARDWARE_ACCESS(rationale) \
  MCUXCLCORE_ANALYSIS_GCC_START_SUPPRESS_WARNING(-Warray-bounds, rationale)
#define MCUXCLCORE_ANALYSIS_STOP_SUPPRESS_HARDWARE_ACCESS() \
  MCUXCLCORE_ANALYSIS_GCC_STOP_SUPPRESS_WARNING(-Warray-bounds)

/* Tool specific handling: Coverity checkers */
#if defined(__COVERITY__)

#define MCUXCLCORE_ANALYSIS_COVERITY_START_DEVIATE(checker_identifier, rationale) \
  MCUXCLCORE_ANALYSIS_EXPAND(MCUXCLCORE_ANALYSIS_DEFER(MCUXCLCORE_ANALYSIS_PRAGMA)(coverity compliance block deviate checker_identifier rationale))
#define MCUXCLCORE_ANALYSIS_COVERITY_STOP_DEVIATE(checker_identifier) \
  MCUXCLCORE_ANALYSIS_EXPAND(MCUXCLCORE_ANALYSIS_DEFER(MCUXCLCORE_ANALYSIS_PRAGMA)(coverity compliance end_block checker_identifier))

#define MCUXCLCORE_ANALYSIS_COVERITY_START_FALSE_POSITIVE(checker_identifier, rationale) \
  MCUXCLCORE_ANALYSIS_EXPAND(MCUXCLCORE_ANALYSIS_DEFER(MCUXCLCORE_ANALYSIS_PRAGMA)(coverity compliance block fp checker_identifier rationale))
#define MCUXCLCORE_ANALYSIS_COVERITY_STOP_FALSE_POSITIVE(checker_identifier) \
  MCUXCLCORE_ANALYSIS_EXPAND(MCUXCLCORE_ANALYSIS_DEFER(MCUXCLCORE_ANALYSIS_PRAGMA)(coverity compliance end_block checker_identifier))

#else
#define MCUXCLCORE_ANALYSIS_COVERITY_START_DEVIATE(checker_identifier, rationale)
#define MCUXCLCORE_ANALYSIS_COVERITY_STOP_DEVIATE(checker_identifier)

#define MCUXCLCORE_ANALYSIS_COVERITY_START_FALSE_POSITIVE(checker_identifier, rationale)
#define MCUXCLCORE_ANALYSIS_COVERITY_STOP_FALSE_POSITIVE(checker_identifier)
#endif

/* Tool specific handling: Clang warnings */
#if defined(__clang__)
#define MCUXCLCORE_ANALYSIS_CLANG_START_SUPPRESS_WARNING(warning_identifier, rationale) \
  MCUXCLCORE_ANALYSIS_EXPAND(MCUXCLCORE_ANALYSIS_DEFER(MCUXCLCORE_ANALYSIS_PRAGMA)(clang diagnostic push)) \
  MCUXCLCORE_ANALYSIS_EXPAND(MCUXCLCORE_ANALYSIS_DEFER(MCUXCLCORE_ANALYSIS_PRAGMA)(clang diagnostic ignored MCUXCLCORE_ANALYSIS_STR(warning_identifier)))
#define MCUXCLCORE_ANALYSIS_CLANG_STOP_SUPPRESS_WARNING(warning_identifier) \
  MCUXCLCORE_ANALYSIS_EXPAND(MCUXCLCORE_ANALYSIS_DEFER(MCUXCLCORE_ANALYSIS_PRAGMA)(clang diagnostic pop))
#else
#define MCUXCLCORE_ANALYSIS_CLANG_START_SUPPRESS_WARNING(warning_identifier, rationale)
#define MCUXCLCORE_ANALYSIS_CLANG_STOP_SUPPRESS_WARNING(warning_identifier)
#endif

/* Tool specific handling: GHS warnings */
#if defined(__ghs__)
#define MCUXCLCORE_ANALYSIS_GHS_START_SUPPRESS_WARNING(warning_identifier, rationale) \
  _Pragma("ghs nowarning " ## warning_identifier)
#define MCUXCLCORE_ANALYSIS_GHS_STOP_SUPPRESS_WARNING(warning_identifier) \
  _Pragma("ghs endnowarning")
#else
#define MCUXCLCORE_ANALYSIS_GHS_START_SUPPRESS_WARNING(warning_identifier, rationale)
#define MCUXCLCORE_ANALYSIS_GHS_STOP_SUPPRESS_WARNING(warning_identifier)
#endif

/* Tool specific handling: GCC warnings */
#if defined(__GNUC__)
#define MCUXCLCORE_ANALYSIS_GCC_START_SUPPRESS_WARNING(warning_identifier, rationale)  \
  MCUXCLCORE_ANALYSIS_EXPAND(MCUXCLCORE_ANALYSIS_DEFER(MCUXCLCORE_ANALYSIS_PRAGMA)(GCC diagnostic push)) \
  MCUXCLCORE_ANALYSIS_EXPAND(MCUXCLCORE_ANALYSIS_DEFER(MCUXCLCORE_ANALYSIS_PRAGMA)(GCC diagnostic ignored MCUXCLCORE_ANALYSIS_STR(warning_identifier)))
#define MCUXCLCORE_ANALYSIS_GCC_STOP_SUPPRESS_WARNING(warning_identifier)  \
  MCUXCLCORE_ANALYSIS_EXPAND(MCUXCLCORE_ANALYSIS_DEFER(MCUXCLCORE_ANALYSIS_PRAGMA)(GCC diagnostic pop))
#else
#define MCUXCLCORE_ANALYSIS_GCC_START_SUPPRESS_WARNING(warning_identifier, rationale)
#define MCUXCLCORE_ANALYSIS_GCC_STOP_SUPPRESS_WARNING(warning_identifier)
#endif

#endif /* MCUXCLCORE_ANALYSIS_H_ */
