/*
 * Copyright 2021, 2024 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef _FSL_P3T11XX_H_
#define _FSL_P3T11XX_H_

#include "fsl_common.h"

/* P3T1175 and P3T1108 TEMP register addresses. */
#define TEMP               0x00
#define CONFIG             0x01
#define TEMP_LOW           0x02
#define TEMP_HIGH          0x03

#define P3T1755_CONFIG_DEFAULT     0x28
#define P3T1085_CONFIG_DEFAULT     0x2210

#define P3T1755_CONFIG_ENABLE_TM   0x2A         /* Enable TM, bit 2 */
#define P3T1085_CONFIG_ENABLE_TM   0x2610       /* Enable TM, bit 3 of MSB */

#define P3T1755_TEMP_LOW_DEFAULT   0x4B00
#define P3T1755_TEMP_HIGH_DEFAULT  0x5000

#define P3T1085_TEMP_LOW_DEFAULT   0xB500
#define P3T1085_TEMP_HIGH_DEFAULT  0x7FF0

#define P3T1755_TEMP_LOW           0x1400       /* 20 degree C */
//#define P3T1755_TEMP_HIGH          0x1900       /* 25 degree C */
//#define P3T1755_TEMP_HIGH          0x1B00       /* 28 degree C */
#define P3T1755_TEMP_HIGH          0x1E00       /* 30 degree C */

#define P3T1085_TEMP_LOW           0x1400       /* 20 degree C */
//#define P3T1085_TEMP_HIGH          0x1900       /* 25 degree C */
//#define P3T1085_TEMP_HIGH          0x1B00       /* 28 degree C */
#define P3T1085_TEMP_HIGH          0x1E00       /* 30 degree C */

typedef struct _p3t11xx_handle
{
    sensor_write_transfer_func_t Sensor_WriteTransfer;
    sensor_read_transfer_func_t Sensor_ReadTransfer;
    i3c_device_t *device_handle;
} p3t11xx_handle_t;

typedef struct _p3t11xx_config
{
    sensor_write_transfer_func_t Sensor_WriteTransfer;
    sensor_read_transfer_func_t Sensor_ReadTransfer;
    i3c_device_t *device_handle;
    bool isReset;
} p3t11xx_config_t;

#if defined(__cplusplus)
extern "C" {
#endif

/*!
 * @brief Write Register with register data buffer.
 *
 * @param handle The pointer to #p3t11xx_handle_t.
 * @param regAddress register address to write.
 * @param regData The pointer to data buffer to be write to the reg.
 * @param dataSize Size of the regData.
 *
 * @return kStatus_Success if success or kStatus_Fail if error.
 */
status_t P3T11XX_WriteReg(p3t11xx_handle_t *handle, uint32_t regAddress, uint8_t *regData, size_t dataSize);

/*!
 * @brief Read Register to speficied data buffer.
 *
 * @param handle The pointer to #icm42688p_handle_t.
 * @param regAddress register address to read.
 * @param regData The pointer to data buffer to store the read out data.
 * @param dataSize Size of the regData to be read.
 *
 * @return kStatus_Success if success or kStatus_Fail if error.
 */
status_t P3T11XX_ReadReg(p3t11xx_handle_t *handle, uint32_t regAddress, uint8_t *regData, size_t dataSize);

/*!
 * @brief Create handle for P3T11XX, reset the sensor per user configuration.
 *
 * @param p3t11xx_handle The pointer to #p3t11xx_handle_t.
 *
 * @return kStatus_Success if success or kStatus_Fail if error.
 */
status_t P3T1755_Init(p3t11xx_handle_t *handle, p3t11xx_config_t *config);

/*!
 * @brief Create handle for P3T11XX, reset the sensor per user configuration.
 *
 * @param p3t11xx_handle The pointer to #p3t11xx_handle_t.
 *
 * @return kStatus_Success if success or kStatus_Fail if error.
 */
status_t P3T1085_Init(p3t11xx_handle_t *handle, p3t11xx_config_t *config);

/*!
 * @brief Read sensor data from fifo.
 *
 * @param handle The pointer to #p3t11xx_handle_t.
 * @param uint32_t_t The pointer to which stores the read out data.
 *
 * @return kStatus_Success if success or kStatus_Fail if error.
 */
status_t P3T1755_ReadSensorData(p3t11xx_handle_t *handle, uint32_t *sensorData);

/*!
 * @brief Read sensor data from fifo.
 *
 * @param handle The pointer to #p3t11xx_handle_t.
 * @param uint32_t_t The pointer to which stores the read out data.
 *
 * @return kStatus_Success if success or kStatus_Fail if error.
 */
status_t P3T1085_ReadSensorData(p3t11xx_handle_t *handle, uint32_t *sensorData);

#if defined(__cplusplus)
}
#endif /* __cplusplus */

#endif /* _FSL_P3T11XX_H_ */
