/*
 * Copyright 2021 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */
#include "PL_platformTypes_HIFI4_FUSIONF1.h"
#include "vit_model.h"
#include "VIT.h"
#include "vit_glue.h"

#include "fsl_debug_console.h"
#undef DSP_PRINTF
#define DSP_PRINTF PRINTF

/*
*   Useful Definitions : not to be changed
*/
// MEMORY
#define MEMORY_ALIGNMENT             8     // in bytes
#define VIT_OPERATING_MODE          (VIT_WAKEWORD_ENABLE | VIT_VOICECMD_ENABLE)
#define NUMBER_OF_CHANNELS          1
#define DEVICE_ID                   VIT_IMXRT500

static VIT_Handle_t              VITHandle = PL_NULL;                      // VIT handle pointer
static VIT_InstanceParams_st     VITInstParams;                            // VIT instance parameters structure
static VIT_ControlParams_st      VITControlParams;                         // VIT control parameters structure
static PL_MemoryTable_st         VITMemoryTable;                           // VIT memory table descriptor
static PL_BOOL                   InitPhase_Error = PL_FALSE;
static VIT_DataIn_st             VIT_InputBuffers = { PL_NULL, PL_NULL, PL_NULL };  // Resetting Input Buffer addresses provided to VIT_process() API
static PL_INT8                   *pMemory[PL_NR_MEMORY_REGIONS];

static PL_UINT8					 *pMem;		 //Used for storing start address of whole allocation of space for model
static PL_UINT8 				 *VIT_Model; //Used to hold aligned start address of model


#define XA_VIT_PRE_PROC_FLAG_PREINIT_DONE      (1 << 0)
#define XA_VIT_PRE_PROC_FLAG_POSTINIT_DONE     (1 << 1)
#define XA_VIT_PRE_PROC_FLAG_RUNNING           (1 << 2)
#define XA_VIT_PRE_PROC_FLAG_OUTPUT            (1 << 3)
#define XA_VIT_PRE_PROC_FLAG_COMPLETE          (1 << 4)

int32_t xa_vit_pre_proc_do_execute_16bit(vit_pre_proc_t *d)
{
    WORD32    k, i, nSize;
    WORD16    *pIn = (WORD16 *) d->input;
    UWORD32   filled = d->input_avail;
    WORD16    input, output;
    VIT_ReturnStatus_en    VIT_Status;
    VIT_VoiceCommand_st     VoiceCommand;                             // Voice Command id
    VIT_DetectionStatus_en VIT_DetectionResults = VIT_NO_DETECTION;  // VIT detection result

    k = (WORD32)(d->buffer_size - filled);

//    DSP_PRINTF("xa_vit_pre_proc_do_execute_16bit 0x%x, 0x%x, filled = %d\n", d->input, d->output, filled);

    (k > 0 ? memset((void *)pIn + filled, 0x00, k) : 0);

    /*
    *   VIT Process
    */
#if (NUMBER_OF_CHANNELS == 1)
        VIT_InputBuffers.pBuffer_Chan1 = pIn;                      // PCM buffer : 16-bit - 16kHz - mono
        VIT_InputBuffers.pBuffer_Chan2 = PL_NULL;
        VIT_InputBuffers.pBuffer_Chan3 = PL_NULL;
#elif (NUMBER_OF_CHANNELS == 2)
    	PL_INT16   DeInterleavedBuffer[VIT_SAMPLES_PER_FRAME*NUMBER_OF_CHANNELS];
    	DeInterleave(pIn, DeInterleavedBuffer, VITInstParams.SamplesPerFrame, VITInstParams.NumberOfChannel);
		VIT_InputBuffers.pBuffer_Chan1 = &DeInterleavedBuffer[0];
		VIT_InputBuffers.pBuffer_Chan2 = &DeInterleavedBuffer[VIT_SAMPLES_PER_FRAME];
		VIT_InputBuffers.pBuffer_Chan3 = PL_NULL;
#endif

    VIT_Status = VIT_Process ( VITHandle,
                           &VIT_InputBuffers,                               // temporal audio input data
                           &VIT_DetectionResults
                          );

    if (VIT_Status != VIT_SUCCESS)
    {
        DSP_PRINTF("VIT_Process error : %d\n", VIT_Status);
        return VIT_SYSTEM_ERROR;                                            // will stop processing VIT and go directly to MEM free
    }


    if (VIT_DetectionResults == VIT_WW_DETECTED)
    {
        DSP_PRINTF(" - WakeWord detected \r\n");

    }
    else if (VIT_DetectionResults == VIT_VC_DETECTED)
    {
        // Retrieve id of the Voice Command detected
        // String of the Command can also be retrieved (when WW and CMDs strings are integrated in Model)
    	VIT_Status = VIT_GetVoiceCommandFound(VITHandle,
                                              &VoiceCommand
                                              );
        if (VIT_Status != VIT_SUCCESS)
        {
            DSP_PRINTF("VIT_GetVoiceCommandFound error : %d\r\n", VIT_Status);
            return VIT_SYSTEM_ERROR;                                              // will stop processing VIT and go directly to MEM free
        }
        else
        {
            DSP_PRINTF(" - Voice Command detected %d", VoiceCommand.Cmd_Id);

            // Retrieve CMD Name : OPTIONAL
            // Check first if CMD string is present
            if (VoiceCommand.pCmd_Name != PL_NULL)
            {
                DSP_PRINTF(" %s\r\n", VoiceCommand.pCmd_Name);
            }
            else
            {
            	DSP_PRINTF("\r\n");
            }
        }
    }

    /* ...put flag saying we have output buffer */
    d->state |= XA_VIT_PRE_PROC_FLAG_OUTPUT;

    /* ...return success result code */
    return 0;
}


VIT_ReturnStatus_en VIT_ModelInfo(void)
{
    VIT_ReturnStatus_en     VIT_Status;                             /* Function call status */
    /*
    *   VIT Get Model Info (OPTIONAL)
    *       To retrieve information on the VIT_Model registered in VIT:
    *               - Model Release Number, number of commands supported
    *               - WakeWord supported (when info is present)
    *               - list of commands (when info is present)
    *
    */
    VIT_ModelInfo_st Model_Info;
    VIT_Status = VIT_GetModelInfo(&Model_Info);
    if (VIT_Status != VIT_SUCCESS)
    {
        DSP_PRINTF("VIT_GetModelInfo error : %d\r\n", VIT_Status);
        return VIT_INVALID_MODEL;                                        
    }

    DSP_PRINTF("VIT Model info \r\n");
    DSP_PRINTF("  VIT Model Release = 0x%04x\r\n", Model_Info.VIT_Model_Release);
    if (Model_Info.pLanguage != PL_NULL)
    {
        DSP_PRINTF("  Language supported : %s \r\n", Model_Info.pLanguage);
    }

    DSP_PRINTF("  Number of Commands supported : %d \r\n", Model_Info.NbOfVoiceCmds);

    if (!Model_Info.WW_VoiceCmds_Strings)               // Check here if Model is containing WW and CMDs strings
    {
        DSP_PRINTF("  VIT_Model integrating WakeWord and Voice Commands strings : NO\r\n");
    }
    else
    {
        const char* ptr;

        DSP_PRINTF("  VIT_Model integrating WakeWord and Voice Commands strings : YES\r\n");
        ptr = Model_Info.pWakeWord;
        if (ptr != PL_NULL)
        {
            DSP_PRINTF("  WakeWord supported : %s \r\n", ptr);
        }
        DSP_PRINTF("  Voice commands supported : \r\n");
        ptr = Model_Info.pVoiceCmds_List;
        if (ptr != PL_NULL)
        {
            for (PL_UINT16 i = 0; i < Model_Info.NbOfVoiceCmds; i++)
            {
                DSP_PRINTF("   '%s' \r\n", ptr);
                ptr += strlen(ptr) + 1;                 // to consider NULL char
            }
        }
    }

    return VIT_SUCCESS;
}


VIT_ReturnStatus_en VIT_Initialize(vit_pre_proc_t *d)
{
    VIT_ReturnStatus_en	 VIT_Status;							 /* Function call status */
 
    d->channels = NUMBER_OF_CHANNELS;
    d->pcm_width = 16;
    d->sample_rate = 16000;
    d->frame_size_us = 10000;
	d->state = XA_VIT_PRE_PROC_FLAG_PREINIT_DONE;

    pMem = malloc(VIT_MODEL_CN_SIZE + VIT_MODEL_ALIGNMENT);
    VIT_Model = (PL_UINT8 *)INSTALLOC_ALIGN(pMem);
    memcpy(VIT_Model, (void *)VIT_MODEL_CN_ADDR, VIT_MODEL_CN_SIZE);
 
    VIT_Status = VIT_SetModel(VIT_Model, VIT_MODEL_IN_RAM);
 
    if (VIT_Status != VIT_SUCCESS)
    {
        return VIT_INVALID_MODEL;										  
    }
//    PRINTF("SetModel = %d\r\n ",VIT_Status);
/*
    VIT_Status = VIT_ModelInfo();
    if (VIT_Status != VIT_SUCCESS)
    {
        return VIT_INVALID_MODEL;										  
    }
*/

//    PRINTF("ModelInfo = %d\r\n ",VIT_Status);

    /*
    *   Configure VIT Instance Parameters
    */
    VITInstParams.SampleRate_Hz			= VIT_SAMPLE_RATE;
    VITInstParams.SamplesPerFrame			= VIT_SAMPLES_PER_FRAME;
    VITInstParams.NumberOfChannel			= NUMBER_OF_CHANNELS;
    VITInstParams.DeviceId 				= DEVICE_ID;
 
    /*
     *   VIT get memory table : Get size info per memory type
     */
    VIT_Status = VIT_GetMemoryTable( PL_NULL,				  // VITHandle param should be NULL
                    &VITMemoryTable,
                    &VITInstParams);
    if (VIT_Status != VIT_SUCCESS)
    {
        return VIT_INVALID_BUFFER_MEMORY_ALIGNMENT;	 
    }
//    DSP_PRINTF("VIT_GetMemoryTable error : %d\n", VIT_Status);
 
    /*
     *   Reserve memory space : Malloc for each memory type
     */
    for (int i = 0; i < PL_NR_MEMORY_REGIONS; i++)
    {
        /* Log the memory size */
        // DSP_PRINTF("Memory region %d, size %d in Bytes\n", (int)i, (int)VITMemoryTable.Region[i].Size);
        if (VITMemoryTable.Region[i].Size != 0)
        {
            // reserve memory space
            // NB : VITMemoryTable.Region[PL_MEMREGION_PERSISTENT_FAST_DATA] should be alloacted
            // 	 in the fastest memory of the platform (when possible) - this is not the case in this example.
            pMemory[i] = malloc(VITMemoryTable.Region[i].Size + MEMORY_ALIGNMENT);
            VITMemoryTable.Region[i].pBaseAddress = (void *)pMemory[i];
 
//            DSP_PRINTF(" Memory region address[%d] %p, size = %d\n", i, VITMemoryTable.Region[i].pBaseAddress, VITMemoryTable.Region[i].Size);
        }
    }
 
    /*
    *	  Create VIT Instance
    */
    VITHandle = PL_NULL;							 // force to null address for correct memory initialization
    VIT_Status = VIT_GetInstanceHandle( &VITHandle,
                                        &VITMemoryTable,
                                        &VITInstParams);
    if (VIT_Status != VIT_SUCCESS)
    {
        InitPhase_Error = PL_TRUE;
    }
//    DSP_PRINTF("VIT_GetInstanceHandle error : %d\n", VIT_Status);
 
    /*
    *	  Test the reset (OPTIONAL)
    */
    if (!InitPhase_Error)
    {
        VIT_Status = VIT_ResetInstance(VITHandle);
        if (VIT_Status != VIT_SUCCESS)
        {
            InitPhase_Error = PL_TRUE;
        }
 //       DSP_PRINTF("VIT_ResetInstance error : %d\n", VIT_Status);
    }
 
    /*
    *	 Set and Apply VIT control parameters
    */
    VITControlParams.OperatingMode   = VIT_OPERATING_MODE;
 
    if (!InitPhase_Error)
    {
        VIT_Status = VIT_SetControlParameters(  VITHandle,
											 &VITControlParams);
        if (VIT_Status != VIT_SUCCESS)
        {
			 InitPhase_Error = PL_TRUE;
        }
//        DSP_PRINTF("VIT_SetControlParameters error : %d\n", VIT_Status);
    }

	d->state |= XA_VIT_PRE_PROC_FLAG_RUNNING;
 
    UWORD64 product = (UWORD64)d->sample_rate * d->channels * (d->pcm_width>>3) * d->frame_size_us;
    d->buffer_size = (UWORD32)(product/1000000);
    
    /* Align the computed buffer size to 8-byte */
    d->buffer_size = (d->buffer_size + 7) & ~0x7;
    
    /* ...mark post-initialization is complete */
    d->state |= XA_VIT_PRE_PROC_FLAG_POSTINIT_DONE;
//	PRINTF("Buffer size %d, %d\n", d->buffer_size, InitPhase_Error);
//	PRINTF("%d, %d, %d, %d, %d\n", VITInstParams.SampleRate_Hz, VITInstParams.SamplesPerFrame, VITInstParams.NumberOfChannel, \
//		VITInstParams.DeviceId, VITControlParams.OperatingMode);
    return VIT_Status;
}

