/*
* Copyright 2020 NXP
* All rights reserved.
*
* SPDX-License-Identifier: BSD-3-Clause
*/
#ifdef __XCC__
#include <xtensa/config/core.h>
#include <xtensa/xos.h>
#else
#include "FreeRTOS.h"
#include "queue.h"
#endif

#ifdef CPU_MIMXRT595SFFOC_cm33
#include "board.h"
#else
#include "board_fusionf1.h"
#endif

//#include "fsl_codec_common.h"
#include "audio_capture_render.h"
#include "fsl_clock.h"

#include "fsl_inputmux.h"
#include "fsl_dma.h"
#include "fsl_dmic.h"
#include "fsl_dmic_dma.h"

#include "fsl_iopctl.h"
#include "pin_mux.h"

#include "fsl_debug_console.h"

#include "ringbuf.h"

#ifdef __XCC__
static XosMsgQueue *p_audio_capture_dmic_queue;
#else
static QueueHandle_t p_audio_capture_dmic_queue;
#endif

#define AUDIO_CAPTURE_DMIC_QUEUE_LENGTH 1

#define DMIC_RX_CHANNEL0 (16)

#ifdef DEBUG_AUDIO_LATENCY
int32_t fill_magic = 0;
#endif

static int16_t audio_capture_dmic_DMABuff[AUDIO_CAPTURE_TRANSFER_SIZE * AUDIO_CAPTURE_BUFFER_COUNT] = { 0 };
static int32_t audio_capture_dmic_buffer_index = 0;
static dma_handle_t s_dmicRxDmaHandle[AUDIO_CAPTURE_CHANNELS];
static dmic_dma_handle_t s_dmicDmaHandle[AUDIO_CAPTURE_CHANNELS];
static int32_t dmic_channel_index[AUDIO_CAPTURE_CHANNELS];
static uint32_t audio_capture_dmic_started = 0;

volatile int16_t *p_dmic_last_address = NULL;
volatile uint32_t dmic_last_size;

#if (XCHAL_DCACHE_SIZE > 0)
AT_NONCACHEABLE_SECTION_ALIGN(dma_descriptor_t s_dmaDescriptorPingpong[AUDIO_CAPTURE_CHANNELS * 2], 16);
#else
SDK_ALIGN(dma_descriptor_t s_dmaDescriptorPingpong[AUDIO_CAPTURE_CHANNELS * 2], 16);
#endif

static dmic_transfer_t s_receiveXfer[AUDIO_CAPTURE_CHANNELS * 2] =
{
#if (AUDIO_CAPTURE_CHANNELS == 1)
  /* transfer configurations for channel0 */
      { 
        .data = &audio_capture_dmic_DMABuff[0],
        .dataWidth = sizeof(int16_t),
        .dataSize = AUDIO_CAPTURE_TRANSFER_SIZE / AUDIO_CAPTURE_CHANNELS * sizeof(int16_t),
        .dataAddrInterleaveSize = kDMA_AddressInterleave1xWidth,
        .linkTransfer = &s_receiveXfer[1],
      },
      { .data = &audio_capture_dmic_DMABuff[AUDIO_CAPTURE_TRANSFER_SIZE],
        .dataWidth = sizeof(int16_t), 
        .dataSize = AUDIO_CAPTURE_TRANSFER_SIZE / AUDIO_CAPTURE_CHANNELS * sizeof(int16_t),
        .dataAddrInterleaveSize = kDMA_AddressInterleave1xWidth,
        .linkTransfer = &s_receiveXfer[0], 
      },
#elif (AUDIO_CAPTURE_CHANNELS == 2)
      /* transfer configurations for channel0 */
      { 
        .data = &audio_capture_dmic_DMABuff[0], 
        .dataWidth = sizeof(int16_t),
        .dataSize = AUDIO_CAPTURE_TRANSFER_SIZE / AUDIO_CAPTURE_CHANNELS * sizeof(int16_t),
        .dataAddrInterleaveSize = kDMA_AddressInterleave2xWidth,
        .linkTransfer = &s_receiveXfer[1],
      },    
      { .data = &audio_capture_dmic_DMABuff[AUDIO_CAPTURE_TRANSFER_SIZE],
        .dataWidth = sizeof(int16_t),
        .dataSize = AUDIO_CAPTURE_TRANSFER_SIZE / AUDIO_CAPTURE_CHANNELS * sizeof(int16_t),
        .dataAddrInterleaveSize = kDMA_AddressInterleave2xWidth,
        .linkTransfer = &s_receiveXfer[0],
      },	  
      /* transfer configurations for channel0 */
      { 
         .data = &audio_capture_dmic_DMABuff[1], .dataWidth = sizeof(int16_t),
         .dataSize = AUDIO_CAPTURE_TRANSFER_SIZE / AUDIO_CAPTURE_CHANNELS * sizeof(int16_t),
         .dataAddrInterleaveSize = kDMA_AddressInterleave2xWidth,
         .linkTransfer = &s_receiveXfer[3], 
      },      
      {  
          .data = &audio_capture_dmic_DMABuff[AUDIO_CAPTURE_TRANSFER_SIZE + 1],
          .dataWidth = sizeof(int16_t),
          .dataSize = AUDIO_CAPTURE_TRANSFER_SIZE / AUDIO_CAPTURE_CHANNELS * sizeof(int16_t),
          .dataAddrInterleaveSize = kDMA_AddressInterleave2xWidth,
          .linkTransfer = &s_receiveXfer[2], 
      }, 
#else
#error "To be implemented"
#endif
};

void dmic_dma_Callback(DMIC_Type *base, dmic_dma_handle_t *handle, status_t status, void *userData) {
  int32_t ch = *(int32_t *) userData;
  AudioSource_Queue_Element element;
  int32_t ret;

  if (ch == AUDIO_CAPTURE_CHANNELS - 1) {
    element.pBufAddress = &audio_capture_dmic_DMABuff[AUDIO_CAPTURE_TRANSFER_SIZE * audio_capture_dmic_buffer_index];
    element.uBufSize = AUDIO_CAPTURE_TRANSFER_SIZE * sizeof(int16_t); /* uBufSize in Bytes */
#ifdef DEBUG_AUDIO_LATENCY
    fill_magic++;
    if(fill_magic == FILL_MAGIC_PERIOD)
    {	
      *element.pBufAddress = DEBUG_AUDIO_MAGICWORD_0; 	
      *(element.pBufAddress + 1) = DEBUG_AUDIO_MAGICWORD_1;		
      *(element.pBufAddress + 2) = DEBUG_AUDIO_MAGICWORD_2;	
      *(element.pBufAddress + 3) = DEBUG_AUDIO_MAGICWORD_3;
      fill_magic = 0;
    }
#endif
    
#ifdef GPIO_DEBUG
    GPIO_PortToggle(GPIO, GPIO_DEBUG_PORT_1, (1 << GPIO_DEBUG_PIN_1));
#endif
    audio_capture_dmic_buffer_index ^= 1;
    if (audio_capture_dmic_started	== 1)
    {
      //			memcpy(audio_capture_dummy, element.pBufAddress, element.uBufSize);
      //          element.pBufAddress = audio_capture_dummy;
      p_dmic_last_address = element.pBufAddress;
      dmic_last_size = element.uBufSize;
#ifdef __XCC__
      ret = xos_msgq_put(p_audio_capture_dmic_queue, (uint32_t *)&element);
      if (ret != XOS_OK) {
        PRINTF("xos_msgq_put p_audio_capture_dmic_queue %d", ret);
      }
#else
      BaseType_t xHigherPriorityTaskWoken;
      
      xQueueSendFromISR(p_audio_capture_dmic_queue, (uint32_t *)&element, &xHigherPriorityTaskWoken);
      if(xHigherPriorityTaskWoken)
      {
        // Actual macro used here is port specific.
        portYIELD_FROM_ISR(xHigherPriorityTaskWoken);
      }
#endif
    }
  }
}

void AudioCapture_Init_Pins(void)
{
  /* Initialize DMIC pins below */
  const uint32_t port5_pin4_config = (IOPCTL_PIO_FUNC4 | /* Pin is configured as pdm_clk01 */
                                      IOPCTL_PIO_INBUF_EN /* Enables input buffer */);
  IOPCTL_PinMuxSet(IOPCTL, 5U, 4U, port5_pin4_config); /* PORT2 PIN16 is configured as DMIC CLK01 */
  
  const uint32_t port5_pin8_config = (IOPCTL_PIO_FUNC4 | /* Pin is configured as pdm_data01 */
                                      IOPCTL_PIO_INBUF_EN /* Enables input buffer */);
  IOPCTL_PinMuxSet(IOPCTL, 5U, 8U, port5_pin8_config); /* PORT2 PIN20 is configured as DMIC DATA01 */
}


status_t AudioCapture_Init_Dmic(void * arg) {
  dmic_channel_config_t dmic_channel_cfg;
  int32_t dmic_enable = 0;
  int32_t i;
  bool use2fs = true;
  
  AudioCapture_Init_Pins();
  
#ifdef __XCC__
  p_audio_capture_dmic_queue = malloc(
                                      XOS_MSGQ_SIZE(AUDIO_CAPTURE_DMIC_QUEUE_LENGTH,
                                                    sizeof(struct _AudioSource_Queue_Element)));
  
  xos_msgq_create(p_audio_capture_dmic_queue, AUDIO_CAPTURE_DMIC_QUEUE_LENGTH,
                  sizeof(struct _AudioSource_Queue_Element), XOS_MSGQ_WAIT_PRIORITY);
#else
  p_audio_capture_dmic_queue = xQueueCreate(AUDIO_CAPTURE_DMIC_QUEUE_LENGTH,
                                            sizeof(struct _AudioSource_Queue_Element));
#endif
  
  CLOCK_AttachClk(kFRO_DIV4_to_DMIC); // 192 / 4 = 48MHz
  CLOCK_SetClkDiv(kCLOCK_DivDmicClk, 30); // 1.6 MHz
//  CLOCK_AttachClk(kAUDIO_PLL_to_DMIC);
//  CLOCK_SetClkDiv(kCLOCK_DivDmicClk, 8); //24.576/8= 3.072MHz
  DMIC_Init(DMIC0);
  //	DMIC_SetIOCFG(DMIC0, kDMIC_PdmDual);
  DMIC_Use2fs(DMIC0, use2fs);
  
  memset(&dmic_channel_cfg, 0U, sizeof(dmic_channel_config_t));
  dmic_channel_cfg.divhfclk = kDMIC_PdmDiv1;
  dmic_channel_cfg.osr = 1600 / AUDIO_CAPTURE_SAMPLERATE_MS;
//  dmic_channel_cfg.osr = 3072 / AUDIO_CAPTURE_SAMPLERATE_MS;
  if (use2fs) {
    dmic_channel_cfg.osr = (dmic_channel_cfg.osr >> 1);
  } else {
    dmic_channel_cfg.osr = (dmic_channel_cfg.osr >> 2);
  }
  dmic_channel_cfg.gainshft = 3U;
  dmic_channel_cfg.preac2coef = kDMIC_CompValueZero;
  dmic_channel_cfg.preac4coef = kDMIC_CompValueZero;
  dmic_channel_cfg.dc_cut_level = kDMIC_DcCut155;
  dmic_channel_cfg.post_dc_gain_reduce = 1U;
  dmic_channel_cfg.saturate16bit = 1U;
  dmic_channel_cfg.sample_rate = kDMIC_PhyFullSpeed;
  
  for (i = 0; i < AUDIO_CAPTURE_CHANNELS; i++) {
    DMIC_EnableChannelDma(DMIC0, (dmic_channel_t) (kDMIC_Channel0 + i), true);
    if ((i % 2) == 0)
      DMIC_ConfigChannel(DMIC0, (dmic_channel_t) (kDMIC_Channel0 + i),
                         kDMIC_Left, &dmic_channel_cfg);
    else
      DMIC_ConfigChannel(DMIC0, (dmic_channel_t) (kDMIC_Channel0 + i),
                         kDMIC_Right, &dmic_channel_cfg);
    
    DMIC_FifoChannel(DMIC0, kDMIC_Channel0 + i, 15, true, true);
    
    dmic_enable |= (1 << i);
  }
  DMIC_EnableChannnel(DMIC0, dmic_enable);
  
  for (i = 0; i < AUDIO_CAPTURE_CHANNELS; i++) {
    dmic_channel_index[i] = i;
    DMA_EnableChannel(DMA1, DMIC_RX_CHANNEL0 + i);
    DMA_SetChannelPriority(DMA1, DMIC_RX_CHANNEL0 + i, kDMA_ChannelPriority1);
    DMA_CreateHandle(&s_dmicRxDmaHandle[i], DMA1, DMIC_RX_CHANNEL0 + i);
    DMIC_TransferCreateHandleDMA(DMIC0, &s_dmicDmaHandle[i],
                                 dmic_dma_Callback, &dmic_channel_index[i], &s_dmicRxDmaHandle[i]);
    DMIC_InstallDMADescriptorMemory(&s_dmicDmaHandle[i],
                                    &s_dmaDescriptorPingpong[2 * i], 2U);
  }
  return kStatus_Success;
}

void AudioCapture_Start_Dmic(void)
{
  int32_t i;

  for (i = 0; i < AUDIO_CAPTURE_CHANNELS; i++) {
    DMIC_TransferReceiveDMA(DMIC0, &s_dmicDmaHandle[i],
                            &s_receiveXfer[2 * i], kDMIC_Channel0 + i);
  }
#ifdef GPIO_DEBUG
  GPIO_PinWrite(GPIO, GPIO_DEBUG_PORT_1, GPIO_DEBUG_PIN_1, 1);
#endif

  audio_capture_dmic_started = 1;
}

void AudioCapture_Stop_Dmic(void)
{
  int32_t i;
  
  for (i = 0; i < AUDIO_CAPTURE_CHANNELS; i++) {
    DMIC_TransferAbortReceiveDMA(DMIC0, &s_dmicDmaHandle[i]);
  }
  
  audio_capture_dmic_started = 0;
}

status_t AudioCapture_GetData_Dmic(int16_t **pData, uint32_t *size, int32_t nocopy)
{	
  AudioSource_Queue_Element element;
  int32_t ret;
#ifdef __XCC__
  ret = xos_msgq_get(p_audio_capture_dmic_queue, (uint32_t *) &element);	
  if (ret == XOS_OK) {
    if (nocopy)
    {
      *pData = element.pBufAddress;
    } else {
      memcpy(*pData, element.pBufAddress, element.uBufSize);
    }
  } 
  else {
    PRINTF("xos_msgq_get dmic %d\n", ret);
  }
#else
  ret = xQueueReceive(p_audio_capture_dmic_queue, (uint32_t *) &element, portMAX_DELAY);
  if (ret == pdPASS)
  {
    if (nocopy)
    {
      *pData = element.pBufAddress;
    } else {
      memcpy(*pData, element.pBufAddress, element.uBufSize);
    }
  } else {
    PRINTF("xQueueReceive dmic %d\n", ret);
  }
#endif
  
  *size = element.uBufSize;
  
  if(p_dmic_last_address != *pData) {
    PRINTF("dmic 0x%x - 0x%x\n", p_dmic_last_address, *pData);
  }
  
  if(dmic_last_size != *size)
  {
    PRINTF("dmic %d - %d\n", dmic_last_size, *size);
  }
  
  return kStatus_Success;
}

status_t AudioCapture_GetData_Dmic_noblocking(int16_t **pData, uint32_t *size, int32_t nocopy)
{	
  AudioSource_Queue_Element element;
  int32_t ret;
#ifdef __XCC__
  if (xos_msgq_empty(p_audio_capture_dmic_queue) == 0) {
    xos_msgq_get(p_audio_capture_dmic_queue, (uint32_t *) &element);
    if(nocopy) {
      *pData = element.pBufAddress;
    } else {
      memcpy(*pData, element.pBufAddress, element.uBufSize);
    }
    *size = element.uBufSize;
    return kStatus_Success;
  } else {
    if(nocopy) {
      *pData = NULL;
    }		
    *size = 0;
    return kStatus_Fail;
  }
#else
  ret = xQueueReceive(p_audio_capture_dmic_queue, (uint32_t *) &element, 0);
  if (ret == pdPASS)
  {
    if (nocopy)
    {
      *pData = element.pBufAddress;
    } else {
      memcpy(*pData, element.pBufAddress, element.uBufSize);
    }
    *size = element.uBufSize;
    return kStatus_Success;
  } else {
    if(nocopy) {
      *pData = NULL;
    }		
    *size = 0;
    return kStatus_Fail;
  }
#endif
}

