/*
 * Copyright 2021 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <string.h>
#include <stdlib.h>
#include <stdbool.h>

#include "dualcore_ringbuffer_of.h"
#include "fsl_sema42.h"

uint32_t global_gate_index = 0;

void Core_Lock(uint8_t gate)
{
#if defined(CPU_MIMXRT595SFFOC_cm33)
    SEMA42_Lock(SEMA42, gate, 1);
#elif defined(CPU_MIMXRT595SFFOC_dsp)
    SEMA42_Lock(SEMA42, gate, 3);
#endif
}

void Core_unLock(uint8_t gate)
{
    SEMA42_Unlock(SEMA42, gate);
}

/*******************************************************************************
 * Code
 ******************************************************************************/
dcringbuf_t *dc_ringbuf_create(uint8_t *mem, uint32_t size)
{
    dcringbuf_t *rb;

    rb = (dcringbuf_t *)mem;
    if (!rb)
        return NULL;

    if (rb->size > 0)
        return rb;

    if(global_gate_index > FSL_FEATURE_SEMA42_GATE_COUNT)
        return NULL;

    rb->head = 0;
    rb->tail = 0;
    rb->occ  = 0;
    rb->size = size - sizeof(dcringbuf_t);
    rb->buf  = (uint8_t *)(mem + sizeof(dcringbuf_t));
    rb->gate = global_gate_index++;
    rb->mode = RINGBUF_MODE_NORMAL;

    return rb;
}

void dc_ringbuf_setmode(dcringbuf_t *rb, uint8_t mode)
{
    Core_Lock(rb->gate);

    rb->mode = mode;

    Core_unLock(rb->gate);
}


void dc_ringbuf_destroy(dcringbuf_t *rb)
{

}

void dc_ringbuf_clear(dcringbuf_t *rb)
{
    rb->head = 0;
    rb->tail = 0;
    rb->occ  = 0;
}

uint32_t dc_ringbuf_get_occupancy(dcringbuf_t *rb)
{
    return rb->occ;
}

uint32_t dc_ringbuf_write(dcringbuf_t *rb, uint8_t *data, uint32_t size)
{
    uint32_t remaining = 0;
    uint8_t *buf = rb->buf;
#if defined(CPU_MIMXRT595SFFOC_dsp)
    buf = ARM_ADDRESS_TO_DSP_ADDRESS(rb->buf);
#endif

    /* MUTEX LOCK */
    // printf("write (%d)\n", size);
    Core_Lock(rb->gate);

    remaining = rb->size - rb->tail;
    
    if (size > (rb->size - rb->occ))
    {
        /* Overflow - cap write size to available space. */
        size = rb->size - rb->occ;
    }

    /* Split write into two if it will overflow. */
    if (size > remaining)
    {
        /* First copy to end of buffer. */
        if(remaining > 0)
            memcpy(buf + rb->tail, data, remaining);
        /* Second copy from start of buffer, remaining size. */
        memcpy(buf, data + remaining, size - remaining);
        /* Set write point to 0 + <second copy size>. */
        rb->tail = size - remaining;
    }
    else
    {
        memcpy(buf + rb->tail, data, size);
        rb->tail += size;
    }

    rb->occ += size;

	if (rb->mode == RINGBUF_MODE_HALF) {
        if(rb->occ > (rb->size >> 1))
        {
            /* Remove data from the buffer end */
			rb->head += size;
			rb->occ -= size;
			if(rb->head > rb->size) {
                rb->head -= rb->size;
			}
		}
	}

    // printf("head: %d, tail: %d, occ: %d\n", rb->head, rb->tail, rb->occ);

    Core_unLock(rb->gate);
    return size;
}

static uint32_t _dc_ringbuf_read_internal(dcringbuf_t *rb, uint8_t *data, uint32_t size, bool peek)
{
    uint32_t remaining = 0;
    uint8_t *buf = rb->buf;
#if defined(CPU_MIMXRT595SFFOC_dsp)
    buf = ARM_ADDRESS_TO_DSP_ADDRESS(rb->buf);
#endif

    /* MUTEX LOCK */
    // printf("read (%d)\n", size);
    Core_Lock(rb->gate);
    
    remaining = rb->size - rb->head;
    
    if (size > rb->occ)
    {
        /* Underrun - cap read at max available. */
        size = rb->occ;
    }

    if (size > remaining)
    {
        if (data)
        {
            /* First copy to end of buffer. */
            if (remaining > 0)
                memcpy(data, buf + rb->head, remaining);
            /* Second copy from start of buffer, remaining size. */
            memcpy(data + remaining, buf, size - remaining);
        }

        if (!peek)
        {
            rb->head = size - remaining;
        }
    }
    else
    {
        if (data)
        {
            memcpy(data, buf + rb->head, size);
        }

        if (!peek)
        {
            rb->head += size;
        }
    }

    if (!peek)
    {
        rb->occ -= size;
    }

    // printf("head: %d, tail: %d, occ: %d\n", rb->head, rb->tail, rb->occ);
    Core_unLock(rb->gate);
    return size;
}

uint32_t dc_ringbuf_read(dcringbuf_t *rb, uint8_t *data, uint32_t size)
{
    return _dc_ringbuf_read_internal(rb, data, size, false);
}

uint32_t dc_ringbuf_peek(dcringbuf_t *rb, uint8_t *data, uint32_t size)
{
    return _dc_ringbuf_read_internal(rb, data, size, true);
}
