/*
 * Copyright 2021-2024 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "fsl_debug_console.h"
#include "pin_mux.h"
#include "board.h"
#include "fsl_common.h"
#include "fsl_power.h"
#include "fsl_inputmux.h"
#include "fsl_pint.h"
#include "fsl_usart.h"
#include "fsl_gint.h"

#include "fsl_irtc.h"
#include <stdbool.h>


irtc_datetime_t datetime, alarmDatetime, datetimeGet;
irtc_config_t irtcConfig;
volatile static bool alarmHappen = false;
/*******************************************************************************
 * Variables
 ******************************************************************************/

uint32_t excludeFromDS[2];
uint32_t wakeupFromDS[4];
uint32_t excludeFromPD[1];
uint32_t wakeupFromPD[4];
uint32_t excludeFromDPD[1];
uint32_t wakeupFromDPD[2];
const char *g_wakeupInfoStr[] = {"Sleep", "Deep Sleep",
		"Powerdown", "Deep Powerdown"};
uint32_t g_currentPowerMode;

/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define DEMO_WAKEUP_WITH_GINT            (1)
#define APP_USART_RX_ERROR               kUSART_RxError
#define APP_RUNNING_INTERNAL_CLOCK       BOARD_BootClockFRO12M()
#define APP_USER_WAKEUP_KEY_GPIO         BOARD_SW1_GPIO
#define APP_USER_WAKEUP_KEY_PORT         BOARD_SW1_GPIO_PORT
#define APP_USER_WAKEUP_KEY_PIN          BOARD_SW1_GPIO_PIN
#define APP_USER_WAKEUP_KEY_INPUTMUX_SEL kINPUTMUX_GpioPort1Pin18ToPintsel

#define DEMO_GINT0_PORT kGINT_Port0

/* Select one input, active low for GINT0 */
#define DEMO_GINT0_POL_MASK ~(1U << BOARD_SW3_GPIO_PIN)
#define DEMO_GINT0_ENA_MASK (1U << BOARD_SW3_GPIO_PIN)

#define APP_EXCLUDE_FROM_DEEPSLEEP (excludeFromDS)

#define APP_EXCLUDE_FROM_POWERDOWN (excludeFromPD)

#define APP_EXCLUDE_FROM_DEEPPOWERDOWN (excludeFromDPD)

#define APP_WAKEUP_FROM_DEEPSLEEP (wakeupFromDS)

#define APP_WAKEUP_FROM_POWERDOWN (wakeupFromPD)

#define APP_WAKEUP_FROM_DEEPPOWERDOWN (wakeupFromDPD)

#define APP_SYSCON_STARTER_MASK SYSCON_STARTERSET_GPIO_INT00_SET_MASK
//#define INPUTMUX INPUTMUX0

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
void DEMO_PreLowPower(void);
void DEMO_PowerDownWakeup(void);
void DEMO_PreDeepPowerDown(void);
static uint32_t APP_GetUserSelection(void);


typedef enum
{
    kSLEEP          = 0U,
    kDEEPSLEEP      = 1U,
    kPOWERDOWN      = 2U,
    kDEEPPOWERDOWN  = 3U,
	kACTIVE         = 4U
} power_modes_custom;
/*******************************************************************************
 * Code
 ******************************************************************************/
/*!
 * @brief ISR for Alarm interrupt
 *
 * This function change state of busyWait.
 */
void RTC_IRQHandler(void)
{
	if (IRTC_GetStatusFlags(RTC) & kIRTC_AlarmFlag)
	{
		alarmHappen = true;
		/* Unlock to allow register write operation */
		IRTC_SetWriteProtection(RTC, false);
		/*Clear alarm flag */
		IRTC_ClearStatusFlags(RTC, kIRTC_AlarmInterruptEnable);
	}
	SDK_ISR_EXIT_BARRIER;
}
void DEMO_PreLowPower(void)
{
	/*!< Configure RTC OSC */
	POWER_EnablePD(kPDRUNCFG_PD_XTAL32K); /*!< Powered down the XTAL 32 kHz RTC oscillator */
	POWER_DisablePD(kPDRUNCFG_PD_FRO32K); /*!< Powered the FRO 32 kHz RTC oscillator */
	CLOCK_AttachClk(kFRO32K_to_OSC32K);   /*!< Switch OSC32K to FRO32K */

	CLOCK_SetFLASHAccessCyclesForFreq(32768U); /*!< Set FLASH wait states for core */

	/*!< Set up dividers */
	CLOCK_SetClkDiv(kCLOCK_DivAhbClk, 1U, false); /*!< Set AHBCLKDIV divider to value 1 */

	/*!< Set up clock selectors - Attach clocks to the peripheries */
	CLOCK_AttachClk(kOSC32K_to_MAIN_CLK); /*!< Switch MAIN_CLK to OSC32K */

	/*< Set SystemCoreClock variable. */
	SystemCoreClock = 32768U;
}
void DEMO_PowerDownWakeup(void)
{
	BOARD_BootClockFRO12M();
	BOARD_InitDebugConsole();
}



void initializeSystem(void)
{
	/* Init board hardware. */
	CLOCK_EnableClock(kCLOCK_Gpio0); /* Enable the clock for GPIO0. */

	/* attach 12 MHz clock to FLEXCOMM0 (debug console) */
	CLOCK_SetClkDiv(kCLOCK_DivFlexcom0Clk, 0u, false);
	CLOCK_SetClkDiv(kCLOCK_DivFlexcom0Clk, 1u, true);
	CLOCK_AttachClk(BOARD_DEBUG_UART_CLK_ATTACH);

	BOARD_InitPins();
	manage_evk_io_optimization();
	BOARD_BootClockFRO12M();
	BOARD_InitDebugConsole();

	excludeFromDS[0]  = kPDRUNCFG_PD_DCDC | kPDRUNCFG_PD_FRO192M | kPDRUNCFG_PD_FRO32K;
	excludeFromDS[1]  = 0;
	excludeFromPD[0]  = kPDRUNCFG_PD_LDOMEM | kPDRUNCFG_PD_FRO32K;
	excludeFromDPD[0] = kPDRUNCFG_PD_LDOMEM | kPDRUNCFG_PD_FRO32K;

	wakeupFromDS[0]  = WAKEUP_RTC_ALARM_WAKEUP;
	wakeupFromDS[1]  = 0;
	wakeupFromDS[2]  = 0;
	wakeupFromDS[3]  = 0;
	wakeupFromPD[0]  = WAKEUP_RTC_ALARM_WAKEUP;
	wakeupFromPD[1]  = 0;
	wakeupFromPD[2]  = 0;
	wakeupFromPD[3]  = 0;
	wakeupFromDPD[0] = WAKEUP_RTC_ALARM_WAKEUP;
	wakeupFromDPD[1] = 0;

	/* Running 12 MHz to Core*/
	APP_RUNNING_INTERNAL_CLOCK;

	/* Attach Main Clock as CLKOUT */
	CLOCK_AttachClk(kMAIN_CLK_to_CLKOUT);

}

void initializeRTC(void)
{
	/* Board pin, clock, debug console init */
	/* power on XTAL32K and select XTAL32K as clock source of RTC */
	POWER_DisablePD(kPDRUNCFG_PD_XTAL32K);
	CLOCK_AttachClk(kXTAL32K_to_OSC32K);

	IRTC_GetDefaultConfig(&irtcConfig);
	if (IRTC_Init(RTC, &irtcConfig) == kStatus_Fail)
	{
		PRINTF("Faile Init RTC\r\n");
	}
}

void setupRTC_Alarm(void)
{
	datetime.year    = 2022;
	datetime.month   = 3;
	datetime.day     = 24;
	datetime.weekDay = 4;
	datetime.hour    = 18;
	datetime.minute  = 55;
	datetime.second  = 30;

	alarmDatetime.year    = 2015;
	alarmDatetime.month   = 1;
	alarmDatetime.day     = 21;
	alarmDatetime.hour    = 18;
	alarmDatetime.minute  = 55;
	alarmDatetime.second  = 32;
	alarmDatetime.weekDay = 0; /* Don't care for alarm, however this should be set to a valid value */

	/* Enable the RTC 32KHz oscillator at CFG0 by writing a 0 */
	IRTC_Enable32kClkDuringRegisterWrite(RTC, true);

	/* Clear all Tamper events by writing a 1 to the bits */
	IRTC_ClearTamperStatusFlag(RTC);

	IRTC_SetDatetime(RTC, &datetime);
	IRTC_GetDatetime(RTC, &datetimeGet);
	IRTC_SetAlarm(RTC, &alarmDatetime);

	/* Enable RTC alarm interrupt */
	IRTC_EnableInterrupts(RTC, kIRTC_AlarmInterruptEnable);

	/* Enable at the NVIC */
	EnableIRQ(RTC_IRQn);

}

void deInitRTC(void)
{
	CLOCK_DisableClock(kCLOCK_Rtc0);
}
/*!
 * @brief Main function
 */
int main(void)
{
	initializeSystem();
	initializeRTC();
	PRINTF("\r\n ~~~~~~~ Begin Demo ~~~~~~~ \r\n");
	PRINTF("\r\nEnergy Measurement Example with RTC enabled\r\n");

	setupRTC_Alarm();
	while (!alarmHappen)
	{
	}

	IRTC_GetDatetime(RTC, &datetimeGet);
	power_modes_custom next_mode = kACTIVE;

	while (1)
	{

		if(kACTIVE == next_mode)
		{
			PRINTF("\r\nIn Active Mode\r\n");
			setupRTC_Alarm();
			while (!alarmHappen)
			{
			}
			next_mode = kSLEEP;
		}

		if(kSLEEP == next_mode)
		{
			PRINTF("\r\nEntering %s ...\r\n", g_wakeupInfoStr[next_mode]);
			POWER_EnterSleep();
			setupRTC_Alarm();
			while (!alarmHappen)
			{
			}
			next_mode = kDEEPSLEEP;
		}

		if(kDEEPSLEEP == next_mode)
		{
			PRINTF("\r\nEntering %s ...\r\n", g_wakeupInfoStr[next_mode]);
			DEMO_PreLowPower();
			POWER_EnterDeepSleep(APP_EXCLUDE_FROM_DEEPSLEEP, kPOWER_SRAM_DSLP_MASK, APP_WAKEUP_FROM_DEEPSLEEP, 0x0);
			setupRTC_Alarm();
			while (!alarmHappen)
			{
			}
			next_mode = kPOWERDOWN;
		}

		if(kPOWERDOWN == next_mode)
		{
			PRINTF("\r\nEntering %s ...\r\n", g_wakeupInfoStr[next_mode]);
			DEMO_PreLowPower();
			POWER_EnterPowerDown(APP_EXCLUDE_FROM_POWERDOWN, kPOWER_SRAM_PDWN_MASK, APP_WAKEUP_FROM_POWERDOWN, 0x20000000);
			DEMO_PowerDownWakeup();
			setupRTC_Alarm();
			while (!alarmHappen)
			{
			}
			next_mode = kDEEPPOWERDOWN;
		}

		if(kDEEPPOWERDOWN == next_mode)
		{
			PRINTF("\r\nEntering %s ...\r\n", g_wakeupInfoStr[next_mode]);
			setupRTC_Alarm();
			POWER_EnterDeepPowerDown(APP_EXCLUDE_FROM_DEEPPOWERDOWN, kPOWER_SRAM_DPWD_MASK, APP_WAKEUP_FROM_DEEPPOWERDOWN, 0);
			while (!alarmHappen)
			{
			}
			next_mode = kACTIVE;
		}

	}
}
