
/*
 * AN13714 SW EXAMPLE
 * ADVANCED FEATURES OF HSCMP
 * V2.0
 *
 * Copyright 2021 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "pin_mux.h"
#include "clock_config.h"
#include "board.h"
#include "LPC55S36.h"
#include "fsl_vref.h"
#include "fsl_power.h"
#include "fsl_dac.h"
#include "freemaster.h"
#include "freemaster_serial_usart.h"
#include "fsl_usart.h"
#include "fsl_gpio.h"
#include "fsl_inputmux.h"


/*******************************************************************************
 * Definitions
 ******************************************************************************/
/*	AOI IN - Input trigger assignments */
#define   AOI_IN_PIN_INT0         	  0b000000
#define   AOI_IN_PIN_INT1             0b000001
#define   AOI_IN_SCT_OUT0             0b000010
#define   AOI_IN_SCT_OUT1             0b000011
#define   AOI_IN_SCT_OUT2             0b000100
#define   AOI_IN_SCT_OUT3             0b000101
#define   AOI_IN_T0_MAT3              0b000110
#define   AOI_IN_T1_MAT3              0b000111
#define   AOI_IN_T2_MAT3              0b001000
#define   AOI_IN_T2_MAT2              0b001001
#define   AOI_IN_T3_MAT2              0b001010
#define   AOI_IN_T4_MAT2              0b001011
#define   AOI_IN_ACMP0_OUT            0b001100
#define   AOI_IN_GPIOINT_BMATCH       0b001101
#define   AOI_IN_ADC0_IRQ             0b001110
#define   AOI_IN_ADC1_IRQ             0b001111
#define   AOI_IN_ADC0_TCOMP0          0b010000
#define   AOI_IN_ADC0_TCOMP1          0b010001
#define   AOI_IN_ADC0_TCOMP2          0b010010
#define   AOI_IN_ADC0_TCOMP3          0b010011
#define   AOI_IN_ADC1_TCOMP0          0b010100
#define   AOI_IN_ADC1_TCOMP1          0b010101
#define   AOI_IN_ADC1_TCOMP2          0b010110
#define   AOI_IN_ADC1_TCOMP3          0b010111
#define   AOI_IN_HSCMP0_OUT           0b011000
#define   AOI_IN_HSCMP1_OUT           0b011001
#define   AOI_IN_HSCMP2_OUT           0b011010
#define   AOI_IN_PWM0_SM0_MUX_TRIG0   0b011011
#define   AOI_IN_PWM0_SM0_MUX_TRIG1   0b011100
#define   AOI_IN_PWM0_SM1_MUX_TRIG0   0b011101
#define   AOI_IN_PWM0_SM1_MUX_TRIG1   0b011110
#define   AOI_IN_PWM0_SM2_MUX_TRIG0   0b011111
#define   AOI_IN_PWM0_SM2_MUX_TRIG1   0b100000
#define   AOI_IN_PWM0_SM3_MUX_TRIG0   0b100001
#define   AOI_IN_PWM0_SM3_MUX_TRIG1   0b100010
#define   AOI_IN_PWM1_SM0_MUX_TRIG0   0b100011
#define   AOI_IN_PWM1_SM0_MUX_TRIG1   0b100100
#define   AOI_IN_PWM1_SM1_MUX_TRIG0   0b100101
#define   AOI_IN_PWM1_SM1_MUX_TRIG1   0b100110
#define   AOI_IN_PWM1_SM2_MUX_TRIG0   0b100111
#define   AOI_IN_PWM1_SM2_MUX_TRIG1   0b101000
#define   AOI_IN_PWM1_SM3_MUX_TRIG0   0b101001
#define   AOI_IN_PWM1_SM3_MUX_TRIG1   0b101010
#define   AOI_IN_ENC0_CMP_POS_MATCH   0b101011
#define   AOI_IN_ENC1_CMP_POS_MATCH   0b101100
#define   AOI_IN_EXTTRIG_IN0          0b101101
#define   AOI_IN_EXTTRIG_IN1          0b101110
#define   AOI_IN_EXTTRIG_IN2          0b101111
#define   AOI_IN_EXTTRIG_IN3          0b110000

#define   AOI_IN_DMA0_TRIGOUT0        0b110011
#define   AOI_IN_DMA0_TRIGOUT1        0b110100
#define   AOI_IN_DMA0_TRIGOUT2        0b110101
#define   AOI_IN_DMA0_TRIGOUT3        0b110110
#define   AOI_IN_DMA0_TRIGOUT4        0b110111
#define   AOI_IN_DMA0_TRIGOUT5        0b111000
#define   AOI_IN_DMA0_TRIGOUT6        0b111001
#define   AOI_IN_DMA1_TRIGOUT0        0b111010
#define   AOI_IN_DMA1_TRIGOUT1        0b111011
#define   AOI_IN_DMA1_TRIGOUT2        0b111100

#define   AOI_FCN_FORCE0	0b00
#define   AOI_FCN_IN		0b01
#define   AOI_FCN_NOT_IN	0b10
#define   AOI_FCN_FORCE1	0b11


#define GPIO_TOGGLE(PORT,PIN)	GPIO->NOT[PORT] = 1UL << PIN

#define PWM_INIT_VAL			(int16_t)(0)
#define PWM_WINDOW_VAL			(int16_t)(15000)
#define PWM_VAL1				(int16_t)(30000)
/*******************************************************************************
 * Prototypes
 ******************************************************************************/
static void HscmpInit(void);
static void DacInit(void);
static void FreemasterUsartInit(void);
static void SystickInit(void);
static void CtimerInit(void);
static void eFlexPwmInit(void);
static void AoiInit(void);
/*******************************************************************************
 * Variables
 ******************************************************************************/
uint16_t ui16SystickIsrCnt;
uint16_t ui16Dac12bValue;
uint8_t ui8DacValue;
uint8_t ui8HystCtrl;
uint8_t ui8CoutInv;
uint8_t ui8WindowInv;
uint8_t ui8CoutaOw;
uint8_t ui8CoutaOwen;
uint8_t ui8CoutSel;
uint8_t ui8WindowCls;
uint32_t ui32RefreshValues;
uint32_t ui32Register;
int16_t i16DutyCycle;
int16_t i16DutyCycleReq;
uint16_t ui16Modulo;
int16_t i16Duty;
int16_t i16DutyWindow;
int32_t i32DutyCalc;

static FMSTR_U8 recBuffer[2048];     /* Recorder #1 sampling buffer */



/*******************************************************************************
 * Code
 ******************************************************************************/

/* Init Freemaster and recorder */
void FreemasterInit()
{
    /* FreeMASTER communication layer initialization */
	FreemasterUsartInit();

	FMSTR_Init();

	FMSTR_REC_BUFF recBuffCfg;
	recBuffCfg.addr          = (FMSTR_ADDR)recBuffer;
	recBuffCfg.size          = sizeof(recBuffer);
	recBuffCfg.basePeriod_ns = 50000;
	recBuffCfg.name          = "FMSTR_REC";
	FMSTR_RecorderCreate(0, &recBuffCfg);
}

/*!
 * @brief Main function
 */
int main(void)
{
    __disable_irq();
    ui8DacValue = 127;
    i16Duty = 1000;
    i16DutyWindow = 1000;
    BOARD_BootClockPLL150M();
    /* Enable flash prefetch for better performance */
    SYSCON->FMCCR |= SYSCON_FMCCR_PREFEN_MASK;
    BOARD_InitPins();
    INPUTMUX_Init(INPUTMUX);
    HscmpInit();
    DacInit();
    SystickInit();
    CtimerInit();
    AoiInit();
    eFlexPwmInit();
    FreemasterInit();
    __enable_irq();

    while (1)
    {
    	FMSTR_Poll();
    }
}

/* Periodic interrupt for updating registers and generating signal using DAC */
void SysTick_Handler()
{
	/* Interrupt check counter */
	ui16SystickIsrCnt++;

    /* PWM duty cycles */
	PWM1->SM[0].VAL2 = (uint16_t)-i16Duty/2;
	PWM1->SM[0].VAL3 = (uint16_t)i16Duty/2;
	PWM1->SM[1].VAL2 = (uint16_t)-i16DutyWindow/2;
	PWM1->SM[1].VAL3 = (uint16_t)i16DutyWindow/2;
	PWM1->MCTRL     |= PWM_MCTRL_LDOK(15);

	/* Creates saw wave */
	if(ui16Dac12bValue >= 4095)
	{
		ui16Dac12bValue = 0;
	}
	else
	{
		ui16Dac12bValue++;
	}
	/* Put data to DAC */
	DAC0->DATA = LPDAC_DATA_DATA(ui16Dac12bValue);
	/* Update HSCMP registers */
	HSCMP0->DCR = (HSCMP0->DCR & ~HSCMP_DCR_DAC_DATA_MASK) | HSCMP_DCR_DAC_DATA(ui8DacValue);
	HSCMP0->CCR1 = (HSCMP0->CCR1 & ~( HSCMP_CCR1_COUTA_OWEN_MASK | HSCMP_CCR1_COUTA_OW_MASK
									| HSCMP_CCR1_WINDOW_INV_MASK | HSCMP_CCR1_COUT_INV_MASK
									| HSCMP_CCR1_COUT_SEL_MASK   | HSCMP_CCR1_WINDOW_CLS_MASK))
			      | (HSCMP_CCR1_WINDOW_INV(ui8WindowInv) | HSCMP_CCR1_COUTA_OW(ui8CoutaOw)
				  |  HSCMP_CCR1_COUTA_OWEN(ui8CoutaOwen) | HSCMP_CCR1_COUT_INV(ui8CoutInv)
				  |  HSCMP_CCR1_COUT_SEL(ui8CoutSel)| HSCMP_CCR1_WINDOW_CLS(ui8WindowCls));
}

/* Timer to capture and create short pulse from rising edge of HSCMP output */
void CTIMER0_IRQHandler()
{
	uint32_t int_stat;
	/* Get Interrupt status flags */
	int_stat = CTIMER0->IR;
	/* Clear the status flags that were set */
	CTIMER0->IR = int_stat;

	GPIO_TOGGLE(1,28);
}


static void SystickInit(void)
{
    SysTick_Config(BOARD_BOOTCLOCKPLL150M_CORE_CLOCK/100000);
	NVIC_EnableIRQ(SysTick_IRQn);
	NVIC_SetPriority(SysTick_IRQn, 2);
}

static void CtimerInit(void)
{
    /* Use 12 MHz clock for some of the Ctimers */
    CLOCK_SetClkDiv(kCLOCK_DivCtimer0Clk, 0u, false);
    CLOCK_SetClkDiv(kCLOCK_DivCtimer0Clk, 1u, true);
    CLOCK_AttachClk(kFRO_HF_to_CTIMER0);
	CLOCK_EnableClock(kCLOCK_Timer0);

	INPUTMUX_AttachSignal(INPUTMUX, 0UL, kINPUTMUX_Hscmp0OutToTimer0Trigger);

	CTIMER0->MCR |= CTIMER_MCR_MR3S_MASK 	/* Counter stop  on match */
				 |  CTIMER_MCR_MR3R_MASK 	/* Counter reset on match */
				 |  CTIMER_MCR_MR3RL_MASK 	/* Reload on match */
				 |  CTIMER_MCR_MR0RL_MASK; 	/* Reload on match */

	CTIMER0->MR[3] 	= 65535;
	CTIMER0->MSR[3] = 65535;

	CTIMER0->MR[0] 	= 32767;
	CTIMER0->MSR[0] = 32767;

	CTIMER0->IR 	= 0x000000FF;

	CTIMER0->EMR 	= CTIMER_EMR_EMC3(1)		/* Set on match */
				 	| CTIMER_EMR_EMC0(2);		/* Set on match */
	CTIMER0->PWMC	= CTIMER_PWMC_PWMEN3_MASK
				    | CTIMER_PWMC_PWMEN0_MASK;
	CTIMER0->PR 	= 0; 						/* Every APB bus clock */
	CTIMER0->TCR   |= CTIMER_TCR_ATCEN_MASK;	/* Enable trigger input */
}

static void eFlexPwmInit(void)
{
	//INPUTMUX_AttachSignal(INPUTMUX, 0UL, kINPUTMUX_Aoi0Out0ToPwm1ExtSyncTrigger);
	//INPUTMUX_AttachSignal(INPUTMUX, 1UL, kINPUTMUX_Aoi0Out0ToPwm1ExtSyncTrigger);

	//INPUTMUX->PWM1_EXTSYNC[0] = INPUTMUX_PWM1_EXTSYNCN_PWM1_EXTSYNC_TRIGIN(0b100010); /* AOI0_OUT0 */
	//INPUTMUX->PWM1_EXTSYNC[1] = INPUTMUX_PWM1_EXTSYNCN_PWM1_EXTSYNC_TRIGIN(0b100010); /* AOI0_OUT0 */
	/* Enable FlexPWM submodule 0~3 function clock source in SYSCON */
	SYSCON->PWM1SUBCTL 		 = SYSCON_PWM1SUBCTL_CLK0_EN_MASK
			                 | SYSCON_PWM1SUBCTL_CLK1_EN_MASK
							 | SYSCON_PWM1SUBCTL_CLK2_EN_MASK
							 | SYSCON_PWM1SUBCTL_CLK3_EN_MASK;
	/* Enable interface clock of FlexPWM */
	SYSCON->AHBCLKCTRLSET[3] |= SYSCON_AHBCLKCTRL3_PWM1_MASK;

	PWM1->SM[0].INIT 		= (uint16_t)(-PWM_VAL1/2);		/* Init Counter VAL */
	PWM1->SM[0].VAL2 		= (uint16_t)(-PWM_VAL1/2);
	PWM1->SM[0].VAL0 		= 0;
	PWM1->SM[0].VAL1 		= (uint16_t)((PWM_VAL1/2)-1);
	PWM1->SM[0].VAL3 		= (int16_t)(0);					/* Initial PWM duty */
	PWM1->SM[0].DISMAP[0] 	= 0xF111U;						/* PWM output in fault status */

	PWM1->SM[1].INIT 		= (uint16_t)(-PWM_VAL1/2);		/* Init Counter VAL */
	PWM1->SM[1].VAL2 		= (uint16_t)(-PWM_VAL1/2);
	PWM1->SM[1].VAL0 		= 0;
	PWM1->SM[1].VAL1 		= (uint16_t)((PWM_VAL1/2)-1);
	PWM1->SM[1].VAL3 		= (int16_t)(0);					/* Initial PWM duty */
	PWM1->SM[1].DISMAP[0] 	= 0xF111U;						/* PWM output in fault status */

	/* Master reload is generated every n-th opportunity */
	PWM1->SM[0].CTRL   = (PWM1->SM[0].CTRL & ~PWM_CTRL_LDFQ_MASK) | PWM_CTRL_LDFQ(0) | PWM_CTRL_PRSC(0x7);
	PWM1->SM[0].CTRL2 |= PWM_CTRL2_INIT_SEL(0b11);	/* Ext sync */

	/* Master reload is generated every n-th opportunity */
	PWM1->SM[1].CTRL   = (PWM1->SM[1].CTRL & ~PWM_CTRL_LDFQ_MASK) | PWM_CTRL_LDFQ(0) | PWM_CTRL_PRSC(0x7);
	PWM1->SM[1].CTRL2 |= PWM_CTRL2_INIT_SEL(0b11);	/* Ext sync */
	PWM1->SM[1].TCTRL  = PWM_TCTRL_PWAOT0(0b1);		/* Out trigger is PWM out */

	PWM1->FSTS   = (PWM1->FSTS & ~PWM_FSTS_FFULL_MASK) | PWM_FSTS_FFULL(0x1);		/* PWM outputs are re-enabled at the start of a full cycle */
	PWM1->FFILT  = (PWM1->FFILT & ~PWM_FFILT_FILT_PER_MASK) | PWM_FFILT_FILT_PER(2);/* Fault Filter Period */
	PWM1->FCTRL  = 0xF0F0;	/* A logic 1 on the fault input indicates a fault condition */
	PWM1->FSTS 	|= 0x000F;	/* Cause a simulated fault */

	/* Start PWMs (set load OK flags and run) */
	PWM1->MCTRL = (PWM1->MCTRL & ~PWM_MCTRL_CLDOK_MASK) | PWM_MCTRL_CLDOK(0xF);
	PWM1->MCTRL = (PWM1->MCTRL & ~PWM_MCTRL_LDOK_MASK)  | PWM_MCTRL_LDOK(0xF);
	PWM1->MCTRL = (PWM1->MCTRL & ~PWM_MCTRL_RUN_MASK)   | PWM_MCTRL_RUN(0xF);
	PWM1->FSTS  = (PWM1->FCTRL & ~PWM_FSTS_FFLAG_MASK)  | PWM_FSTS_FFLAG(0xF);

	/* Enable A&B PWM outputs for submodules one, two and three */
	PWM1->OUTEN = (PWM1->OUTEN & ~PWM_OUTEN_PWMA_EN_MASK) | PWM_OUTEN_PWMA_EN(7);
}

static void AoiInit(void)
{
	CLOCK_EnableClock(kCLOCK_Aoi0);
	CLOCK_EnableClock(kCLOCK_Aoi1);

	INPUTMUX->AOI0_IN[0] = INPUTMUX_AOI0_IN_IN(AOI_IN_T0_MAT3);
	INPUTMUX->AOI0_IN[1] = INPUTMUX_AOI0_IN_IN(AOI_IN_PWM1_SM1_MUX_TRIG0);

	AOI0->BFCRT[0].BFCRT01 = AOI_BFCRT01_PT0_AC(AOI_FCN_NOT_IN) /* Input 1 Complement of T0_MAT3 */
			               | AOI_BFCRT01_PT0_BC(AOI_FCN_IN)		/* Input 2 PWM1_SM1_MUX_TRIG0 - PWM WINDOW*/
						   | AOI_BFCRT01_PT0_CC(AOI_FCN_FORCE1)
						   | AOI_BFCRT01_PT0_DC(AOI_FCN_FORCE1)
						   | AOI_BFCRT01_PT1_AC(AOI_FCN_FORCE0)
			               | AOI_BFCRT01_PT1_BC(AOI_FCN_FORCE0)
						   | AOI_BFCRT01_PT1_CC(AOI_FCN_FORCE0)
						   | AOI_BFCRT01_PT1_DC(AOI_FCN_FORCE0);

	AOI0->BFCRT[0].BFCRT23 = AOI_BFCRT23_PT2_AC(AOI_FCN_FORCE0)
			               | AOI_BFCRT23_PT2_BC(AOI_FCN_FORCE0)
						   | AOI_BFCRT23_PT2_CC(AOI_FCN_FORCE0)
						   | AOI_BFCRT23_PT2_DC(AOI_FCN_FORCE0)
						   | AOI_BFCRT23_PT3_AC(AOI_FCN_FORCE0)
						   | AOI_BFCRT23_PT3_BC(AOI_FCN_FORCE0)
					       | AOI_BFCRT23_PT3_CC(AOI_FCN_FORCE0)
						   | AOI_BFCRT23_PT3_DC(AOI_FCN_FORCE0);
}

static void HscmpInit(void)
{
	/* Disable VREF power down */
	POWER_DisablePD(kPDRUNCFG_PD_VREF);
	/* Disable CMP bias power down */
	POWER_DisablePD(kPDRUNCFG_PD_CMPBIAS);
	/* Disable CMP0 power down */
	POWER_DisablePD(kPDRUNCFG_PD_HSCMP0);
	/* Disable CMP0 DAC power down */
	POWER_DisablePD(kPDRUNCFG_PD_HSCMP0_DAC);

	CLOCK_EnableClock(kCLOCK_Hscmp0);
	CLOCK_EnableClock(kCLOCK_InputMux);

	/* CCR0 CONFIGURATION */
	HSCMP0->CCR0 = HSCMP_CCR0_LINKEN(0)			/* 	CMP-to-DAC Link Enable
													Controls the link from the CMP enable to the DAC enable.
													0 - Disable the CMP-to-DAC link: enabling or disabling the DAC is independent from enabling or disabling the CMP
													1 - Enable the CMP-to-DAC link: the DAC enable/disable is controlled by the CMP_EN bit instead of DCR[DAC_EN] */
				 | HSCMP_CCR0_CMP_STOP_EN(0);	/* Comparator STOP Mode Enable
													Allows software to enable the analog comparator or the DAC when the device is in STOP mode.
													CMP_STOP_EN has no effect in trigger mode.
													0 - Disable the analog comparator regardless of CMP_EN.
													1 - Allow the analog comparator to be enabled by CMP_EN. */
	/* CCR1 CONFIGURATION */
	HSCMP0->CCR1 = HSCMP_CCR1_FILT_PER(0x00)	/*	Filter Sample Period
									    			Specifies the sampling period (in bus clock cycles) of the comparator output filter.
									    			Programming FILT_PER	to 0x00 bypasses the filter.
									    			Filter programming and latency details are provided in the functional description section.
		                                			FILT_PER has no effect in Sampling mode (CCR1[SAMPLE_EN]=1) */
				 | HSCMP_CCR1_FILT_CNT(0x0)		/*  Filter Sample Count
									    			Specifies the number of consecutive samples that must agree before the comparator output filter accepts
													the sample as a new valid output state. For information regarding filter programming and latency, see the
													functional description section.
													000 - Filter is bypassed: COUT = COUTA
													001 - 1 consecutive sample (Comparator output is simply sampled.)
													010 - 2 consecutive samples
													011 - 3 consecutive samples
													100 - 4 consecutive samples
													101 - 5 consecutive samples
													110 - 6 consecutive samples
													111 - 7 consecutive samples */
				 | HSCMP_CCR1_EVT_SEL(0x0)		/*  COUT Event Select
									    			Selects which COUT signal edge (rising, falling, or both) defines a COUT event.
													00 - Rising edge
													01 - Falling edge
													1x - Both edges */
				 | HSCMP_CCR1_WINDOW_CLS(0)		/*  COUT Event Window Close
													Enables a COUT event (defined as a COUT rising edge, falling edge, or both)
													to close an active window. See the EVT_SEL field to configure the COUT event
													0 - COUT event cannot close the window
													1 - COUT event can close the window */
				 | HSCMP_CCR1_WINDOW_INV(0)		/* 	WINDOW/SAMPLE Signal Invert
													Inverts the WINDOW/SAMPLE signal.
													0 - Do not invert
													1 - Invert */
				 | HSCMP_CCR1_COUTA_OW(0)		/*  COUTA Output Level for Closed Window
													Defines the COUTA signal value when the window is closed.
													Valid only in Windowing mode and when COUTA_OWEN=1.
													NOTE
													0 - COUTA is 0
													1 - COUTA is 1 */
				 | HSCMP_CCR1_COUTA_OWEN(0)		/*  COUTA_OW Enable
													Enables the COUTA signal value to be defined by the COUTA_OW bit when the window is closed.
													Valid only in Windowing mode.
													0 - COUTA holds the last sampled value
													1 - COUTA is defined by the COUTA_OW bit */
				 | HSCMP_CCR1_COUT_PEN(1)		/*  Comparator Output Pin Enable
													Enables the comparator output to become an available signal option for a selected package pin.
													0 - Not available
													1 - Available */
				 | HSCMP_CCR1_COUT_SEL(1)		/*  Comparator Output Select
													Selects which comparator output option, COUT or COUTA, to use for CMPO.
													0 - Use COUT (filtered)
													1 - Use COUTA (unfiltered) */
				 | HSCMP_CCR1_COUT_INV(0)		/*  Comparator Invert
													Selects the polarity of the analog comparator function, affecting the value driven to the COUT output (on
													both the device pin and as CSR[COUT]) when CCR0[CMP_EN] is 0.
													COUT_INV has no effect in trigger mode.
													0 - Do not invert
													1 - Invert */
				 | HSCMP_CCR1_DMA_EN(0)			/* 	DMA Enable
													Enables DMA transfers triggered from the HSCMP module. When DMA_EN and the corresponding
													interrupt enable bit are set, a DMA request is asserted when CFR or CFF is set.
													0 - Disable
													1 - Enable */
				 | HSCMP_CCR1_SAMPLE_EN(0)		/*  Sampling Enable
													Enables Sampling mode.
													0 - Disable
													1 - Enable */
				 | HSCMP_CCR1_WINDOW_EN(0);		/*  Windowing Enable
													Enables Windowing mode.
													Valid only when SAMPLE_EN=0.
													0 - Disable
													1 - Enable */

	/* CCR2 CONFIGURATION */
	HSCMP0->CCR2 =	HSCMP_CCR2_INMSEL(0)		/* 	Input Minus Select
													Selects the minus input of the comparator.
													These selections connect directly to the minus input of the comparator.
													00 - IN0: from the 8-bit DAC output
													01 - IN1: from the analog 8-1 mux */
				  | HSCMP_CCR2_INPSEL(1)      	/* 	Input Plus Select
													Selects the plus input of the comparator.
													00 - IN0: from the 8-bit DAC output
													01 - IN1: from the analog 8-1 mux */
				  | HSCMP_CCR2_MSEL(0x7)		/* 	Selects the input used for the negative mux.
													MSEL has no effect in trigger mode.
													MSEL has no effect if INMSEL = 0
													000 - HSCMP0_IN0 PIO0_24
													001 - HSCMP0_IN1 PIO1_12
													010 - unconnected
													011 - HSCMP0_IN3 PIO1_5
													100 - HSCMP0_IN4 OPAMP0_OUT PIO1_9
													101 - DAC0_OUT PIO1_22
													110 - Reserved
													111 - Internal 8-bit DAC output */
				  | HSCMP_CCR2_PSEL(0x5)		/* 	Selects the input used for the positive mux.
													PSEL has no effect in trigger mode.
													PSEL has no effect if INPSEL = 0
													000 - HSCMP0_IN0 PIO0_24
													001 - HSCMP0_IN1 PIO1_12
													010 - unconnected
													011 - HSCMP0_IN3 PIO1_5
													100 - HSCMP0_IN4 OPAMP0_OUT PIO1_9
													101 - DAC0_OUT PIO1_22
													110 - Reserved
													111 - Internal 8-bit DAC output */
				  | HSCMP_CCR2_HYSTCTR(0x0)		/*	Comparator Hysteresis Control
													Selects the level of internally generated hysteresis for the comparator output. See the chip data sheet to
													get the specific values for each hysteresis level.
													00 - Level 0
													01 - Level 1
													10 - Level 2
													11 - Level 3 */

				 | HSCMP_CCR2_OFFSET(0)			/*	Comparator Offset Control
													Selects the level of internally generated voltage offset for the comparator output. See the chip data sheet
													to get the specific values for each offset level.
													0 - Level 0: The hysteresis selected by HYSTCTR is valid for both directions (rising and falling).
													1 - Level 1: Hysteresis does not apply when INP (input-plus) crosses INM (input-minus) in the rising direction
													or when INM crosses INP in the falling direction. Hysteresis still applies for INP crossing INM in the falling
													direction */

				 | HSCMP_CCR2_CMP_NPMD(0)		/*	CMP Nano Power Mode Select
													Enables Nano Power mode for the comparator.
													0 - Disable (Mode is determined by CMP_HPMD.)
													1 - Enable */

				 | HSCMP_CCR2_CMP_HPMD(1);		/* 	CMP High Power Mode Select
													Selects Low or High Power(Speed) mode for the comparator.
													Valid only when not in Nano Power mode (CMP_NPMD=0).
													NOTE
													0 - Low power(speed) comparison mode
													1 - High power(speed) comparison mode */

	/* DCR CONFIGURATION */
	HSCMP0->DCR = HSCMP_DCR_DAC_DATA(130)	/* 	DAC Output Voltage Select
												Selects the DAC output (DACO) voltage from one of 256 distinct levels.
												The DACO range is from Vin/256 to Vin: DACO = (Vin/256) * (DAC_DATA + 1) */
				| HSCMP_DCR_DACOE(1)		/* 	DAC Output Enable
												Enables the DAC output to be available for other on-chip peripherals.
												0 - Disable
												1 - Enable */
				| HSCMP_DCR_VRSEL(0)		/* 	DAC Reference High Voltage Source Select
												Selects the high voltage reference source for the Vin supply of the DAC's resistor ladder network.
												See the chip-specific HSCMP information for the source of vrefh0 and vrefh1.
												0 - vrefh0 VDD_MAIN
												1 - vrefh1 VREF_OUT */
				| HSCMP_DCR_DAC_HPMD(1) 	/* 	DAC High Power Mode Select
												Enables the DAC high power mode.
												0 - Disable
												1 - Enable */
				| HSCMP_DCR_DAC_EN(1);		/* 	DAC Enable
												Enables the DAC. When disabled, the DAC is powered down to conserve power.
												0 - Disable
												1 - Enable */

	/* CCR0 ENABLE COMPARATOR */
	HSCMP0->CCR0 |= HSCMP_CCR0_CMP_EN(1);
}

static void DacInit(void)
{
	/* 12-bit DAC INIT BEGIN */
	CLOCK_SetClkDiv(kCLOCK_DivDac0Clk, 1U, true);
	CLOCK_AttachClk(kMAIN_CLK_to_DAC0);
	/* Disable DAC0 power down */
	POWER_DisablePD(kPDRUNCFG_PD_DAC0);
	CLOCK_EnableClock(kCLOCK_Dac0);
	/* Reset the logic. */
	DAC_SetReset(DAC0, kDAC_ResetLogic);
	DAC_ClearReset(DAC0, kDAC_ResetLogic);

	/* Reset the FIFO. */
	DAC_SetReset(DAC0, kDAC_ResetFIFO);
	DAC_ClearReset(DAC0, kDAC_ResetFIFO);

	DAC0->GCR =
		  LPDAC_GCR_BUF_SPD_CTRL(1)			/*	OPAMP as buffer, speed control signal
											0 - Lower low power mode
											1 - Low power mode */

		| LPDAC_GCR_IREF_ZTC_EXT_SEL(0)		/*	Internal ZTC Current Reference Select
											This bit is to select the Internal ZTC Current Reference.
											0 - Internal ZTC Current Reference not selected
											1 - Internal ZTC Current Reference selected */

		| LPDAC_GCR_IREF_PTAT_EXT_SEL(1)	/* 	Internal PTAT Current Reference Select
											This bit is to select the Internal PTAT Current Reference.
											0 - Internal PTAT Current Reference not selected
											1 - Internal PTAT Current Reference selected */

		| LPDAC_GCR_BUF_EN(1)				/* 	Buffer Enable
											This bit is to use Opamp as the DAC analog buffer.
											0 - Opamp is not used as buffer
											1 - Opamp is used as buffer */

		| LPDAC_GCR_LATCH_CYC(0)			/* 	RCLK cycles before data latch
											This register is used to configure the DAC sync cycles which is helpful to reduce glitch on the output. The
											sync time is (LATCH_CYC+1) RCLK cycles. User should configure this register according to the RCLK
											frequency. The recommended sync time is at least 40ns. Example 1, if fRCLK is 48MHz (period is 20.8ns),
											user can configure LATCH_CYC to 0x1 to generate a 41.6ns sync time. Example 2, if fRCLK is 6MHz
											(period is 166.7ns), user can configure LATCH_CYC to 0x0 to generate a 166.7ns sync time. Reference
											the chip-specific DAC information to see the RCLK frequency. */

		| LPDAC_GCR_PTGEN(0)				/*	DAC periodic trigger mode enable
											This bit is to enable the periodic trigger mode when FIFOEN is set. */

		| LPDAC_GCR_TRGSEL(1)				/*	DAC Trigger Select
											This bit is to select the trigger source for the FIFO mode.
											0 - The DAC hardware trigger is selected.
											1 - The DAC software trigger is selected. */

		| LPDAC_GCR_SWMD(1)					/* 	Swing Back Mode
											This bit is to select the swing back mode when FIFOEN is set.
											0 - Swing back mode disable
											1 - Swing back mode enable */

		| LPDAC_GCR_FIFOEN(0)				/* 	FIFO Enable
											This bit is to select the FIFO mode or buffer mode.
											0 - FIFO mode is disabled and buffer mode is enabled.
												Any data written to DATA[DATA] goes to buffer then goes to conversion.
											1 - FIFO mode is enabled. Data will be first read from FIFO to buffer then goes to conversion */

		| LPDAC_GCR_DACRFS(0x0)				/* 	DAC Reference Select
											This field to select the source of reference voltage high.
											00 - The DAC selects VREFH1 - VDDA supply rail
											01 - The DAC selects VREFH2 - VREF_OUT
											10 - The DAC selects VREFH3 - VREFP
											11 - Reserved. */

		| LPDAC_GCR_DACEN(1);				/*	DAC Enable
											Starts the Programmable Reference Generator operation.
											0 - The DAC system is disabled.
											1 - The DAC system is enabled. */
}

/*
 * USART Module initialization
 */
static void FreemasterUsartInit(void)
{
    /* attach main clock divide to FLEXCOMM0 (debug console) */
    CLOCK_AttachClk(BOARD_DEBUG_UART_CLK_ATTACH);

    usart_config_t config;
    /*
     *   usartConfig->baudRate_Bps = 115200U;
     *   usartConfig->parityMode = kUSART_ParityDisabled;
     *   usartConfig->stopBitCount = kUSART_OneStopBit;
     *   usartConfig->bitCountPerChar = kUSART_8BitsPerChar;
     *   usartConfig->loopback = false;
     *   usartConfig->enableTx = false;
     *   usartConfig->enableRx = false;
    */
    USART_GetDefaultConfig(&config);
    /* Override the Default configuration to satisfy FreeMASTER needs */
    config.baudRate_Bps = 115200U;
    config.enableTx = true;
    config.enableRx = true;

    USART_Init((USART_Type*)BOARD_DEBUG_UART_BASEADDR, &config, BOARD_DEBUG_UART_CLK_FREQ);

    /* Register communication module used by FreeMASTER driver. */
    FMSTR_SerialSetBaseAddress((USART_Type*)BOARD_DEBUG_UART_BASEADDR);

#if FMSTR_SHORT_INTR || FMSTR_LONG_INTR
    /* Enable UART interrupts. */
    EnableIRQ(BOARD_UART_IRQ);
    EnableGlobalIRQ(0);
#endif

}
