/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "pin_mux.h"
#include "board.h"
#include "fsl_usart.h"
#include <stdbool.h>

#include <stdio.h>

#include "fsl_debug_console.h"
#include "lpc860_image.h"
/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define DEMO_USART          USART2
#define DEMO_USART_CLK_SRC  kCLOCK_Flexcomm2
#define DEMO_USART_CLK_FREQ CLOCK_GetFlexCommClkFreq(2U)

#define LPC800_IMAGE_SIZE   (sizeof(lpc860_image))
/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/

uint8_t txbuff[]   = "Usart polling example\r\nBoard will send back received characters\r\n";
uint8_t rxbuff[20] = {0};

uint8_t sync_command[] = "?";
uint8_t sync_response[] = "Synchronized\r\nOK\r\n";

uint8_t crystal_freq_command[] = "12000\r\n";
uint8_t crystal_freq_response[] = "12000\r\nOK\r\n";

uint8_t unlock_command[] = "U 23130\r\n";
uint8_t unlock_response[] = "U 23130\r\n0\r\n";

uint8_t full_chip_prepare_command[] = "P 0 63\r\n";
uint8_t full_chip_prepare_response[] = "P 0 63\r\n0\r\n";

uint8_t full_chip_erase_command[] = "E 0 63\r\n";
uint8_t full_chip_erase_response[] = "E 0 63\r\n0\r\n";

uint8_t echo_disable_command[] = "A 0\r\n";
uint8_t echo_disable_response[] = "A 0\r\n0\r\n";

uint8_t CMD_SUCCESS[] = "0\r\n";
uint8_t RESPONSE_OK[] = "OK\r\n";

uint8_t command_buff[128] = {0};
uint8_t command_resp[128] = {0};

uint8_t lpc800_sector_buff[1024] = {0};

/*******************************************************************************
 * Code
 ******************************************************************************/
/*!
 * @brief Main function
 */
int main(void)
{
    uint8_t ch;
    usart_config_t config;

    const uint8_t *p_image_location = (const uint8_t *)lpc860_image;
    uint32_t target_flash_addr = 0;
    uint32_t lpc800_image_align_size = 0;

    /* attach 12 MHz clock to FLEXCOMM0 (debug console) */
    CLOCK_SetClkDiv(kCLOCK_DivFlexcom0Clk, 0u, false);
    CLOCK_SetClkDiv(kCLOCK_DivFlexcom0Clk, 1u, true);
    CLOCK_AttachClk(BOARD_DEBUG_UART_CLK_ATTACH);

    CLOCK_SetClkDiv(kCLOCK_DivFlexcom2Clk, 0u, false);
    CLOCK_SetClkDiv(kCLOCK_DivFlexcom2Clk, 1u, true);
    CLOCK_AttachClk(kFRO12M_to_FLEXCOMM2);

    BOARD_InitPins();
    BOARD_BootClockPLL150M();
    BOARD_InitDebugConsole();
    /*
     * config.baudRate_Bps = 115200U;
     * config.parityMode = kUSART_ParityDisabled;
     * config.stopBitCount = kUSART_OneStopBit;
     * config.loopback = false;
     * config.enableTx = false;
     * config.enableRx = false;
     */
    USART_GetDefaultConfig(&config);
    config.baudRate_Bps = BOARD_DEBUG_UART_BAUDRATE;
    config.enableTx     = true;
    config.enableRx     = true;

    USART_Init(DEMO_USART, &config, DEMO_USART_CLK_FREQ);

    while (1)
    {
        PRINTF("Please make sure LPC800 has entered ISP mode!!\r\n");
        PRINTF("Please input a character to erase or program LPC800, 'E' means erase only, other character means erase and program:\r\n");
        ch = GETCHAR();
        PUTCHAR(ch);
        PRINTF("\r\n");

        lpc800_image_align_size = 0;
        p_image_location = (const uint8_t *)lpc860_image;
        target_flash_addr = 0;

        /* Step 1: Sync the baudrate */
        USART_WriteBlocking(DEMO_USART, (const uint8_t *)sync_command, 1);
        USART_ReadBlocking(DEMO_USART, (uint8_t *)command_resp, 14);
        USART_WriteBlocking(DEMO_USART, (const uint8_t *)command_resp, strlen(command_resp));
        memset(command_resp, 0, sizeof(command_resp));
        USART_ReadBlocking(DEMO_USART, (uint8_t *)command_resp, 18);
        if (strcmp(command_resp, sync_response) != 0)
        {
            PRINTF("SYNC ERROR!\r\n");
            continue;
        }
        memset(command_resp, 0, sizeof(command_resp));
        USART_WriteBlocking(DEMO_USART, (const uint8_t *)crystal_freq_command, strlen(crystal_freq_command));
        USART_ReadBlocking(DEMO_USART, (uint8_t *)command_resp, strlen(crystal_freq_response));
        if (strcmp(command_resp, crystal_freq_response) != 0)
        {
            PRINTF("CRYSTAL ERROR!\r\n");
            continue;
        }

        /* Step 2: Unlock */
        memset(command_resp, 0, sizeof(command_resp));
        USART_WriteBlocking(DEMO_USART, (const uint8_t *)unlock_command, strlen(unlock_command));
        USART_ReadBlocking(DEMO_USART, (uint8_t *)command_resp, strlen(unlock_response));
        if (strcmp(command_resp, unlock_response) != 0)
        {
            PRINTF("UNLOCK ERROR!\r\n");
            continue;
        }

        /* Step 3: Full chip prepare for erase */
        memset(command_resp, 0, sizeof(command_resp));
        USART_WriteBlocking(DEMO_USART, (const uint8_t *)full_chip_prepare_command, strlen(full_chip_prepare_command));
        USART_ReadBlocking(DEMO_USART, (uint8_t *)command_resp, strlen(full_chip_prepare_response));
        if (strcmp(command_resp, full_chip_prepare_response) != 0)
        {
            PRINTF("FULL CHIP PREPARE ERROR!\r\n");
            continue;
        }

        /* Step 4: Full chip erase */
        memset(command_resp, 0, sizeof(command_resp));
        USART_WriteBlocking(DEMO_USART, (const uint8_t *)full_chip_erase_command, strlen(full_chip_erase_command));
        USART_ReadBlocking(DEMO_USART, (uint8_t *)command_resp, strlen(full_chip_erase_response));
        if (strcmp(command_resp, full_chip_erase_response) != 0)
        {
            PRINTF("FULL CHIP PREPARE ERROR!\r\n");
            continue;
        }

        if (ch == 'E')
        {
            PRINTF("Erase only, erase done, please re-enter ISP mode for further operation!\r\n");
            PRINTF("\r\n");
            continue;
        }

        /* Step 5: Full chip prepare for program */
        memset(command_resp, 0, sizeof(command_resp));
        USART_WriteBlocking(DEMO_USART, (const uint8_t *)full_chip_prepare_command, strlen(full_chip_prepare_command));
        USART_ReadBlocking(DEMO_USART, (uint8_t *)command_resp, strlen(full_chip_prepare_response));
        if (strcmp(command_resp, full_chip_prepare_response) != 0)
        {
            PRINTF("FULL CHIP PREPARE ERROR!\r\n");
            continue;
        }

        /* Step 6: Echo disable */
        memset(command_resp, 0, sizeof(command_resp));
        USART_WriteBlocking(DEMO_USART, (const uint8_t *)echo_disable_command, strlen(echo_disable_command));
        USART_ReadBlocking(DEMO_USART, (uint8_t *)command_resp, strlen(echo_disable_response));
        if (strcmp(command_resp, echo_disable_response) != 0)
        {
            PRINTF("ECHO ERROR!\r\n");
            continue;
        }

        lpc800_image_align_size = SDK_SIZEALIGN(sizeof(lpc860_image), 1024);

        PRINTF("image size is %d, aligned size is %d \r\n", sizeof(lpc860_image), lpc800_image_align_size);

        for (uint32_t image_sent_size = 0; image_sent_size < lpc800_image_align_size; image_sent_size += 1024)
        {
            /* Step 7: Write the image to LPC845 RAM */
            memset(command_buff, 0, sizeof(command_buff));
            memset(command_resp, 0, sizeof(command_resp));

            sprintf((char *)command_buff, "W %d %d\r\n", 0x10000800,1024);
            USART_WriteBlocking(DEMO_USART, (const uint8_t *)command_buff, strlen(command_buff));
            USART_ReadBlocking(DEMO_USART, (uint8_t *)command_resp, strlen(CMD_SUCCESS));

            if (strcmp(command_resp, CMD_SUCCESS) != 0)
            {
                PRINTF("Write memory command error\r\n");
                break;
            }
            memset(command_resp, 0, sizeof(command_resp));
            memset(lpc800_sector_buff, 0, sizeof(lpc800_sector_buff));
            if (LPC800_IMAGE_SIZE - image_sent_size >= 1024)
            {
                memcpy(lpc800_sector_buff, p_image_location, 1024);
            }
            else
            {
                memcpy(lpc800_sector_buff, p_image_location, sizeof(lpc860_image) - image_sent_size);
            }

            USART_WriteBlocking(DEMO_USART, (const uint8_t *)lpc800_sector_buff, 1024);

            /* Step 8: Copy image to LPC800 flash */
            memset(command_buff, 0, sizeof(command_buff));
            memset(command_resp, 0, sizeof(command_resp));
            sprintf((char *)command_buff, "C %d %d %d\r\n", target_flash_addr, 0x10000800, 1024);
            USART_WriteBlocking(DEMO_USART, (const uint8_t *)command_buff, strlen(command_buff));
            USART_ReadBlocking(DEMO_USART, (uint8_t *)command_resp, strlen(CMD_SUCCESS));
            if (strcmp(command_resp, CMD_SUCCESS) != 0)
            {
                PRINTF("Copy command error\r\n");
                break;
            }
            target_flash_addr += 1024;
            p_image_location += 1024;
        }
        PRINTF("Programming is complete, please re-enter ISP mode for further operation!\r\n\r\n");
    }
}
