/*
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * Copyright 2016-2022 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "fsl_debug_console.h"
#include "pin_mux.h"
#include "board.h"
#include "fsl_pwm.h"
#include "fsl_lpadc.h"
#include "fsl_clock.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/
/* The PWM base address */
#define BOARD_PWM_BASEADDR        PWM1
#define PWM_SRC_CLK_FREQ          CLOCK_GetFreq(kCLOCK_BusClk)
#define DEMO_PWM_FAULT_LEVEL      true
#define APP_DEFAULT_PWM_FREQUENCE (10000UL)
#define PWM_MODULO                (PWM_SRC_CLK_FREQ/APP_DEFAULT_PWM_FREQUENCE)

/* Definition for default PWM frequence in hz. */
#ifndef APP_DEFAULT_PWM_FREQUENCE
#define APP_DEFAULT_PWM_FREQUENCE (1000UL)
#endif

/* Parameter setting for different PWM mode */
#define PWMA_DUTYCYCLE     (60U)      //range: 0-100
#define PWMB_DUTYCYCLE     (40U)      //range: 0-100
#define PWMA_PHASE         (36U)      //range: 0-360
#define PWM_PHASESHIFT     (36U)      //range: 0-360
#define PWMA_RISING        (20U)      //range: 0-100
#define PWMA_FALLING       (60U)      //range: 0-100
#define PWMB_RISING        (40U)      //range: 0-100
#define PWMB_FALLING       (80U)      //range: 0-100
#define PWM_TRIG0          (40U)      //range: 0-100
#define PWM_TRIG1          (60U)      //range: 0-100
#define PWM_DEADTIME       (100U)     //unit:  ns

/* Definition for LPADC */
#define DEMO_LPADC_BASE             ADC0
#define DEMO_LPADC_IRQn             ADC0_IRQn
#define DEMO_LPADC_IRQ_HANDLER_FUNC ADC0_IRQHandler
#define DEMO_LPADC_USER_CHANNEL     2U
#define DEMO_LPADC_USER_CMDID       1U

/* Use VREF_OUT driven from the VREF block as the reference voltage */
#define DEMO_LPADC_VREF_SOURCE           kLPADC_ReferenceVoltageAlt3
#define DEMO_LPADC_DO_OFFSET_CALIBRATION true
#define DEMO_LPADC_OFFSET_VALUE_A        0x10U
#define DEMO_LPADC_OFFSET_VALUE_B        0x10U
#define DEMO_VREF_BASE                   VREF0

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
void PWM_DemoInit(void);
void LPADC_DemoInit(void);
void PWM_ModeSelect(void);


/*******************************************************************************
 * Variables
 ******************************************************************************/
volatile uint8_t g_pwmMode        = 0;
volatile bool    g_dblen_bit      = 0;
volatile bool    g_split_bit      = 0;

const uint32_t g_LpadcFullRange   = 4096U;
const uint32_t g_LpadcResultShift = 3U;
volatile bool g_LpadcConversionCompletedFlag = false;
volatile uint32_t g_LpadcInterruptCounter    = 0U;
lpadc_conv_result_t g_LpadcResultConfigStruct;
volatile uint32_t g_systickCounter = 0;

volatile uint8_t s_CmdRecv = 0;

/*******************************************************************************
 * Code
 ******************************************************************************/
/*!
 * @brief Main function
 */
int main(void)
{

    /* attach FRO 12M to FLEXCOMM4 (debug console) */
    CLOCK_SetClkDiv(kCLOCK_DivFlexcom4Clk, 1u);
    CLOCK_AttachClk(BOARD_DEBUG_UART_CLK_ATTACH);

    /* Board pin, clock, debug console init */
    BOARD_InitPins();
    BOARD_PowerMode_OD();
    BOARD_InitBootClocks();
    BOARD_InitDebugConsole();

    /* PWM mode select using keyboard input */
    PWM_ModeSelect();

    /* Call the PWM and LPADC init function */
    PWM_DemoInit();
    LPADC_DemoInit();

    /* Set systick reload value to generate 1ms interrupt */
    if (SysTick_Config(SystemCoreClock / 1000U))
    {
        while (1)
        {
        }
    }

    PRINTF("\r\nPWM Start\r\n");

    while (1U)
    {
    	while (!g_LpadcConversionCompletedFlag)
        {
        }
    	if((g_systickCounter%1000 == 0) && (g_pwmMode == 4))
    	{
            PRINTF("ADC value: %d\r\nADC interrupt count: %d\r\n",
                  ((g_LpadcResultConfigStruct.convValue) >> g_LpadcResultShift), g_LpadcInterruptCounter);
    	}
        g_LpadcConversionCompletedFlag = false;
    }
}

void PWM_DemoInit(void)
{
    /* Structure of initialize PWM */
    pwm_config_t pwmConfig;
    pwm_fault_param_t faultConfig;

    /* Parameter set for different PWM mode */
    uint16_t dutyCyclePercent[2];
	uint32_t deadTimeVal;
    uint32_t pwmHighPulse[2];
    uint32_t pwmTrigger[2];
    uint32_t phaseA;
    uint32_t phaseShift;
    uint32_t risingA, fallingA, risingB, fallingB;
    uint32_t pwmSourceClockInHz;
    uint32_t pwmFrequencyInHz = APP_DEFAULT_PWM_FREQUENCE;

    /* Parameter calculating for different PWM mode */
    pwmSourceClockInHz = PWM_SRC_CLK_FREQ;                                // PWM source clock
    dutyCyclePercent[0] = PWMA_DUTYCYCLE;                                 // PWM_A dutycycle
    dutyCyclePercent[1] = PWMB_DUTYCYCLE;                                 // PWM_B dutycycle
    pwmHighPulse[0] = PWM_MODULO * dutyCyclePercent[0] / 100UL;           // PWM counter value for PWM_A
    pwmHighPulse[1] = PWM_MODULO * dutyCyclePercent[1] / 100UL;           // PWM counter value for PWM_B
    phaseA = PWMA_PHASE * PWM_MODULO / 360UL;                             // Phase for PWM_A
    phaseShift = PWM_PHASESHIFT * PWM_MODULO / 360UL;                     // Phase shift for PWM_B
    risingA  = PWMA_RISING  * PWM_MODULO / 100UL;                         // Rising  value for PWM_A
    fallingA = PWMA_FALLING * PWM_MODULO / 100UL;                         // Falling value for PWM_A
    risingB  = PWMB_RISING  * PWM_MODULO / 100UL;                         // Rising  value for PWM_B
    fallingB = PWMB_FALLING * PWM_MODULO / 100UL;                         // Falling value for PWM_B
    pwmTrigger[0] = PWM_TRIG0 * PWM_MODULO / 100UL;                       // Trigger0 value set
    pwmTrigger[1] = PWM_TRIG1 * PWM_MODULO / 100UL;                       // Trigger1 value set
    deadTimeVal = PWM_DEADTIME * (PWM_SRC_CLK_FREQ / 1000000UL) / 1000UL; // Deadtime value set

    /* Enable PWM1 SUB Clockn */
    SYSCON->PWM1SUBCTL |= SYSCON_PWM1SUBCTL_CLK0_EN_MASK;
    /* Enable interface clock of PWM1*/
    SYSCON->AHBCLKCTRLSET[3] |= SYSCON_AHBCLKCTRL3_PWM1_MASK;
    /*
     * pwmConfig.enableDebugMode = false;
     * pwmConfig.enableWait = false;
     * pwmConfig.reloadSelect = kPWM_LocalReload;
     * pwmConfig.clockSource = kPWM_BusClock;
     * pwmConfig.prescale = kPWM_Prescale_Divide_1;
     * pwmConfig.initializationControl = kPWM_Initialize_LocalSync;
     * pwmConfig.forceTrigger = kPWM_Force_Local;
     * pwmConfig.reloadFrequency = kPWM_LoadEveryOportunity;
     * pwmConfig.reloadLogic = kPWM_ReloadImmediate;
     * pwmConfig.pairOperation = kPWM_Independent;
     */
    PWM_GetDefaultConfig(&pwmConfig);

    /* Use full cycle reload */
    pwmConfig.reloadLogic = kPWM_ReloadPwmFullCycle;
    /* Enable debug mode, support debug PWM output */
    pwmConfig.enableDebugMode = true;
    /* In adc triggering demo, PWM in ComplementaryPwmA pair operation */
    if(g_pwmMode == 4)
    {
    	pwmConfig.pairOperation = kPWM_ComplementaryPwmA;
    }

    /* Initialize submodule 0 */
    if (PWM_Init(BOARD_PWM_BASEADDR, kPWM_Module_0, &pwmConfig) == kStatus_Fail)
    {
        PRINTF("PWM initialization failed\n");
    }

    /*
     *   config->faultClearingMode = kPWM_Automatic;
     *   config->faultLevel = false;
     *   config->enableCombinationalPath = true;
     *   config->recoverMode = kPWM_NoRecovery;
     */
    PWM_FaultDefaultConfig(&faultConfig);

#ifdef DEMO_PWM_FAULT_LEVEL
    faultConfig.faultLevel = DEMO_PWM_FAULT_LEVEL;
#endif
    BOARD_PWM_BASEADDR -> SM[0].CAPTCOMPB  = PWM_CAPTCOMPB_EDGCMPB(2U);     //Set the compare value associated with the edge counter for the PWM_B input capture
    /* Sets up the PWM fault protection */
    PWM_SetupFaults(BOARD_PWM_BASEADDR, kPWM_Fault_0, &faultConfig);
    PWM_SetupFaults(BOARD_PWM_BASEADDR, kPWM_Fault_1, &faultConfig);
    PWM_SetupFaults(BOARD_PWM_BASEADDR, kPWM_Fault_2, &faultConfig);
    PWM_SetupFaults(BOARD_PWM_BASEADDR, kPWM_Fault_3, &faultConfig);

    /* Set PWM fault disable mapping for submodule 0 PWMA and PWMB*/
    PWM_SetupFaultDisableMap(BOARD_PWM_BASEADDR, kPWM_Module_0, kPWM_PwmA, kPWM_faultchannel_0,
                             kPWM_FaultDisable_0 | kPWM_FaultDisable_1 | kPWM_FaultDisable_2 | kPWM_FaultDisable_3);
    PWM_SetupFaultDisableMap(BOARD_PWM_BASEADDR, kPWM_Module_0, kPWM_PwmB, kPWM_faultchannel_0,
                             kPWM_FaultDisable_0 | kPWM_FaultDisable_1 | kPWM_FaultDisable_2 | kPWM_FaultDisable_3);

    /* Parameter setting for INIT, VAL0, VAL1, DTCNT0 and DTCNT1 */
	BOARD_PWM_BASEADDR -> SM[0].INIT = PWM_INIT_INIT((uint16_t)(-(PWM_MODULO / 2)));
	BOARD_PWM_BASEADDR -> SM[0].VAL0 = PWM_VAL0_VAL0((uint16_t)(0U));
	BOARD_PWM_BASEADDR -> SM[0].VAL1 = PWM_VAL1_VAL1((uint16_t)((PWM_MODULO / 2) - 1));

	/* Center aligned PWM */
    if(g_pwmMode == 0)
    {
    	BOARD_PWM_BASEADDR -> SM[0].VAL2 = PWM_VAL2_VAL2((uint16_t)(-(pwmHighPulse[0] / 2)));
    	BOARD_PWM_BASEADDR -> SM[0].VAL3 = PWM_VAL3_VAL3((uint16_t)( (pwmHighPulse[0] / 2)));
    	BOARD_PWM_BASEADDR -> SM[0].VAL4 = PWM_VAL4_VAL4((uint16_t)(-(pwmHighPulse[1] / 2)));
    	BOARD_PWM_BASEADDR -> SM[0].VAL5 = PWM_VAL5_VAL5((uint16_t)( (pwmHighPulse[1] / 2)));
    }
    /* Edge aligned PWM */
    else if(g_pwmMode == 1)
    {
    	BOARD_PWM_BASEADDR -> SM[0].VAL2 = PWM_VAL2_VAL2((uint16_t)(-(PWM_MODULO / 2)));
    	BOARD_PWM_BASEADDR -> SM[0].VAL3 = PWM_VAL3_VAL3((uint16_t)(-(PWM_MODULO / 2) + pwmHighPulse[0]));
    	BOARD_PWM_BASEADDR -> SM[0].VAL4 = PWM_VAL4_VAL4((uint16_t)(-(PWM_MODULO / 2)));
    	BOARD_PWM_BASEADDR -> SM[0].VAL5 = PWM_VAL5_VAL5((uint16_t)(-(PWM_MODULO / 2) + pwmHighPulse[1]));
    }
    /* Phase shifted PWM */
    else if(g_pwmMode == 2)
    {
    	BOARD_PWM_BASEADDR -> SM[0].VAL2 = PWM_VAL2_VAL2((uint16_t)(-(PWM_MODULO / 2) + phaseA));
    	BOARD_PWM_BASEADDR -> SM[0].VAL3 = PWM_VAL3_VAL3((uint16_t)(-(PWM_MODULO / 2) + phaseA + pwmHighPulse[0]));
    	BOARD_PWM_BASEADDR -> SM[0].VAL4 = PWM_VAL4_VAL4((uint16_t)(-(PWM_MODULO / 2) + phaseA + phaseShift));
    	BOARD_PWM_BASEADDR -> SM[0].VAL5 = PWM_VAL5_VAL5((uint16_t)(-(PWM_MODULO / 2) + phaseA + phaseShift + pwmHighPulse[0]));
    }
    /* Double switching PWM */
    else if(g_pwmMode == 3)
    {
    	BOARD_PWM_BASEADDR -> SM[0].VAL2 = PWM_VAL2_VAL2((uint16_t)(-(PWM_MODULO / 2) + risingA));
    	BOARD_PWM_BASEADDR -> SM[0].VAL3 = PWM_VAL3_VAL3((uint16_t)(-(PWM_MODULO / 2) + fallingA));
    	BOARD_PWM_BASEADDR -> SM[0].VAL4 = PWM_VAL4_VAL4((uint16_t)(-(PWM_MODULO / 2) + risingB));
    	BOARD_PWM_BASEADDR -> SM[0].VAL5 = PWM_VAL5_VAL5((uint16_t)(-(PWM_MODULO / 2) + fallingB));
    	if(g_dblen_bit == 1)
    	{
    	    BOARD_PWM_BASEADDR -> SM[0].CTRL |= PWM_CTRL_DBLEN_MASK;
    	}
    	if(g_split_bit == 1)
    	{
    	    BOARD_PWM_BASEADDR -> SM[0].CTRL |= PWM_CTRL_SPLIT_MASK;
    	}
    }
    /* ADC triggering */
    else if(g_pwmMode == 4)
    {
    	BOARD_PWM_BASEADDR -> SM[0].VAL2 = PWM_VAL2_VAL2((uint16_t)(-(pwmHighPulse[0] / 2)));
    	BOARD_PWM_BASEADDR -> SM[0].VAL3 = PWM_VAL3_VAL3((uint16_t)( (pwmHighPulse[0] / 2)));
    	BOARD_PWM_BASEADDR -> SM[0].VAL4 = PWM_VAL4_VAL4((uint16_t)(-(PWM_MODULO / 2) + pwmTrigger[0]));
    	BOARD_PWM_BASEADDR -> SM[0].VAL5 = PWM_VAL5_VAL5((uint16_t)(-(PWM_MODULO / 2) + pwmTrigger[1]));
    	BOARD_PWM_BASEADDR -> SM[0].DTCNT0 = deadTimeVal;
    	BOARD_PWM_BASEADDR -> SM[0].DTCNT1 = deadTimeVal;
    	BOARD_PWM_BASEADDR -> SM[0].TCTRL |= PWM_TCTRL_OUT_TRIG_EN(1 << 4);   //PWM_OUT_TRIG0 will set when the counter value matches the VAL4 value
    	BOARD_PWM_BASEADDR -> SM[0].TCTRL |= PWM_TCTRL_OUT_TRIG_EN(1 << 5);   //PWM_OUT_TRIG1 will set when the counter value matches the VAL5 value
    }
    /* E-Capture */
    else if(g_pwmMode == 5)
    {
    	BOARD_PWM_BASEADDR -> SM[0].VAL2 = PWM_VAL2_VAL2((uint16_t)(-(PWM_MODULO / 2)));
    	BOARD_PWM_BASEADDR -> SM[0].VAL3 = PWM_VAL3_VAL3((uint16_t)(-(PWM_MODULO / 2) + pwmHighPulse[0]));

    	BOARD_PWM_BASEADDR -> SM[0].CAPTCTRLB |= PWM_CAPTCTRLB_INP_SELB(1U);    //0: Raw PWM_B input signal selected as source, 1: Set the output of the edge counter/compare circuitry as the source for the input capture
    	BOARD_PWM_BASEADDR -> SM[0].CAPTCTRLB |= PWM_CAPTCTRLB_EDGB0(2U);       //Capture 0 capture rising edges, only works when INP_SELB = 0
    	BOARD_PWM_BASEADDR -> SM[0].CAPTCTRLB |= PWM_CAPTCTRLB_EDGB1(1U);       //Capture 1 capture falling edges, only works when INP_SELB = 0
    	BOARD_PWM_BASEADDR -> SM[0].CAPTCTRLB |= PWM_CAPTCTRLB_EDGCNTB_EN(1U);  //Edge Counter B Enabled
//    	BOARD_PWM_BASEADDR -> SM[0].CAPTCTRLB |= PWM_CAPTCTRLB_ONESHOTB(1U);    //Enabled one shot mode
    	BOARD_PWM_BASEADDR -> SM[0].CAPTCTRLB |= PWM_CAPTCTRLB_ARMB(1U);        //Input capture operation as specified by CAPTCTRLB[EDGBx] is enabled

    	BOARD_PWM_BASEADDR -> SM[0].CAPTCOMPB  = PWM_CAPTCOMPB_EDGCMPB(5U);     //Set the compare value associated with the edge counter for the PWM_B input capture, works when INP_SELB = 1

    	BOARD_PWM_BASEADDR -> SM[0].INTEN     |= PWM_INTEN_CB0IE(1U);           //Counter B0 interrupt Enabled
    	BOARD_PWM_BASEADDR -> SM[0].INTEN     |= PWM_INTEN_CB1IE(1U);           //Counter B1 interrupt Enabled

    	EnableIRQ(FLEXPWM1_SUBMODULE0_IRQn);
    }
    /* Default: Center aligned PWM */
    else
    {
    	BOARD_PWM_BASEADDR -> SM[0].VAL2 = PWM_VAL2_VAL2((uint16_t)(-(pwmHighPulse[0] / 2)));
    	BOARD_PWM_BASEADDR -> SM[0].VAL3 = PWM_VAL3_VAL3((uint16_t)( (pwmHighPulse[0] / 2)));
    	BOARD_PWM_BASEADDR -> SM[0].VAL4 = PWM_VAL4_VAL4((uint16_t)(-(pwmHighPulse[1] / 2)));
    	BOARD_PWM_BASEADDR -> SM[0].VAL5 = PWM_VAL5_VAL5((uint16_t)( (pwmHighPulse[1] / 2)));
    }

   /* PWMA Output is forced to logic 0 state prior to consideration of output polarity control */
    BOARD_PWM_BASEADDR -> SM[0].OCTRL &= ~((uint16_t)PWM_OCTRL_PWMAFS_MASK);
    BOARD_PWM_BASEADDR -> SM[0].OCTRL |= (((uint16_t)(0u) << (uint16_t)PWM_OCTRL_PWMAFS_SHIFT) &
                                  (uint16_t)PWM_OCTRL_PWMAFS_MASK);

    /* PWMB Output is forced to logic 0 state prior to consideration of output polarity control */
    BOARD_PWM_BASEADDR -> SM[0].OCTRL &= ~((uint16_t)PWM_OCTRL_PWMBFS_MASK);
    BOARD_PWM_BASEADDR -> SM[0].OCTRL |= (((uint16_t)(0u) << (uint16_t)PWM_OCTRL_PWMBFS_SHIFT) &
                                  (uint16_t)PWM_OCTRL_PWMBFS_MASK);

    /* A high level on the PWMA and PWMB pin represents the "on" or "active" state */
    BOARD_PWM_BASEADDR -> SM[0].OCTRL &= ~((uint16_t)1U << (uint16_t)PWM_OCTRL_POLA_SHIFT);
    BOARD_PWM_BASEADDR -> SM[0].OCTRL &= ~((uint16_t)1U << (uint16_t)PWM_OCTRL_POLB_SHIFT);

    /* PWMA and PWMB Output Enable */
    /* In PWM mode 5, PWMA output, PWMB input capture */
    BOARD_PWM_BASEADDR -> OUTEN |= ((uint16_t)1U << ((uint16_t)PWM_OUTEN_PWMA_EN_SHIFT + (uint16_t)(0u)));
    if(g_pwmMode != 5)
    {
        BOARD_PWM_BASEADDR -> OUTEN |= ((uint16_t)1U << ((uint16_t)PWM_OUTEN_PWMB_EN_SHIFT + (uint16_t)(0u)));  // In E-Capture mode, PWM_B set as input
    }

    /* Set the load okay bit for submodule 0 to load registers from their buffer */
    PWM_SetPwmLdok(BOARD_PWM_BASEADDR, kPWM_Control_Module_0, true);

    /* Start the PWM generation from Submodule 0 */
    PWM_StartTimer(BOARD_PWM_BASEADDR, kPWM_Control_Module_0);
}

void LPADC_DemoInit(void)
{
    /* Structure of initialize LPADC */
    lpadc_config_t mLpadcConfigStruct;
    lpadc_conv_trigger_config_t mLpadcTriggerConfigStruct;
    lpadc_conv_command_config_t mLpadcCommandConfigStruct;

    /* attach FRO HF to ADC0 */
    CLOCK_SetClkDiv(kCLOCK_DivAdc0Clk, 1u);
    CLOCK_AttachClk(kFRO_HF_to_ADC0);

    /* enable VREF */
    SPC0->ACTIVE_CFG1 |= 0x1;

    LPADC_GetDefaultConfig(&mLpadcConfigStruct);
    mLpadcConfigStruct.enableAnalogPreliminary = true;
#if defined(DEMO_LPADC_VREF_SOURCE)
    mLpadcConfigStruct.referenceVoltageSource = DEMO_LPADC_VREF_SOURCE;
#endif /* DEMO_LPADC_VREF_SOURCE */
#if defined(FSL_FEATURE_LPADC_HAS_CTRL_CAL_AVGS) && FSL_FEATURE_LPADC_HAS_CTRL_CAL_AVGS
    mLpadcConfigStruct.conversionAverageMode = kLPADC_ConversionAverage1;
#endif /* FSL_FEATURE_LPADC_HAS_CTRL_CAL_AVGS */
    mLpadcConfigStruct.conversionAverageMode = kLPADC_ConversionAverage1;
    LPADC_Init(DEMO_LPADC_BASE, &mLpadcConfigStruct);

#if defined(FSL_FEATURE_LPADC_HAS_CTRL_CALOFS) && FSL_FEATURE_LPADC_HAS_CTRL_CALOFS
#if defined(FSL_FEATURE_LPADC_HAS_OFSTRIM) && FSL_FEATURE_LPADC_HAS_OFSTRIM
    /* Request offset calibration. */
#if defined(DEMO_LPADC_DO_OFFSET_CALIBRATION) && DEMO_LPADC_DO_OFFSET_CALIBRATION
    LPADC_DoOffsetCalibration(DEMO_LPADC_BASE);
#else
    LPADC_SetOffsetValue(DEMO_LPADC_BASE, DEMO_LPADC_OFFSET_VALUE_A, DEMO_LPADC_OFFSET_VALUE_B);
#endif /* DEMO_LPADC_DO_OFFSET_CALIBRATION */
#endif /* FSL_FEATURE_LPADC_HAS_OFSTRIM */

#if defined(FSL_FEATURE_LPADC_HAS_CTRL_CALOFSMODE) && FSL_FEATURE_LPADC_HAS_CTRL_CALOFSMODE
    LPADC_SetOffsetCalibrationMode(DEMO_LPADC_BASE, DEMO_LPADC_OFFSET_CALIBRATION_MODE);
    LPADC_DoOffsetCalibration(DEMO_LPADC_BASE);
#endif /* FSL_FEATURE_LPADC_HAS_CTRL_CALOFSMODE */
    /* Request gain calibration. */
    LPADC_DoAutoCalibration(DEMO_LPADC_BASE);
#endif /* FSL_FEATURE_LPADC_HAS_CTRL_CALOFS */

#if (defined(FSL_FEATURE_LPADC_HAS_CFG_CALOFS) && FSL_FEATURE_LPADC_HAS_CFG_CALOFS)
    /* Do auto calibration. */
    LPADC_DoAutoCalibration(DEMO_LPADC_BASE);
#endif /* FSL_FEATURE_LPADC_HAS_CFG_CALOFS */

    /* Set conversion CMD configuration. */
    LPADC_GetDefaultConvCommandConfig(&mLpadcCommandConfigStruct);
    mLpadcCommandConfigStruct.channelNumber = DEMO_LPADC_USER_CHANNEL;
#if defined(DEMO_LPADC_USE_HIGH_RESOLUTION) && DEMO_LPADC_USE_HIGH_RESOLUTION
    mLpadcCommandConfigStruct.conversionResolutionMode = kLPADC_ConversionResolutionHigh;
#endif /* DEMO_LPADC_USE_HIGH_RESOLUTION */
    LPADC_SetConvCommandConfig(DEMO_LPADC_BASE, DEMO_LPADC_USER_CMDID, &mLpadcCommandConfigStruct);

    /* Set trigger configuration. */
    LPADC_GetDefaultConvTriggerConfig(&mLpadcTriggerConfigStruct);
    mLpadcTriggerConfigStruct.targetCommandId       = DEMO_LPADC_USER_CMDID;
    mLpadcTriggerConfigStruct.enableHardwareTrigger = true;
    LPADC_SetConvTriggerConfig(DEMO_LPADC_BASE, 0U, &mLpadcTriggerConfigStruct); /* Configure the trigger0. */
    LPADC_SetConvTriggerConfig(DEMO_LPADC_BASE, 1U, &mLpadcTriggerConfigStruct); /* Configure the trigger1. */

    CLOCK_EnableClock(kCLOCK_InputMux);                                          // Enable inputmux clock
    INPUTMUX -> ADC0_TRIG[0] = INPUTMUX_ADC0_TRIGM_ADC0_TRIG_TRIGIN(32U);        // ADC0 trigger 0 <- PWM1_SM0_MUX_TRIG0
    INPUTMUX -> ADC0_TRIG[1] = INPUTMUX_ADC0_TRIGM_ADC0_TRIG_TRIGIN(33U);        // ADC0 trigger 1 <- PWM1_SM0_MUX_TRIG1

    /* Enable the watermark interrupt. */
    #if (defined(FSL_FEATURE_LPADC_FIFO_COUNT) && (FSL_FEATURE_LPADC_FIFO_COUNT == 2U))
        LPADC_EnableInterrupts(DEMO_LPADC_BASE, ADC_IE_TCOMP_IE(1U | (1U << 1U)));           /* Enable trigger0 and trigger1 interrupt */
    #else
        LPADC_EnableInterrupts(DEMO_LPADC_BASE, kLPADC_FIFOWatermarkInterruptEnable);
    #endif /* FSL_FEATURE_LPADC_FIFO_COUNT */

    EnableIRQ(DEMO_LPADC_IRQn);

    PRINTF("ADC Full Range: %d\r\n", g_LpadcFullRange);
#if defined(FSL_FEATURE_LPADC_HAS_CMDL_CSCALE) && FSL_FEATURE_LPADC_HAS_CMDL_CSCALE
    if (kLPADC_SampleFullScale == mLpadcCommandConfigStruct.sampleScaleMode)
    {
        PRINTF("Full channel scale (Factor of 1).\r\n");
    }
    else if (kLPADC_SamplePartScale == mLpadcCommandConfigStruct.sampleScaleMode)
    {
        PRINTF("Divided input voltage signal. (Factor of 30/64).\r\n");
    }
#endif

    /* When the number of datawords stored in the ADC Result FIFO is greater
     * than watermark value(0U), LPADC watermark interrupt would be triggered.
     */
}

void PWM_ModeSelect()
{
    /* PWM mode select using C */
    PRINTF("\r\n***FlexPWM driver example***\r\n");
    PRINTF("\r\nPlease Select PWM mode by input 0/1/2/...\r\n");
    PRINTF("0 - Center aligned PWM   \r\n");
    PRINTF("1 - Edge aligned PWM     \r\n");
    PRINTF("2 - Phase shifted PWM    \r\n");
    PRINTF("3 - Double switching PWM \r\n");
    PRINTF("4 - ADC triggering       \r\n");
    PRINTF("5 - E-Capture            \r\n");

    /* Waiting for keyboard input */
    s_CmdRecv = GETCHAR();
    if(s_CmdRecv == '0')
    {
    	g_pwmMode = 0;
    	PRINTF("PWM mode 0: Center aligned PWM\r\n");
    }
    else if(s_CmdRecv == '1')
    {
    	g_pwmMode = 1;
    	PRINTF("PWM mode 1: Edge aligned PWM\r\n");
    }
    else if(s_CmdRecv == '2')
    {
    	g_pwmMode = 2;
    	PRINTF("PWM mode 2: Phase shifted PWM\r\n");
    }
    else if(s_CmdRecv == '3')
    {
    	g_pwmMode = 3;
    	PRINTF("PWM mode 3: Double switching PWM\r\n");
    	PRINTF("DBLEN: 0 or 1 \r\n");
    	s_CmdRecv = GETCHAR();
    	if(s_CmdRecv == '1')
    	{
    	    g_dblen_bit = 1;
    	    PRINTF("\r\nDBLEN ON\r\n");
    	}
    	else
    	{
    		PRINTF("\r\nDBLEN OFF\r\n");
    	}
    	PRINTF("SPLIT: 0 or 1 \r\n");
    	s_CmdRecv = GETCHAR();
    	if(s_CmdRecv == '1')
    	{
    		g_split_bit = 1;
    		PRINTF("\r\nSPLIT ON\r\n");
    	}
    	else
    	{
    		PRINTF("\r\nSPLIT OFF\r\n");
    	}
    }
    else if(s_CmdRecv == '4')
    {
    	g_pwmMode = 4;
    	PRINTF("PWM mode 4: ADC triggering\r\n");
    }
    else if(s_CmdRecv == '5')
    {
    	g_pwmMode = 5;
    	PRINTF("PWM mode 5: E-Capture\r\n");
    }
    else
    {
    	g_pwmMode = 0;
    	PRINTF("Input code error, run in default mode\r\n");
    	PRINTF("PWM mode 0: Center aligned PWM\r\n");
    }
}

void DEMO_LPADC_IRQ_HANDLER_FUNC(void)
{
    GPIO_PortToggle(GPIO2, 1U << 4U);
	g_LpadcInterruptCounter++;
#if (defined(FSL_FEATURE_LPADC_FIFO_COUNT) && (FSL_FEATURE_LPADC_FIFO_COUNT == 2U))
    if (LPADC_GetConvResult(DEMO_LPADC_BASE, &g_LpadcResultConfigStruct, 0U))
#else
    if (LPADC_GetConvResult(DEMO_LPADC_BASE, &g_LpadcResultConfigStruct))
#endif /* FSL_FEATURE_LPADC_FIFO_COUNT */
    {
        g_LpadcConversionCompletedFlag = true;
    }

    DEMO_LPADC_BASE -> STAT |= ADC_STAT_TCOMP_INT_MASK;  // Clear trigger interrupt flag

    SDK_ISR_EXIT_BARRIER;
}

void SysTick_Handler(void)
{
    g_systickCounter++;
}

void FLEXPWM1_SUBMODULE0_IRQHandler(void)
{
    GPIO_PortToggle(GPIO2, 1U << 4U);
    BOARD_PWM_BASEADDR -> SM[0].STS |= PWM_STS_CFB0(1U);              // Clear Capture B0 interrupt flag
    BOARD_PWM_BASEADDR -> SM[0].STS |= PWM_STS_CFB1(1U);              // Clear Capture B1 interrupt flag
//	BOARD_PWM_BASEADDR -> SM[0].CAPTCTRLB |= PWM_CAPTCTRLB_ARMB(1U);  // Re-arm Capture circuit

    SDK_ISR_EXIT_BARRIER;
}
