/*
 * Copyright 2016-2023 NXP
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of NXP Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
 * @brief   Application entry point.
 */
#include <stdio.h>
#include "board.h"
#include "pin_mux.h"
#include "clock_config.h"
#include "fsl_debug_console.h"
#include "fsl_flash.h"
#include "fsl_flash_ffr.h"
#include "fsl_common.h"
//#include "fsl_usart.h"
//#include "fsl_usart.h"

////////////////////////////////////////////////////////////////////////////////
// Definitions
///////////////////////////////////////////////////////////////////////////////
//#define APP_USART_RX_ERROR               kUSART_RxError
#define APP_READ_ONLY_UUID 			     0
#define APP_INCREASE_VERSION_ONLY        1
#define APP_INCREASE_VENDOR_USAGE_ONLY	 2

////////////////////////////////////////////////////////////////////////////////
// Variables
////////////////////////////////////////////////////////////////////////////////
volatile uint32_t g_UUID[4];
uint32_t g_CFPAData[FLASH_FFR_MAX_PAGE_SIZE / sizeof(uint32_t)];
flash_config_t flashInstance;
uint16_t vendorUsage_lower2bytes;
uint16_t vendorUsage_upper2bytes;

////////////////////////////////////////////////////////////////////////////////
// Prototypes
////////////////////////////////////////////////////////////////////////////////
void error_trap(void);
static uint32_t APP_GetUserSelection();
uint32_t ProtectFlashCfg( uint32_t ch);

/*
 * @brief   Application entry point.
 */
int main(void) {

	/* Init board hardware. */
	/* attach 12 MHz clock to FLEXCOMM0 (debug console) */
	CLOCK_AttachClk(BOARD_DEBUG_UART_CLK_ATTACH);

    BOARD_InitPins();
	BOARD_BootClockPLL150M();

#ifndef BOARD_INIT_DEBUG_CONSOLE_PERIPHERAL
	/* Init FSL debug console. */
	BOARD_InitDebugConsole();
#endif

	/* Print basic information for Flash Driver API.*/
	PRINTF("\r\nMCX-N9XX-EVK Flash Read and Write example...\r\n");

	PRINTF("\r\nWrite in the console one of the following options...\r\n");
	PRINTF("\r\n 0 to READ  UUID and CFPA page ...\r\n");
	PRINTF("\r\n 1 to Increase Version Only ...\r\n");
	PRINTF("\r\n 2 to Increase Vendor Usage...\r\n");
	PRINTF("\r\n\r\n");

	volatile uint32_t usr_cmd ;
	while(1) {
		/* wait for commands from the console*/
		usr_cmd = APP_GetUserSelection();
		/* process the user command**/
		ProtectFlashCfg( usr_cmd);
	}

	/** program should never reach this line*/
	return 0 ;
}

/*
 * @brief Gets called when an error occurs.
 */
void error_trap(void)
{
	PRINTF("\r\n\r\n\r\n\t---- HALTED DUE TO FLASH ERROR! ----");
	while (1);
}

static uint32_t APP_GetUserSelection()
{
	uint32_t ch;
#if 0
	/* Clear rx overflow error once it happens during low power mode. */
	if (APP_USART_RX_ERROR == (APP_USART_RX_ERROR & USART_GetStatusFlags((USART_Type *)BOARD_DEBUG_UART_BASEADDR)))
	{
		USART_ClearStatusFlags((USART_Type *)BOARD_DEBUG_UART_BASEADDR, APP_USART_RX_ERROR);
	}
#endif
	/**  usr cannot exit this get char loop until he gives a correct option.*/
	while(1)
	{
		ch = GETCHAR();
		if ((ch < '0') || (ch > '2')) /* Only '0', '1', '2' */
		{
			continue;
		}
		else
		{
			ch = ch - '0'; /* Only '0', '1', '2'. */
			break;
		}
	}

	switch (ch)
	{
	case 0:
		ch = APP_READ_ONLY_UUID;
		PRINTF("\r\n %c \r\n", ch + '0');
		break;
	case 1:
		ch = APP_INCREASE_VERSION_ONLY;
		PRINTF("\r\n %c \r\n", ch+ '0');
		break;
	case 2:
		ch = APP_INCREASE_VENDOR_USAGE_ONLY;
		PRINTF("\r\n %c \r\n", ch+ '0');
		break;
	}
	return ch;
}

uint32_t ProtectFlashCfg( uint32_t ch)
{
	int32_t kCommandIndex = ch;
	static uint32_t status;

	switch (kCommandIndex)
	{
	/* 0, Read chip UUID and CFPA page data */
	case APP_READ_ONLY_UUID:
		PRINTF("Read CHIP UUID and read CFPA...\r\n");
		/* Initialize flash driver */
		FLASH_Init(&flashInstance);
		if (FFR_Init(&flashInstance) == kStatus_Success)
		{
			PRINTF("Flash init successfull!!. Halting...\r\n");
		}
		else
		{
			error_trap();
		}

		status =  FFR_GetUUID(&flashInstance, (uint8_t *)g_UUID);
		PRINTF("\nUUID 0x%8x 0x%8x 0x%8x 0x%8x\r\n", g_UUID[0], g_UUID[1], g_UUID[2], g_UUID[3]);

		/* Read data stored in the Customer Factory page */
		status = FFR_GetCustomerInfieldData(&flashInstance, (uint8_t *)g_CFPAData, 0x0, FLASH_FFR_MAX_PAGE_SIZE);

		PRINTF("\r\nCFPA page data:\r\n");
		PRINTF("Header 0x%08x ,          Version  0x%08x ,   SecureFW Version 0x%08x , NonSecureFW Version    0x%08x\r\n", g_CFPAData[0], g_CFPAData[1], g_CFPAData[2], g_CFPAData[3]);
		PRINTF("ImageKey Revoke 0x%08x , RothRevoke 0x%08x , Vendor Usage   0x%08x\r\n", g_CFPAData[4], g_CFPAData[20], g_CFPAData[21]);
		PRINTF("NS PIN 0x%08x ,          NS DFLT  0x%08x ,   Enable FA Mode 0x%08x\r\n", g_CFPAData[22], g_CFPAData[23], g_CFPAData[24]);
		PRINTF("\r\n");

		PRINTF("\r\nMCX-N9XX-EVK Flash Read and Write example...\r\n");

		PRINTF("\r\nWrite in the console one of the following options...\r\n");
		PRINTF("\r\n 0 to READ  UUID and CFPA page ...\r\n");
		PRINTF("\r\n 1 to Increase Version Only ...\r\n");
		PRINTF("\r\n 2 to Increase Vendor Usage...\r\n");
		PRINTF("\r\n\r\n");
		break;

	/* 1, Increase CFPA page version */
	case APP_INCREASE_VERSION_ONLY:

		/* Initialize flash driver */
		FLASH_Init(&flashInstance);

		if (FFR_Init(&flashInstance) == kStatus_Success)
		{
			PRINTF("Flash init successfull!!. Halting...\r\n");
		}
		else
		{
			error_trap();
		}

		status = FFR_GetCustomerInfieldData(&flashInstance, (uint8_t *)g_CFPAData, 0x0, FLASH_FFR_MAX_PAGE_SIZE);

		/* Clean-up CFPA area */
		g_CFPAData[offsetof(cfpa_cfg_info_t, dcfgNsPin)]  = 0;
		g_CFPAData[offsetof(cfpa_cfg_info_t, dcfgNsDflt)] = 0;

		/*Increase Monotonic counter*/
		uint32_t *p32    = (uint32_t *)(uint32_t)g_CFPAData;
		uint32_t version = p32[1];
		uint32_t vendor_usage = p32[21];

		if (version == 0xFFFFFFFFu)
		{
			return kStatus_Fail;
		}
		version++;
		p32[1] = version;

		PRINTF("\nVersion to write: 0x%08x \r\n", version);

		status_t markStatus = FFR_InfieldPageWrite(&flashInstance, (uint8_t *)g_CFPAData, FLASH_FFR_MAX_PAGE_SIZE);
		if (kStatus_FLASH_Success == markStatus)
		{
			status = kStatus_Success;
			PRINTF("CFPA Write Done!\r\n");
		}
		else
		{
			status = kStatus_Fail;
			PRINTF("CFPA Write Failed!\r\n");
		}

		/* Core reset to allow updating the main CFPA page */
		NVIC_SystemReset();
		break;

	/* 2, Increase Vendor usage field */
	case APP_INCREASE_VENDOR_USAGE_ONLY:

		/* Initialize flash driver */
		FLASH_Init(&flashInstance);

		if (FFR_Init(&flashInstance) == kStatus_Success)
		{
			PRINTF("Flash init successfull!!. Halting...\r\n");
		}
		else
		{
			error_trap();
		}

		status = FFR_GetCustomerInfieldData(&flashInstance, (uint8_t *)g_CFPAData, 0x0, FLASH_FFR_MAX_PAGE_SIZE);

		/* Clean-up CFPA area */
		g_CFPAData[offsetof(cfpa_cfg_info_t, dcfgNsPin)]  = 0;
		g_CFPAData[offsetof(cfpa_cfg_info_t, dcfgNsDflt)] = 0;

		/*Increase Monotonic counter*/
		p32    = (uint32_t *)(uint32_t)g_CFPAData;
		version = p32[1];

		if (version == 0xFFFFFFFFu)
		{
			return kStatus_Fail;
		}
		version++;
		p32[1] = version;

		PRINTF("\nCFPA page version to write: 0x%08x \r\n", version);

		/* Increase Vendor usage; the lower 2 bytes of VENDOR_USAGE field contain value of DBG_VENDOR_USAGE[15:0] which is a monotonic counter
		   and upper 2 bytes contain the inverse value of the lower 2 bytes */
		vendor_usage = p32[21];
		if(vendor_usage == 0x0)
		{
			vendorUsage_lower2bytes = 0xFFFF;
			vendorUsage_upper2bytes  = 0x0000;
		}
		else
		{
			vendorUsage_lower2bytes = vendor_usage & 0xFFFF;
			vendorUsage_upper2bytes = vendor_usage >> 16;
		}

		vendorUsage_upper2bytes -= 0x1;
		vendorUsage_lower2bytes += 0x1;
		vendor_usage = (vendorUsage_upper2bytes << 16) | vendorUsage_lower2bytes;
		p32[21] = vendor_usage;

		PRINTF("Vendor_Usage to write: 0x%08x \r\n", vendor_usage);

		markStatus = FFR_InfieldPageWrite(&flashInstance, (uint8_t *)g_CFPAData, FLASH_FFR_MAX_PAGE_SIZE);
		if (kStatus_FLASH_Success == markStatus)
		{
			status = kStatus_Success;
			PRINTF("CFPA Write Done!\r\n");
		}
		else
		{
			status = kStatus_Fail;
			PRINTF("CFPA Write Failed!\r\n");
		}

		/* Core reset to allow updating the main CFPA page */
		NVIC_SystemReset();
		break;

	default:
		PRINTF("Command index is wrong\r\n");
		break;
	}

	PRINTF(" \r\n");
	return 0 ;
}
