/*
 * Copyright 2023 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "fsl_device_registers.h"
#include "fsl_debug_console.h"
#include "pin_mux.h"
#include "clock_config.h"
#include "board.h"
#include "task0.h"
#include "task1.h"
/*******************************************************************************
 * Definitions
 ******************************************************************************/

typedef struct _segment_table_t
{
    uint32_t* ram_addr;
    uint32_t* rom_addr;
    uint32_t size;
} segment_table_t;

typedef enum _segment_index_t
{
    SEGMENT0 = 0,
    SEGMENT1,
    SEGMENTNUM,
} segment_index_t;

#if defined(__CC_ARM) || defined(__ARMCC_VERSION)
extern uint32_t Image$$RW_m_segment0$$Base[];
extern uint32_t Load$$RW_m_segment0$$Base[];
extern uint32_t Image$$RW_m_segment0$$Length[];
extern uint32_t Image$$RW_m_segment1$$Base[];
extern uint32_t Load$$RW_m_segment1$$Base[];
extern uint32_t Image$$RW_m_segment1$$Length[];
#define SEGMENT0_RAM_ADDR Image$$RW_m_segment0$$Base
#define SEGMENT0_ROM_ADDR Load$$RW_m_segment0$$Base
#define SEGMENT0_SIZE     (uint32_t)Image$$RW_m_segment0$$Length
#define SEGMENT1_RAM_ADDR Image$$RW_m_segment1$$Base
#define SEGMENT1_ROM_ADDR Load$$RW_m_segment1$$Base
#define SEGMENT1_SIZE     (uint32_t)Image$$RW_m_segment1$$Length
#elif defined(__MCUXPRESSO)
extern uint32_t __base_SRAM_ITC_cm7[];
extern uint32_t __load_start_segment0[];
extern uint32_t __load_stop_segment0[];
extern uint32_t __load_size_segment0[];
extern uint32_t __load_start_segment1[];
extern uint32_t __load_stop_segment1[];
extern uint32_t __load_size_segment1[];
#define SEGMENT0_RAM_ADDR __base_SRAM_ITC_cm7
#define SEGMENT0_ROM_ADDR __load_start_segment0
#define SEGMENT0_SIZE     (uint32_t)__load_size_segment0
#define SEGMENT1_RAM_ADDR __base_SRAM_ITC_cm7
#define SEGMENT1_ROM_ADDR __load_start_segment1
#define SEGMENT1_SIZE     (uint32_t)__load_size_segment1
#elif defined(__ICCARM__) || defined(__GNUC__)
#pragma section = "Overlay"
#pragma section = "segment0_init"
#pragma section = "segment1_init"
#define SEGMENT0_RAM_ADDR __section_begin("Overlay")
#define SEGMENT0_ROM_ADDR __section_begin("segment0_init")
#define SEGMENT0_SIZE     __section_size ("segment0_init")
#define SEGMENT1_RAM_ADDR __section_begin("Overlay")
#define SEGMENT1_ROM_ADDR __section_begin("segment1_init")
#define SEGMENT1_SIZE     __section_size ("segment1_init")
#endif

segment_table_t segment_table[SEGMENTNUM] = 
{
    {SEGMENT0_RAM_ADDR, SEGMENT0_ROM_ADDR, SEGMENT0_SIZE},
    {SEGMENT1_RAM_ADDR, SEGMENT1_ROM_ADDR, SEGMENT1_SIZE},
};

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Code
 ******************************************************************************/

static void load_code(segment_index_t index)
{
    memcpy(segment_table[index].ram_addr, segment_table[index].rom_addr, segment_table[index].size);
    __DSB();
    __ISB();
}

AT_QUICKACCESS_SECTION_CODE(void task2(void))
{
    static uint32_t count;
    PRINTF("task2 at %p (count = %d)\r\n", &task2, count++);
}

/*!
 * @brief Main function
 */
int main(void)
{
    /* Init board hardware. */
    BOARD_ConfigMPU();
    BOARD_InitPins();
    BOARD_BootClockRUN();
    BOARD_InitDebugConsole();

    while (1)
    {
        load_code(SEGMENT0);    // Dynamically load code in SEGMENT0
    	task0();
        load_code(SEGMENT1);    // Dynamically load code in SEGMENT1
    	task1();
        task2();
        PRINTF(" Press any key to start again.\r\n\r\n");
        GETCHAR();
    }
}
