/*
 * Copyright 2018-2020, 2024 NXP
 * SPDX-License-Identifier: BSD-3-Clause
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms.  By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms.  If you do not agree to
 * be bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#include <stdbool.h>
#include "fsl_clock.h"
#include "fsl_gpio.h"
#include "fsl_iocon.h"
#include "pin_mux.h"

#define USB_AUDIO_LATENCY_DEBUG_ENABLE   1
   
/* LPC55S69 EVK P18_9 */
#define AUDIO_TX_GPIO_PORT        1
#define AUDIO_TX_GPIO_PIN         6u
#define AUDIO_TX_GPIO_CLOCK  kCLOCK_Gpio1

/* LPC55S69 EVK P18_3 */
#define AUDIO_RX_GPIO_PORT        1
#define AUDIO_RX_GPIO_PIN        10u
#define AUDIO_RX_GPIO_CLOCK  kCLOCK_Gpio1

/* LPC55S69 EVK P18_5 */
#define USB_TX_GPIO_PORT        1
#define USB_TX_GPIO_PIN         4u
#define USB_TX_GPIO_CLOCK  kCLOCK_Gpio1

/* LPC55S69 EVK P18_7 */
#define USB_RX_GPIO_PORT         1
#define USB_RX_GPIO_PIN          7u
#define USB_RX_GPIO_CLOCK  kCLOCK_Gpio1
   
#if USB_AUDIO_LATENCY_DEBUG_ENABLE == 1

void AUDIO_LatencyDebug_Init(void)
{
    gpio_pin_config_t gpio_config = {
        kGPIO_DigitalOutput,
        0,
    };

    CLOCK_EnableClock(AUDIO_TX_GPIO_CLOCK);
    CLOCK_EnableClock(AUDIO_RX_GPIO_CLOCK);

    GPIO_PinInit(GPIO, AUDIO_TX_GPIO_PORT, AUDIO_TX_GPIO_PIN, &gpio_config);
    GPIO_PinInit(GPIO, AUDIO_RX_GPIO_PORT, AUDIO_RX_GPIO_PIN, &gpio_config);
}

void AUDIO_LatencyDebug_SendingAudioToI2s(bool sending)
{
	GPIO_PinWrite(GPIO, AUDIO_TX_GPIO_PORT, AUDIO_TX_GPIO_PIN, (uint8_t)sending);
}

void AUDIO_LatencyDebug_ReceivingAudioFromI2s(bool receiving)
{
	GPIO_PinWrite(GPIO, AUDIO_RX_GPIO_PORT, AUDIO_RX_GPIO_PIN, (uint8_t)receiving);
}

void AUDIO_LatencyDebug_Deinit(void)
{
    IOCON_PinMuxSet(IOCON, AUDIO_TX_GPIO_PORT,  AUDIO_TX_GPIO_PIN, IOCON_ANALOG_EN);
    IOCON_PinMuxSet(IOCON, AUDIO_RX_GPIO_PORT,  AUDIO_RX_GPIO_PIN, IOCON_ANALOG_EN);
}


void USB_LatencyDebug_Init(void)
{
    gpio_pin_config_t gpio_config = {
        kGPIO_DigitalOutput,
        0,
    };
    
    CLOCK_EnableClock(USB_RX_GPIO_CLOCK);
    CLOCK_EnableClock(USB_TX_GPIO_CLOCK);
        
    IOCON_PinMuxSet(IOCON, USB_TX_GPIO_PORT, USB_TX_GPIO_PIN, IOCON_DIGITAL_EN);    
    GPIO_PinInit(GPIO, USB_TX_GPIO_PORT, USB_TX_GPIO_PIN, &gpio_config);
    
    IOCON_PinMuxSet(IOCON, USB_RX_GPIO_PORT, USB_RX_GPIO_PIN, IOCON_DIGITAL_EN);    
    GPIO_PinInit(GPIO, USB_RX_GPIO_PORT, USB_RX_GPIO_PIN, &gpio_config);

}

void USB_LatencyDebug_ReceivingAudioFromHost(bool receiving)
{
    GPIO_PinWrite(GPIO, USB_RX_GPIO_PORT, USB_RX_GPIO_PIN, (uint8_t)receiving);
}

void USB_LatencyDebug_SendingAudioToHost(bool transmitting)
{
    GPIO_PinWrite(GPIO, USB_TX_GPIO_PORT, USB_TX_GPIO_PIN, (uint8_t)transmitting);
}

void USB_LatencyDebug_Deinit(void)
{
    IOCON_PinMuxSet(IOCON, USB_TX_GPIO_PORT,  USB_TX_GPIO_PIN, IOCON_ANALOG_EN);
    IOCON_PinMuxSet(IOCON, USB_RX_GPIO_PORT,  USB_RX_GPIO_PIN, IOCON_ANALOG_EN);
}

#else 
void AUDIO_LatencyDebug_Init(void)
{
}

void AUDIO_LatencyDebug_SendingAudioToI2s(bool sending)
{
}

void AUDIO_LatencyDebug_ReceivingAudioFromI2s(bool receiving)
{
}

void AUDIO_LatencyDebug_Deinit(void)
{
}


void USB_LatencyDebug_Init(void)
{
}

void USB_LatencyDebug_ReceivingAudioFromHost(bool receiving)
{
}

void USB_LatencyDebug_SendingAudioToHost(bool transmitting)
{
}

void USB_LatencyDebug_Deinit(void)
{
}
#endif
