/* ---------------------------------------------------------------------------- */
/* Copyright 2020 NXP.                                                          */
/*                                                                              */
/* This software is owned or controlled by NXP and may only be used strictly    */
/* in accordance with the applicable license terms. By expressly accepting such */
/* terms or by downloading, installing, activating and/or otherwise using the   */
/* software, you are agreeing that you have read, and that you agree to comply  */
/* with and are bound by, such license terms. If you do not agree to be bound   */
/* by the applicable license terms, then you may not retain, install, activate  */
/* or otherwise use the software.                                               */
/* ---------------------------------------------------------------------------- */
/*
 * The Clear BSD License
 * Copyright 2017 NXP
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer below) provided
 * that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of the copyright holder nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * NO EXPRESS OR IMPLIED LICENSES TO ANY PARTY'S PATENT RIGHTS ARE GRANTED BY THIS LICENSE.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "fsl_device_registers.h"
#include "board.h"
#include <stdio.h>
#include <stdlib.h>
#include "usb_device_config.h"

#if ((defined(USB_DEVICE_CONFIG_CDC_ACM)) && (USB_DEVICE_CONFIG_CDC_ACM > 0U))
    #include "virtual_com.h"
    #include "app_defines.h"
    #include "usb.h"
    #include "usb_device.h"
    #include "usb_device_class.h"
    #include "usb_device_cdc_acm.h"
    #include "usb_device_ch9.h"
    #include "fsl_debug_console.h"
    #include "usb_app_device_descriptor.h"
    #include "composite.h"
    #include "framework.h"
    #include "usb_common.h"
    #include "usb_log.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/

/* Line coding of cdc device */
USB_DMA_INIT_DATA_ALIGN(USB_DATA_ALIGN_SIZE) static uint8_t s_lineCoding[USB_DEVICE_CONFIG_CDC_ACM][LINE_CODING_SIZE] = {
    {   /* E.g. 0x00,0xC2,0x01,0x00 : 0x0001C200 is 115200 bits per second */
        (LINE_CODING_DTERATE >> 0U) & 0x000000FFU, (LINE_CODING_DTERATE >> 8U) & 0x000000FFU,
        (LINE_CODING_DTERATE >> 16U) & 0x000000FFU, (LINE_CODING_DTERATE >> 24U) & 0x000000FFU, LINE_CODING_CHARFORMAT,
        LINE_CODING_PARITYTYPE, LINE_CODING_DATABITS
    },
    #if SERIAL_PORT_TYPE_USBCDC
    {   /* E.g. 0x00,0xC2,0x01,0x00 : 0x0001C200 is 115200 bits per second */
        (LINE_CODING_DTERATE >> 0U) & 0x000000FFU, (LINE_CODING_DTERATE >> 8U) & 0x000000FFU,
        (LINE_CODING_DTERATE >> 16U) & 0x000000FFU, (LINE_CODING_DTERATE >> 24U) & 0x000000FFU, LINE_CODING_CHARFORMAT,
        LINE_CODING_PARITYTYPE, LINE_CODING_DATABITS
    },
    #endif /* SERIAL_PORT_TYPE_USBCDC */
};

/* Abstract state of cdc device */
USB_DMA_INIT_DATA_ALIGN(USB_DATA_ALIGN_SIZE) static uint8_t s_abstractState[USB_DEVICE_CONFIG_CDC_ACM][COMM_FEATURE_DATA_SIZE] = {
    { (STATUS_ABSTRACT_STATE >> 0U) & 0x00FFU, (STATUS_ABSTRACT_STATE >> 8U) & 0x00FFU },
    #if SERIAL_PORT_TYPE_USBCDC
    { (STATUS_ABSTRACT_STATE >> 0U) & 0x00FFU, (STATUS_ABSTRACT_STATE >> 8U) & 0x00FFU },
    #endif /* SERIAL_PORT_TYPE_USBCDC */
};

/* Country code of cdc device */
USB_DMA_INIT_DATA_ALIGN(USB_DATA_ALIGN_SIZE) static uint8_t s_countryCode[USB_DEVICE_CONFIG_CDC_ACM][COMM_FEATURE_DATA_SIZE] = {
    { (COUNTRY_SETTING >> 0U) & 0x00FFU, (COUNTRY_SETTING >> 8U) & 0x00FFU },
    #if SERIAL_PORT_TYPE_USBCDC
    { (COUNTRY_SETTING >> 0U) & 0x00FFU, (COUNTRY_SETTING >> 8U) & 0x00FFU },
    #endif /* SERIAL_PORT_TYPE_USBCDC */
};

/* CDC ACM information */
USB_DMA_INIT_DATA_ALIGN(USB_DATA_ALIGN_SIZE) static usb_cdc_acm_info_t s_usbCdcAcmInfo[USB_DEVICE_CONFIG_CDC_ACM] = {
    { { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 }, 0, 0, 0, 0, 0 },
    #if SERIAL_PORT_TYPE_USBCDC
    { { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 }, 0, 0, 0, 0, 0 },
    #endif
};
/* Data buffer for receiving and sending*/
USB_DMA_NONINIT_DATA_ALIGN(USB_DATA_ALIGN_SIZE) static uint8_t s_currRecvBuf[USB_DEVICE_CONFIG_CDC_ACM][DATA_BUFF_SIZE];

static volatile usb_device_composite_struct_t *g_deviceComposite;
static vcom_ctrl_ctxt_t gs_VcomCtrlCtxt[USB_DEVICE_CONFIG_CDC_ACM];

/*******************************************************************************
 * Code
 ******************************************************************************/

static bool IsVcomHostPresent(vcom_ctrl_ctxt_t *ctrlCtxt)
{
    bool ableToSend = true;
    usb_cdc_acm_info_t *acmInfo = NULL;

    acmInfo = ctrlCtxt->vcomInstance->usbCdcAcmInfo;

    USB_CRITICAL_SECTION({
        ableToSend = acmInfo->dtePresent;
    });

    return ableToSend;
}

static error_t SendOverVcom(vcom_ctrl_ctxt_t *ctrlCtxt, const uint8_t *data, const uint32_t len, const vcom_transfer_complete_cb_t cb)
{
    usb_status_t error = kStatus_USB_Success;

    if (!IsVcomHostPresent(ctrlCtxt)) {
        PERR("No VCOM host present to send data to");
        return kERROR_Fail;
    }

    ctrlCtxt->data = data;
    ctrlCtxt->dataIndex = data;
    ctrlCtxt->toWrite = len;
    ctrlCtxt->cb = cb;
    uint16_t lengthToWrite;

    if (ctrlCtxt->toWrite > ctrlCtxt->vcomInstance->bulkOutEndpointMaxPacketSize) {
        lengthToWrite = ctrlCtxt->vcomInstance->bulkOutEndpointMaxPacketSize;
    } else {
        lengthToWrite = ctrlCtxt->toWrite;
    }

    error = USB_DeviceCdcAcmSend(ctrlCtxt->vcomInstance->cdcAcmHandle,
                                 ctrlCtxt->vcomInstance->bulkInEndpoint,
                                 (uint8_t *)ctrlCtxt->dataIndex,
                                 lengthToWrite);

    if (kStatus_USB_Success != error) {
        PERR("CDC Acm Send Failed !");
        return kERROR_Fail;
    }

    ctrlCtxt->state = kUSB_VcomStateSending;

    return kERROR_Ok;
}

void UsbVcom_Init(void)
{
    memset(&gs_VcomCtrlCtxt, 0, sizeof(gs_VcomCtrlCtxt));
}

    #if USB_HCI_VCOM_ENABLE
error_t UsbVcom_SendHci(const uint8_t *data, const uint32_t len, const vcom_transfer_complete_cb_t cb)
{
    return UsbVcom_Send(USB_VCOM_INSTANCE_HCI, data, len, cb);
}
    #endif

    #if USB_DEVICE_CONFIG_CDC_ACM
error_t UsbVcom_Send(usbVcomInstance_t instance, const uint8_t *data, const uint32_t len, const vcom_transfer_complete_cb_t cb)
{
    if (instance >= USB_DEVICE_CONFIG_CDC_ACM) {
        return kERROR_Fail;
    }

    return SendOverVcom(&gs_VcomCtrlCtxt[instance], data, len, cb);
}
    #endif

static void SendCmplHandler(void *cookie)
{
    vcom_ctrl_ctxt_t *CtrlCtxt = (vcom_ctrl_ctxt_t *)cookie;
    volatile usb_cdc_vcom_struct_t *vcomInstance = NULL;
    usb_status_t error = kStatus_USB_Success;

    assert(CtrlCtxt);
    vcomInstance = CtrlCtxt->vcomInstance;
    assert(vcomInstance);

    if (CtrlCtxt->toWrite) {
        /* Still data to write */
        if (kERROR_Ok != SendOverVcom(CtrlCtxt,
                                      (uint8_t *)CtrlCtxt->dataIndex,
                                      CtrlCtxt->toWrite,
                                      CtrlCtxt->cb)) {
            PERR("Failed to send rest of data over vcom: 0x%x", CtrlCtxt->toWrite);
        }

        /* Data is being transferred, wait for next complete DPC */
        return;
    }

    /* Already open the recv-endpoint before calling user callback */
    error = USB_DeviceCdcAcmRecv(vcomInstance->cdcAcmHandle,
                                 vcomInstance->bulkOutEndpoint,
                                 vcomInstance->currRecvBuf,
                                 vcomInstance->bulkOutEndpointMaxPacketSize);
    if (kStatus_USB_Busy == error) {
        USB_PDBG("recv was still scheduled");
    } else if (kStatus_USB_Success != error) {
        PERR("recv failed in send handler: 0x%x", error);
    }

    if (kStatus_USB_Success != error) {
        CtrlCtxt->state = kUSB_VcomStateReady;
    }

    if (CtrlCtxt->cb) {
        CtrlCtxt->cb();
    }
}

static void DataRecvHandler(void *cookie)
{
    vcom_ctrl_ctxt_t *CtrlCtxt = (vcom_ctrl_ctxt_t *)cookie;
    volatile usb_cdc_vcom_struct_t *vcomInstance = NULL;

    assert(CtrlCtxt);
    vcomInstance = CtrlCtxt->vcomInstance;
    assert(vcomInstance);

    /* Currently receiving */
    CtrlCtxt->state = kUSB_VcomStateReceiving;

    #if USB_HCI_VCOM_ENABLE
    if (vcomInstance == &g_deviceComposite->cdcVcom[USB_VCOM_INSTANCE_HCI]) {
        /* data received from instance 0, this is control data */
        if (CtrlCtxt->rcvdCb) {
            CtrlCtxt->rcvdCb(vcomInstance->currRecvBuf, vcomInstance->recvSize);
        }
    }
    #endif
    #if SERIAL_PORT_TYPE_USBCDC
    if (vcomInstance == &g_deviceComposite->cdcVcom[USB_VCOM_INSTANCE_LOGGING]) {
        Serial_DataRecv(vcomInstance->currRecvBuf, vcomInstance->recvSize);
    }
    #endif
    /*
     * If user send something and waited until completion in callback (== StateReady),
     * we are already ready and don't have to schedule Recv() here
     * If user is sending something (but it didn't complete yet == StateSending)
     * we will schedule Recv() when command is send, so don't schedule Recv() here
     */
    if (kUSB_VcomStateReady != CtrlCtxt->state && kUSB_VcomStateSending != CtrlCtxt->state) {
        /* schedule buffer for next receive event */
        usb_status_t error = USB_DeviceCdcAcmRecv(vcomInstance->cdcAcmHandle,
                                                  vcomInstance->bulkOutEndpoint,
                                                  vcomInstance->currRecvBuf,
                                                  vcomInstance->bulkOutEndpointMaxPacketSize);

        if (kStatus_USB_Success != error) {
            PERR("Error in CdcAcmRecv: %d", error);
        } else {
            CtrlCtxt->state = kUSB_VcomStateReady;
        }
    }
}

static void HostPresenceHandler(void *cookie)
{
    vcom_ctrl_ctxt_t *CtrlCtxt = (vcom_ctrl_ctxt_t *)cookie;
    assert(CtrlCtxt);
    if (CtrlCtxt->hostCb == NULL) {
        return;
    }

    volatile usb_cdc_vcom_struct_t *vcomInstance = CtrlCtxt->vcomInstance;
    assert(vcomInstance);
    if (vcomInstance == NULL) {
        return;
    }

    bool present = vcomInstance->usbCdcAcmInfo->dtePresent;
    CtrlCtxt->hostCb(present);
}

static usb_status_t HandleRecvIsr(vcom_ctrl_ctxt_t *vcomCtrlCtxt, uint32_t dataRecvCount)
{
    if (USB_UNINITIALIZED_VAL_32 == dataRecvCount) {
        return kStatus_USB_TransferCancel;
    }

    volatile usb_cdc_vcom_struct_t *vcomInstance = vcomCtrlCtxt->vcomInstance;
    vcomInstance->recvSize = dataRecvCount;

    #if SERIAL_PORT_TYPE_USBCDC
    /* Logging VCOM */
    if (&gs_VcomCtrlCtxt[USB_VCOM_INSTANCE_LOGGING] == vcomCtrlCtxt) {
        DataRecvHandler(&gs_VcomCtrlCtxt[USB_VCOM_INSTANCE_LOGGING]);
        return kStatus_USB_Success;
    }
    #endif

    /*
     * Other VCOM
     * new data received, pass it up to the higher layers
     */
    if (kERROR_Ok != FRAMEWORK_ScheduleDpc(&vcomCtrlCtxt->dataRecvSrp)) {
        PERR("Could not schedule vcom receive srp");
        return kStatus_USB_Error;
    }

    return kStatus_USB_Success;
}

static usb_status_t HandleSendIsr(vcom_ctrl_ctxt_t *vcomCtrlCtxt, uint32_t dataSendCount)
{
    volatile usb_cdc_vcom_struct_t *vcomInstance = vcomCtrlCtxt->vcomInstance;

    vcomCtrlCtxt->dataIndex += dataSendCount;
    vcomCtrlCtxt->toWrite -= dataSendCount;

    #if SERIAL_PORT_TYPE_USBCDC
    /* logging VCOM */
    if (&gs_VcomCtrlCtxt[USB_VCOM_INSTANCE_LOGGING] == vcomCtrlCtxt) {
        /*
         * Only update toWrite, NO dpc,
         * Call Cmpl Handler in ISR, rest will be handled in USB_VcomWriteBlocking
         */
        SendCmplHandler((void *)vcomCtrlCtxt);
        return kStatus_USB_Success;
    }
    #endif /* SERIAL_PORT_TYPE_USBCDC */

    if (0 == vcomCtrlCtxt->toWrite &&
        (dataSendCount != 0 && 0 == dataSendCount % vcomInstance->bulkInEndpointMaxPacketSize)) {
        /* If the last packet is the size of endpoint, then send also zero-ended packet,
        ** meaning that we want to inform the host that we do not have any additional
        ** data, so it can flush the output.
        */
        return USB_DeviceCdcAcmSend(vcomInstance->cdcAcmHandle, vcomInstance->bulkInEndpoint, NULL, 0);
    }

    /*
     * Other VCOM,
     * Still stuff to do, output will be flushed with next send
     */
    if (kERROR_Ok != FRAMEWORK_ScheduleDpc(&vcomCtrlCtxt->sendCmplSrp)) {
        PERR("Could not schedule vcom receive srp");
        return kStatus_USB_Error;
    }

    return kStatus_USB_Success;
}

/* DTE (Data Terminal Equipment) is the VCOM host */
static usb_status_t HandleDteChangeIsr(vcom_ctrl_ctxt_t *vcomCtrlCtxt)
{
    /* schedule a DPC to call the 'hostCb' */
    if (vcomCtrlCtxt->hostCb != NULL) {
        if (kERROR_Ok != FRAMEWORK_ScheduleDpc(&vcomCtrlCtxt->hostPresenceSrp)) {
            PERR("Could not schedule vcom host presence changed srp");
            return kStatus_USB_Error;
        }
    }
    return kStatus_USB_Success;
}

/*!
 * @brief CDC class specific callback function.
 *
 * This function handles the CDC class specific requests.
 *
 * @param handle          The CDC ACM class handle.
 * @param event           The CDC ACM class event type.
 * @param param           The parameter of the class specific request.
 *
 * @return A USB error code or kStatus_USB_Success.
 */
usb_status_t USB_DeviceCdcVcomCallback(class_handle_t handle, uint32_t event, void *param)
{
    uint32_t len;
    uint8_t *uartBitmap;
    usb_cdc_acm_info_t *acmInfo;
    usb_device_cdc_acm_request_param_struct_t *acmReqParam;
    usb_device_endpoint_callback_message_struct_t *epCbParam;
    volatile usb_cdc_vcom_struct_t *vcomInstance = NULL;
    vcom_ctrl_ctxt_t *vcomCtrlCtxt = NULL;
    usb_status_t error = kStatus_USB_Error;
    uint8_t i;
    acmReqParam = (usb_device_cdc_acm_request_param_struct_t *)param;
    epCbParam = (usb_device_endpoint_callback_message_struct_t *)param;

    for (i = 0; i < USB_DEVICE_CONFIG_CDC_ACM; i++) {
        if (handle == g_deviceComposite->cdcVcom[i].cdcAcmHandle) {
            break;
        }
    }
    if (i >= USB_DEVICE_CONFIG_CDC_ACM) {
        return error;
    }

    vcomInstance = &g_deviceComposite->cdcVcom[i];
    vcomCtrlCtxt = &gs_VcomCtrlCtxt[i];
    acmInfo = vcomInstance->usbCdcAcmInfo;
    switch (event) {
        case kUSB_DeviceCdcEventSendResponse:
        {
            if ((epCbParam->length != 0) && (!(epCbParam->length % vcomInstance->bulkInEndpointMaxPacketSize))) {
                error = HandleSendIsr(vcomCtrlCtxt, epCbParam->length);
            } else if ((1 == vcomInstance->attach) && (1 == vcomInstance->startTransactions)) {
                if ((epCbParam->buffer != NULL) || ((epCbParam->buffer == NULL) && (epCbParam->length == 0))) {
                    error = HandleSendIsr(vcomCtrlCtxt, epCbParam->length);
                }
            } else {
                /* TODO: How did I get here? */
                assert(false);
            }
        }
        break;
        case kUSB_DeviceCdcEventRecvResponse:
        {
            /* PDBG("CDC event receive response"); */
            if ((1 == vcomInstance->attach) && (1 == vcomInstance->startTransactions)) {
                error = HandleRecvIsr(vcomCtrlCtxt, epCbParam->length);
            }
        }
        break;
        case kUSB_DeviceCdcEventSerialStateNotif:
            ((usb_device_cdc_acm_struct_t *)handle)->hasSentState = 0;
            error = kStatus_USB_Success;
            break;
        case kUSB_DeviceCdcEventSendEncapsulatedCommand:
            break;
        case kUSB_DeviceCdcEventGetEncapsulatedResponse:
            break;
        case kUSB_DeviceCdcEventSetCommFeature:
            if (USB_DEVICE_CDC_FEATURE_ABSTRACT_STATE == acmReqParam->setupValue) {
                if (1 == acmReqParam->isSetup) {
                    *(acmReqParam->buffer) = vcomInstance->abstractState;
                } else {
                    *(acmReqParam->length) = 0;
                }
            } else if (USB_DEVICE_CDC_FEATURE_COUNTRY_SETTING == acmReqParam->setupValue) {
                if (1 == acmReqParam->isSetup) {
                    *(acmReqParam->buffer) = vcomInstance->countryCode;
                } else {
                    *(acmReqParam->length) = 0;
                }
            } else {
            }
            error = kStatus_USB_Success;
            break;
        case kUSB_DeviceCdcEventGetCommFeature:
            if (USB_DEVICE_CDC_FEATURE_ABSTRACT_STATE == acmReqParam->setupValue) {
                *(acmReqParam->buffer) = vcomInstance->abstractState;
                *(acmReqParam->length) = COMM_FEATURE_DATA_SIZE;
            } else if (USB_DEVICE_CDC_FEATURE_COUNTRY_SETTING == acmReqParam->setupValue) {
                *(acmReqParam->buffer) = vcomInstance->countryCode;
                *(acmReqParam->length) = COMM_FEATURE_DATA_SIZE;
            } else {
            }
            error = kStatus_USB_Success;
            break;
        case kUSB_DeviceCdcEventClearCommFeature:
            break;
        case kUSB_DeviceCdcEventGetLineCoding:
            *(acmReqParam->buffer) = vcomInstance->lineCoding;
            *(acmReqParam->length) = LINE_CODING_SIZE;
            error = kStatus_USB_Success;
            break;
        case kUSB_DeviceCdcEventSetLineCoding:
        {
            if (1 == acmReqParam->isSetup) {
                *(acmReqParam->buffer) = vcomInstance->lineCoding;
            } else {
                *(acmReqParam->length) = 0;
            }
        }
            error = kStatus_USB_Success;
            break;
        case kUSB_DeviceCdcEventSetControlLineState:
        {
            vcomInstance->usbCdcAcmInfo->dteStatus = acmReqParam->setupValue;
            /* activate/deactivate Tx carrier */
            if (acmInfo->dteStatus & USB_DEVICE_CDC_CONTROL_SIG_BITMAP_CARRIER_ACTIVATION) {
                acmInfo->uartState |= USB_DEVICE_CDC_UART_STATE_TX_CARRIER;
            } else {
                acmInfo->uartState &= (uint16_t) ~USB_DEVICE_CDC_UART_STATE_TX_CARRIER;
            }

            /* activate carrier and DTE */
            if (acmInfo->dteStatus & USB_DEVICE_CDC_CONTROL_SIG_BITMAP_DTE_PRESENCE) {
                acmInfo->uartState |= USB_DEVICE_CDC_UART_STATE_RX_CARRIER;
            } else {
                acmInfo->uartState &= (uint16_t) ~USB_DEVICE_CDC_UART_STATE_RX_CARRIER;
            }

            /* Indicates to DCE if DTE is present or not */
            bool curDtePresent = (acmInfo->dteStatus & USB_DEVICE_CDC_CONTROL_SIG_BITMAP_DTE_PRESENCE) ? true : false;
            if (acmInfo->dtePresent != curDtePresent) {  /* DTE presence changed */
                acmInfo->dtePresent = curDtePresent;
                error = HandleDteChangeIsr(vcomCtrlCtxt);
            }

            /* Initialize the serial state buffer */
            acmInfo->serialStateBuf[0] = NOTIF_REQUEST_TYPE;                /* bmRequestType */
            acmInfo->serialStateBuf[1] = USB_DEVICE_CDC_NOTIF_SERIAL_STATE; /* bNotification */
            acmInfo->serialStateBuf[2] = 0x00;                              /* wValue */
            acmInfo->serialStateBuf[3] = 0x00;
            acmInfo->serialStateBuf[4] = 0x00; /* wIndex */
            acmInfo->serialStateBuf[5] = 0x00;
            acmInfo->serialStateBuf[6] = UART_BITMAP_SIZE; /* wLength */
            acmInfo->serialStateBuf[7] = 0x00;
            /* Notifiy to host the line state */
            acmInfo->serialStateBuf[4] = acmReqParam->interfaceIndex;
            /* Lower byte of UART BITMAP */
            uartBitmap = (uint8_t *)&acmInfo->serialStateBuf[NOTIF_PACKET_SIZE + UART_BITMAP_SIZE - 2];
            uartBitmap[0] = acmInfo->uartState & 0xFFu;
            uartBitmap[1] = (acmInfo->uartState >> 8) & 0xFFu;
            len = (uint32_t)(NOTIF_PACKET_SIZE + UART_BITMAP_SIZE);
            if (0 == ((usb_device_cdc_acm_struct_t *)handle)->hasSentState) {
//                error = USB_DeviceCdcAcmSend(handle, vcomInstance->interruptEndpoint, acmInfo->serialStateBuf, len);
//                if (kStatus_USB_Success != error) {
//                    usb_echo("kUSB_DeviceCdcEventSetControlLineState error!");
//                }
                ((usb_device_cdc_acm_struct_t *)handle)->hasSentState = 1;
            }

            /* Update status */
            if (acmInfo->dteStatus & USB_DEVICE_CDC_CONTROL_SIG_BITMAP_CARRIER_ACTIVATION) {
                /*  To do: CARRIER_ACTIVATED */
            } else {
                /* To do: CARRIER_DEACTIVATED */
            }
            if (acmInfo->dteStatus & USB_DEVICE_CDC_CONTROL_SIG_BITMAP_DTE_PRESENCE) {
                /* DTE_ACTIVATED */
                if (1 == vcomInstance->attach) {
                    vcomInstance->startTransactions = 1;
                }
            } else {
                /* DTE_DEACTIVATED */
                if (1 == vcomInstance->attach) {
                    vcomInstance->startTransactions = 0;
                }
            }
        }
        break;
        case kUSB_DeviceCdcEventSendBreak:
            break;
        default:
            break;
    }

    return error;
}

/*!
 * @brief Virtual COM device set configuration function.
 *
 * This function sets configuration for CDC class.
 *
 * @param handle The CDC ACM class handle.
 * @param configure The CDC ACM class configure index.
 *
 * @return A USB error code or kStatus_USB_Success.
 */
usb_status_t USB_DeviceCdcVcomSetConfigure(class_handle_t handle, uint8_t configure)
{
    /* TODO: Use handle to differentiate between different VCOMs */
    (void)handle;

    if (USB_COMPOSITE_CONFIGURE_INDEX == configure) {
    #if USB_HCI_VCOM_ENABLE
        /*endpoint information for cdc 1*/
        g_deviceComposite->cdcVcom[USB_VCOM_INSTANCE_HCI].attach = 1;

        g_deviceComposite->cdcVcom[USB_VCOM_INSTANCE_HCI].interruptEndpoint = USB_CDC_VCOM_CIC_INTERRUPT_IN_ENDPOINT;
        g_deviceComposite->cdcVcom[USB_VCOM_INSTANCE_HCI].interruptEndpointMaxPacketSize = g_cdcVcomCicEndpoints[0].maxPacketSize;

        g_deviceComposite->cdcVcom[USB_VCOM_INSTANCE_HCI].bulkInEndpoint = USB_CDC_VCOM_DIC_BULK_IN_ENDPOINT;
        g_deviceComposite->cdcVcom[USB_VCOM_INSTANCE_HCI].bulkInEndpointMaxPacketSize = g_cdcVcomDicEndpoints[0].maxPacketSize;

        g_deviceComposite->cdcVcom[USB_VCOM_INSTANCE_HCI].bulkOutEndpoint = USB_CDC_VCOM_DIC_BULK_OUT_ENDPOINT;
        g_deviceComposite->cdcVcom[USB_VCOM_INSTANCE_HCI].bulkOutEndpointMaxPacketSize = g_cdcVcomDicEndpoints[1].maxPacketSize;

        /* Schedule buffer for receive */
        if (kStatus_USB_Success != USB_DeviceCdcAcmRecv(g_deviceComposite->cdcVcom[USB_VCOM_INSTANCE_HCI].cdcAcmHandle,
                                                        g_deviceComposite->cdcVcom[USB_VCOM_INSTANCE_HCI].bulkOutEndpoint,
                                                        s_currRecvBuf[USB_VCOM_INSTANCE_HCI],
                                                        g_deviceComposite->cdcVcom[USB_VCOM_INSTANCE_HCI].bulkOutEndpointMaxPacketSize)) {
            PERR("Failed to schedule receive when configuring VCOM USB_VCOM_INSTANCE_HCI");
            return kStatus_USB_Error;
        }

        /* Ready to receive/send data */
        gs_VcomCtrlCtxt[USB_VCOM_INSTANCE_HCI].state = kUSB_VcomStateReady;
    #endif

    #if SERIAL_PORT_TYPE_USBCDC
        /*endpoint information for cdc 2*/
        g_deviceComposite->cdcVcom[USB_VCOM_INSTANCE_LOGGING].attach = 1;

        g_deviceComposite->cdcVcom[USB_VCOM_INSTANCE_LOGGING].interruptEndpoint = USB_CDC_VCOM_CIC_INTERRUPT_IN_ENDPOINT_2;
        g_deviceComposite->cdcVcom[USB_VCOM_INSTANCE_LOGGING].interruptEndpointMaxPacketSize = g_cdcVcomCicEndpoints_2[0].maxPacketSize;

        g_deviceComposite->cdcVcom[USB_VCOM_INSTANCE_LOGGING].bulkInEndpoint = USB_CDC_VCOM_DIC_BULK_IN_ENDPOINT_2;
        g_deviceComposite->cdcVcom[USB_VCOM_INSTANCE_LOGGING].bulkInEndpointMaxPacketSize = g_cdcVcomDicEndpoints_2[0].maxPacketSize;

        g_deviceComposite->cdcVcom[USB_VCOM_INSTANCE_LOGGING].bulkOutEndpoint = USB_CDC_VCOM_DIC_BULK_OUT_ENDPOINT_2;
        g_deviceComposite->cdcVcom[USB_VCOM_INSTANCE_LOGGING].bulkOutEndpointMaxPacketSize = g_cdcVcomDicEndpoints_2[1].maxPacketSize;

        /*
         * NOTE:
         * Don't schedule buffer for receive,
         * We only schedule buffer for receive when user explicitly requests it
         * with USB_VcomReadBlocking
         */

        /* Ready to receive/send data */
        gs_VcomCtrlCtxt[USB_VCOM_INSTANCE_LOGGING].state = kUSB_VcomStateReady;
    #endif
    }
    return kStatus_USB_Success;
}

/*!
 * @brief Virtual COM device initialization function.
 *
 * This function initializes the device with the composite device class information.
 *
 * @param deviceComposite The pointer to the composite device structure.
 * @param rcvdCb pointer to a callback function for when data is received over the vcom.
 * @param hostCb pointer to a callback function for when the presence of the vcom host changed.
 *
 * @return A USB error code or kStatus_USB_Success.
 */
usb_status_t USB_DeviceCdcVcomInit(usb_device_composite_struct_t *deviceComposite, vcom_data_rcvd_cb rcvdCb, vcom_host_presence_cb hostCb)
{
    g_deviceComposite = deviceComposite;

    UsbVcom_Init();

    for (uint8_t i = 0; i < USB_DEVICE_CONFIG_CDC_ACM; i++) {
        g_deviceComposite->cdcVcom[i].lineCoding = (uint8_t *)&s_lineCoding[i];
        g_deviceComposite->cdcVcom[i].abstractState = (uint8_t *)&s_abstractState[i];
        g_deviceComposite->cdcVcom[i].countryCode = (uint8_t *)&s_countryCode[i];
        g_deviceComposite->cdcVcom[i].usbCdcAcmInfo = &s_usbCdcAcmInfo[i];
        g_deviceComposite->cdcVcom[i].usbCdcAcmInfo->dtePresent = false;
        g_deviceComposite->cdcVcom[i].currRecvBuf = (uint8_t *)&s_currRecvBuf[i][0];
        g_deviceComposite->cdcVcom[i].deviceHandle = g_deviceComposite->deviceHandle;

        /* Our config */
        gs_VcomCtrlCtxt[i].vcomInstance = &g_deviceComposite->cdcVcom[i];
        gs_VcomCtrlCtxt[i].state = kUSB_VcomStateIdle;
        gs_VcomCtrlCtxt[i].rcvdCb = rcvdCb;
        gs_VcomCtrlCtxt[i].hostCb = hostCb;

        gs_VcomCtrlCtxt[i].dataRecvSrp = (framework_dpc_srp_t) {
            .cb = DataRecvHandler,
            .ctxt = (void *)&gs_VcomCtrlCtxt[i],
        };

        gs_VcomCtrlCtxt[i].sendCmplSrp = (framework_dpc_srp_t) {
            .cb = SendCmplHandler,
            .ctxt = (void *)&gs_VcomCtrlCtxt[i],
        };

        gs_VcomCtrlCtxt[i].hostPresenceSrp = (framework_dpc_srp_t) {
            .cb = HostPresenceHandler,
            .ctxt = (void *)&gs_VcomCtrlCtxt[i],
        };
    }

    return kStatus_USB_Success;
}

void UsbVcom_Reset(void)
{
    for (uint8_t i = 0; i < USB_DEVICE_CONFIG_CDC_ACM; i++) {
        g_deviceComposite->cdcVcom[i].recvSize = 0;
        g_deviceComposite->cdcVcom[i].attach = 0;
    }
}

#endif

