/* ---------------------------------------------------------------------------- */
/* Copyright 2018, 2024 NXP.                                                    */
/* SPDX-License-Identifier: BSD-3-Clause                                        */
/*                                                                              */
/* This software is owned or controlled by NXP and may only be used strictly    */
/* in accordance with the applicable license terms. By expressly accepting such */
/* terms or by downloading, installing, activating and/or otherwise using the   */
/* software, you are agreeing that you have read, and that you agree to comply  */
/* with and are bound by, such license terms. If you do not agree to be bound   */
/* by the applicable license terms, then you may not retain, install, activate  */
/* or otherwise use the software.                                               */
/* ---------------------------------------------------------------------------- */

/* ---------------------------------------------------------------------------- */
/* Include files                                                                */
/* ---------------------------------------------------------------------------- */
#include <stdint.h>
#include <stdbool.h>

#include "audio_ringbuffer.h"
#include "critical.h"

#include "fsl_i2s_dma.h"                                         

#include "audio_mixer.h"
#include "fsl_debug_console.h"
#include "fsl_adapter_audio.h"

/* need to include "usb.h, usb_device.h, usb_device_class.h" before include "usb_device_descriptor.h" */
#include "usb.h"
#include "usb_device.h"
#include "usb_device_class.h"
#include "usb_device_descriptor.h"

#include "audio.h"
#include "audio_latency_debug.h"



/* ---------------------------------------------------------------------------- */
/* Defines                                                                      */
/* ---------------------------------------------------------------------------- */
/** Audio buffer size */
#define AUDIO_INPUT_RING_BUFFER_SIZE     MIXER_INPUT_RING_BUFFER_SIZE


/** Audio in buffer half marker */
#define AUDIO_IN_RING_BUFFER_HALF_FILL   (AUDIO_INPUT_RING_BUFFER_SIZE / 2)


/** Current audio Service status */
static uint32_t s_audioService_RxStatus = 0;

/** Set a status flag of the audio Service */
#define AUDIO_SERVICE_SET_STATUS(x)  (s_audioService_RxStatus |= ((x)))

/** Clears a status flag of the audio Service */
#define AUDIO_SERVICE_CLEAR_STATUS(x)  (s_audioService_RxStatus &= ~((x)))

/** Check the current status of the audio Service */
#define AUDIO_SERVICE_CHECK_STATUS(x)  (s_audioService_RxStatus & ((x)))

/* I2S_TX_WORD_LENGTH_BYTES = 2 by default */
#define RX_DMA_TRANSFER_SIZE  (HS_ISO_IN_ENDP_PACKET_SIZE / AUDIO_IN_FORMAT_SIZE * I2S_TX_WORD_LENGTH_BYTES)

/* ---------------------------------------------------------------------------- */
/* Local variables                                                              */
/* ---------------------------------------------------------------------------- */
/** ring buffer required to allocate the input samples of the application. */
phRingBuffer_t s_audioService_RingBufferIn;

/** defined buffer to store the audio input samples using the dma. */
uint8_t __attribute__((__aligned__(4))) s_audioService_BufferIn[AUDIO_INPUT_RING_BUFFER_SIZE];

/** Required to control the buffer input receptions */
static uint32_t s_audioService_RxFillThreshold = 0;

/* ---------------------------------------------------------------------------- */
/* Global variables                                                              */
/* ---------------------------------------------------------------------------- */
uint8_t g_RxDmaStartFlag = 0;

extern hal_audio_transfer_t s_RxTransfer;
extern uint8_t g_I2sTxStarted;
extern HAL_AUDIO_HANDLE_DEFINE(audioRxHandle);

/* ---------------------------------------------------------------------------- */
/* Local functions                                                           */
/* ---------------------------------------------------------------------------- */
void audio_StopRx(void);

/* ---------------------------------------------------------------------------- */
/* Global functions                                                           */
/* ---------------------------------------------------------------------------- */
void RxCallback(hal_audio_handle_t handle, hal_audio_status_t completionStatus, void *callbackParam);

/**
 * @brief Call back function of DMA Rx
 *
 * @param handle pointer to DMA handle
 * @param userData pointer to Callback userdata
 * @return void.
 */
void RxCallback(hal_audio_handle_t handle, hal_audio_status_t completionStatus, void *callbackParam)
{
    AUDIO_LatencyDebug_ReceivingAudioFromI2s(true);

    /** update the input ring buffer address */
    critical_Enter(); /* Ring buffer is shared resource between DMA and USB. */
  
    s_audioService_RingBufferIn.pWritePointer += RX_DMA_TRANSFER_SIZE;
    if( s_audioService_RingBufferIn.pWritePointer > (volatile uint8_t *)s_audioService_RingBufferIn.EndAddress)
    {
        s_audioService_RingBufferIn.pWritePointer = (volatile uint8_t *)s_audioService_RingBufferIn.StartAddress;
    }

    s_RxTransfer.data     = (uint8_t *)s_audioService_RingBufferIn.pWritePointer;
    s_RxTransfer.dataSize = RX_DMA_TRANSFER_SIZE;

    HAL_AudioTransferReceiveNonBlocking((hal_audio_handle_t)&audioRxHandle[0], &s_RxTransfer);

    critical_Exit();
    /** trigger proper events */
    AUDIO_SERVICE_SET_STATUS(AUDIO_SERVICE_RX_COMPLETE_MASK);

    AUDIO_LatencyDebug_ReceivingAudioFromI2s(false);  
}

/* ---------------------------------------------------------------------------- */
/* Local functions-RX                                                           */
/* ---------------------------------------------------------------------------- */
static void audio_RxSpaceWarningCb(uint32_t Filling, void *pCtxt)
{
    (void)Filling;    /* make release build happy */
    (void)pCtxt;      /* make release build happy */

    if (AUDIO_SERVICE_CHECK_STATUS(AUDIO_SERVICE_RX_STARTED_MASK)) {
        /* Restart the audio */
        audio_StopRx();
        /*
         * Don't call audio_StartRx().
         * The audio will be automatically started when the
         * ring buffer filling threshold is reached.
         */
        PRINTF("Restart RX Audio\r\n");
    }
    PRINTF("RxSpaceWarning, F=%d\r\n", Filling);
}


/**
 * Initializes and starts the I2S-Rx as part of audio-SAI
 */
void audio_InitRx(void)
{
}

void audio_StartRx(void)
{
    g_RxDmaStartFlag = 1;
    /* Prepare two RX DMA transfer */
    s_RxTransfer.data     = (uint8_t *)s_audioService_RingBufferIn.pWritePointer;
    s_RxTransfer.dataSize = RX_DMA_TRANSFER_SIZE;
    /* Enable RX and RX DMA request */
    HAL_AudioTransferReceiveNonBlocking((hal_audio_handle_t)&audioRxHandle[0], &s_RxTransfer);

    s_audioService_RingBufferIn.pWritePointer += RX_DMA_TRANSFER_SIZE;
    s_RxTransfer.data     = (uint8_t *)s_audioService_RingBufferIn.pWritePointer;
    s_RxTransfer.dataSize = RX_DMA_TRANSFER_SIZE;
    HAL_AudioTransferReceiveNonBlocking((hal_audio_handle_t)&audioRxHandle[0], &s_RxTransfer);
    
    if(g_I2sTxStarted != 1)
    {
         /* enable BCLK and WS for microphone channel */
         I2S_Enable(I2S7);
    }
}

/* ---------------------------------------------------------------------------- */
/* Public Functions-RX                                                          */
/* ---------------------------------------------------------------------------- */

void audio_ReadBuffer(uint8_t *pOutBuffer, uint16_t DataSize)
{
    s_audioService_RxFillThreshold += RX_DMA_TRANSFER_SIZE;

    critical_Enter(); /* Ring buffer is shared resource between DMA and USB. */
    /*--Read several bytes to 's_wavBuff' from the ring buffer 's_audioService_RingBufferIn'*/
    audio_ringbuffer_Read(&s_audioService_RingBufferIn, (uint8_t *)pOutBuffer, (DataSize * 2));
    
    critical_Exit();

    /** Check if Rx has not already started, Cannot proceed if RX has started*/
    if (!AUDIO_SERVICE_CHECK_STATUS(AUDIO_SERVICE_RX_STARTED_MASK)) {
        AUDIO_SERVICE_SET_STATUS(AUDIO_SERVICE_RX_COMPLETE_MASK);
        /** check if the I2S Rx is enabled, if not, that means the ring buffer  needs to be filled up to 50% */
        if (s_audioService_RxFillThreshold >= AUDIO_IN_RING_BUFFER_HALF_FILL) {  
            audio_StartRx();
              
            /** set the internal flags */
            AUDIO_SERVICE_SET_STATUS(AUDIO_SERVICE_RX_STARTED_MASK);
            AUDIO_SERVICE_CLEAR_STATUS(AUDIO_SERVICE_RX_COMPLETE_MASK);
        }
    }
}

void audio_StopRx(void)  
{
    /** resets the audio buffer address to start*/
    critical_Enter(); /* Ring buffer is shared resource between DMA and USB. */
    audio_ringbuffer_Reset(&s_audioService_RingBufferIn);
    critical_Exit();
    s_audioService_RxFillThreshold = 0;
    g_RxDmaStartFlag = 0;

    /* Aborts DMA transfer */
    HAL_AudioTransferAbortReceive((hal_audio_handle_t)audioRxHandle);

    /* Resets the AudioService status */
    AUDIO_SERVICE_CLEAR_STATUS(AUDIO_SERVICE_RX_STARTED_MASK);
    AUDIO_SERVICE_SET_STATUS(AUDIO_SERVICE_RX_COMPLETE_MASK);
}

uint32_t audio_RxAvailableSpace(void)
{
    uint32_t Filling;

    critical_Enter(); /* Ring buffer is shared resource between DMA and USB. */
    Filling = audio_ringbuffer_GetFilling(&s_audioService_RingBufferIn);
    critical_Exit();

    /* return the rx available space */
    return Filling;
}

uint32_t audio_GetRxBufferSize(void)
{
    /** returns the current buffer size */
    return((uint32_t)s_audioService_RingBufferIn.BufferSize);
}


void audio_RingbufferRxInit(void)
{
    /** Initialize the ring buffer for rx */
    critical_Enter(); /* Ring buffer is shared resource between DMA and USB. */
    audio_ringbuffer_Init(&s_audioService_RingBufferIn, (uint8_t *)&s_audioService_BufferIn[0], sizeof(s_audioService_BufferIn));
    critical_Exit();

    audio_ringbuffer_ConfigureFillingWarning(
        &s_audioService_RingBufferIn,
        RX_DMA_TRANSFER_SIZE,
        audio_RxSpaceWarningCb,
        NULL);
}
/** @} */
