/*
 * Copyright (c) 2013 - 2015, Freescale Semiconductor, Inc.
 * Copyright 2016-2017, 2025 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <stdio.h>
#include "rpmsg_lite.h"
#include "rpmsg_queue.h"
#include "rpmsg_ns.h"
#include "fsl_device_registers.h"
#include "fsl_debug_console.h"
#include "pin_mux.h"
#include "clock_config.h"
#include "board.h"
#include "dsp_support.h"

#include "FreeRTOS.h"
#include "task.h"
#include "shared_memory_def.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define BOOT_HIFI1 (1)
#define CPU0_TASK_STACK_SIZE (256U)
#define HIFI1_TASK_STACK_SIZE (256U)
#define CPU1_NS_ANNOUNCE_STRING "SenseDomain - CPU1"

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
static void cpu0_task(void *param);
static void hifi1_task(void *param);
/*******************************************************************************
 * Code
 ******************************************************************************/

/*!
 * @brief Main function
 */
int main(void)
{
    /* Init board hardware. */
    BOARD_InitPins();
    BOARD_BootClockRUN();
    BOARD_InitDebugConsole();
    CLOCK_EnableClock(kCLOCK_SenseAccessRamArbiter0);
    CLOCK_EnableClock(kCLOCK_Gpio8);
    RESET_ClearPeripheralReset(kGPIO8_RST_SHIFT_RSTn);

    /* Enable GPIO access from DSP side */
    GPIO_EnablePinControlNonSecure(BOARD_LED_RED_GPIO, 1 << BOARD_LED_RED_GPIO_PIN);
    GPIO_EnablePinControlNonPrivilege(BOARD_LED_RED_GPIO, 1 << BOARD_LED_RED_GPIO_PIN);

    PRINTF("[CPU1] boot up - RPMSG \r\n");

#if BOOT_HIFI1
    if (xTaskCreate(hifi1_task, "hifi1 task", HIFI1_TASK_STACK_SIZE, NULL, 3, NULL) !=
        pdPASS)
    {
        PRINTF("\r\nFailed to create HiFi1 task\r\n");
        while (1);
    }
#endif

    if (xTaskCreate(cpu0_task, "cpu0 task", CPU0_TASK_STACK_SIZE, NULL, 4, NULL) !=
        pdPASS)
    {
        PRINTF("\r\nFailed to create CPU0 task\r\n");
        while (1);
    }
	
    vTaskStartScheduler();
}

static int32_t cpu1_ept_read_cb(void *payload, uint32_t payload_len, uint32_t src, void *priv)
{
    PRINTF("[CPU1]  RECV %s\r\n", payload);
    return 0;
}

static void app_nameservice_isr_cb(uint32_t new_ept, const char *new_ept_name, uint32_t flags, void *user_data)
{
    uint32_t *data = (uint32_t *)user_data;
    if(data) {
        *data = new_ept;
        PRINTF("Remote Name, %d - %s\r\n", new_ept, new_ept_name);
    }
}

static void cpu0_task(void *param)
{
    volatile uint32_t cpu0_addr = 0;
    struct rpmsg_lite_instance * p_cpu1_cpu0_rpmsg;
    struct rpmsg_lite_endpoint *p_cpu1_cpu0_ep = NULL;
    rpmsg_queue_handle cpu1_cpu0_queue;
    rpmsg_ns_handle cpu0_ns_handle;
    cpu0_cpu1_parcel_t cpu0_cpu1_parcel;
    cpu1_cpu0_parcel_t cpu1_cpu0_parcel;
    int32_t status;
    uint32_t response_counter = 0;

    p_cpu1_cpu0_rpmsg = rpmsg_lite_remote_init((void *)CPU0_CPU1_RPMSG_LITE_BASE,
		                           RL_PLATFORM_IMXRT700_M33_0_M33_1_LINK_ID, RL_NO_FLAGS);
    cpu1_cpu0_queue = rpmsg_queue_create(p_cpu1_cpu0_rpmsg);

    p_cpu1_cpu0_ep = rpmsg_lite_create_ept(p_cpu1_cpu0_rpmsg, CPU1_EPT_ADDR, rpmsg_queue_rx_cb, cpu1_cpu0_queue);
    cpu0_ns_handle = rpmsg_ns_bind(p_cpu1_cpu0_rpmsg, app_nameservice_isr_cb, ((void *)0));
    (void)cpu0_ns_handle;
    rpmsg_lite_wait_for_link_up(p_cpu1_cpu0_rpmsg, RL_BLOCK);
    PRINTF("[CPU1] Link up!\r\n");
	
    SDK_DelayAtLeastUs(1000, SystemCoreClock);
    rpmsg_ns_announce(p_cpu1_cpu0_rpmsg, p_cpu1_cpu0_ep, CPU1_NS_ANNOUNCE_STRING, (uint32_t)RL_NS_CREATE);
    PRINTF("[CPU1] Name service announce sent.\r\n");

    while (1)
    {
        status = rpmsg_queue_recv(p_cpu1_cpu0_rpmsg, cpu1_cpu0_queue, NULL, (char *)&cpu0_cpu1_parcel, 
                                  sizeof(cpu0_cpu1_parcel), NULL, RL_BLOCK);
     	if (status != RL_SUCCESS)
     	{
            PRINTF("Failed to get item from RPMsg queue.\r\n");
     	} else {
            PRINTF("[CPU1]  RECV %s\r\n", (char*)cpu0_cpu1_parcel.data);
        }
		
        snprintf((char *)cpu1_cpu0_parcel.data, 16, "C1->C0:%d", response_counter++);
        
        rpmsg_lite_send(p_cpu1_cpu0_rpmsg, p_cpu1_cpu0_ep, CPU0_EPT_ADDR, 
			            (char *)cpu1_cpu0_parcel.data, sizeof(cpu1_cpu0_parcel), RL_BLOCK);
    }
}

#if BOOT_HIFI1
static void hifi1_task(void *param)
{
    volatile uint32_t hifi1_addr = 0;
    struct rpmsg_lite_instance * p_cpu1_hifi1_rpmsg;
    struct rpmsg_lite_endpoint *p_cpu1_hifi1_ep = NULL;
    rpmsg_ns_handle hifi1_ns_handle;
    cpu0_hifi4_parcel_t cpu1_hifi1_parcel;
    volatile uint32_t message_counter = 0;

    BOARD_DSP_Init();

    p_cpu1_hifi1_rpmsg = rpmsg_lite_master_init((void *)CPU1_HIFI1_RPMSG_LITE_BASE, 
		                        CPU1_HIFI1_SH_MEM_TOTAL_SIZE, RL_PLATFORM_IMXRT700_M33_1_HIFI1_LINK_ID, RL_NO_FLAGS);
    
    p_cpu1_hifi1_ep = rpmsg_lite_create_ept(p_cpu1_hifi1_rpmsg, CPU1_EPT_ADDR, cpu1_ept_read_cb, NULL);

    hifi1_ns_handle = rpmsg_ns_bind(p_cpu1_hifi1_rpmsg, app_nameservice_isr_cb, (void *)&hifi1_addr);
    (void)hifi1_ns_handle;;

    while(hifi1_addr != HIFI1_EPT_ADDR)
    {
        vTaskDelay(5);
    }

    while(1)
    {
        snprintf((char *)cpu1_hifi1_parcel.data, 16, "C1->H1:%d", message_counter++);
        rpmsg_lite_send(p_cpu1_hifi1_rpmsg, p_cpu1_hifi1_ep, HIFI1_EPT_ADDR,
			           (char *)&cpu1_hifi1_parcel, sizeof(cpu1_hifi1_parcel), RL_BLOCK);
        vTaskDelay(1000);
    }
}
#endif