/*
 * Copyright 2020-2025 NXP
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <stdio.h>
#include <stdlib.h>

#include "usb_device_config.h"
#include "usb.h"
#include "usb_device.h"

#include "usb_device_class.h"
#include "usb_device_audio.h"
#include "usb_device_descriptor.h"

#include "fsl_device_registers.h"
#include "fsl_debug_console.h"
#if (defined(FSL_FEATURE_SOC_SYSMPU_COUNT) && (FSL_FEATURE_SOC_SYSMPU_COUNT > 0U))
#include "fsl_sysmpu.h"
#endif /* FSL_FEATURE_SOC_SYSMPU_COUNT */

#if (defined(FSL_FEATURE_SOC_USBPHY_COUNT) && (FSL_FEATURE_SOC_USBPHY_COUNT > 0U))
#include "usb_phy.h"
#endif

#include "pin_mux.h"

#include "GlobalDef.h"

#if EnableCodec==1
	#include "fsl_codec_common.h"
	#include "fsl_codec_adapter.h"
#endif


#include "main.h"
#include "clock_config.h"
#include "board.h"
#include "CircularBuf.h"
#include "SubFunc.h"
#include "AudioDmaIO.h"
#include "Sweep.h"

#include "NxpCm7ASRC.h"
#include "AsrcProcess.h"

#define EnableKd							0

#define Def_KpValueInPNoIControl			(1/3000.0f)			//use this value when AUDIO_SPEAKER_UsbDnBufLengthInMs is 5  (5*48=240 samples)

//default starting values
#define Def_KpValueAtFsIn48KHz				(1.0f/  39500.0f)
#define Def_KiValueAtFsIn48KHz				(1.0f/1750000.0f)
#define Def_KdValueAtFsIn48KHz				(-1.0f/17000.0f)
#define Def_PI_ErrAccMaxValueAtFsIn48KHz	(100000.0f)

extern void UpdateAudioPll(float r);

T_ASRCProcessor AsrcProcessor1;

int BufAmountOfData;
float AsrcInputAudioBuf[AudioFrameSizeInSamplePerCh*2];

//#define AOD_AverageLength	16
#define AOD_AverageLength	8
//#define AOD_AverageLength	4
int AsrcBufAmountOfData[AOD_AverageLength];

float Kp;
float Ki;
float Kd;
int PI_Control_SampleAOD_ErrAccumulated;
int PI_ErrAccMaxValue;
int PreErr;

//set PID coefficients to default starting values
float KpValueInPNoIControl=Def_KpValueInPNoIControl;

float KpValueAtFsIn48KHz=Def_KpValueAtFsIn48KHz;
float KiValueAtFsIn48KHz=Def_KiValueAtFsIn48KHz;
float KdValueAtFsIn48KHz=Def_KdValueAtFsIn48KHz;
float PI_ErrAccMaxValueAtFsIn48KHz=Def_PI_ErrAccMaxValueAtFsIn48KHz;

float AsrcDriftingValueCurrent;
float AsrcDriftingValueTarget;
int AsrcFsInCurrent;
int AsrcFsInTarget;

#if EnableAsrcLatencyTest==1
	int AsrcLatencyTestCnt=0;
#endif
int InitASRCIsSuccessful(void)
{
    memset(&AsrcProcessor1,0,sizeof(AsrcProcessor1));

    //											Asrc         fs in, ratio,    input chunk size          sub fir length,      to use cubic,  channel number
    if(ReturnStat_Success!=InitAsrcProcessor(&AsrcProcessor1,48000,1.0399f,AudioFrameSizeInSamplePerCh,ASRC_QualityHigh,     1,             2))
    	return 0;

    AsrcDriftingValueCurrent=1.0399f;
    AsrcDriftingValueTarget=1.0399f;
    AsrcFsInCurrent=48000;
    AsrcFsInTarget=48000;

    return 1;
}

__attribute__((__section__("CodeQuickAccess")))
void AsrcProcessAndSaveToCirBuf(float *SrcPtr)
{
	int AsrcOutputSampleNum;
	int AmountOfDataAcc=0;

	long long TmpS32MixedBuf[AudioFrameSizeInSamplePerCh+1];		//at most AudioFrameSizeInSamplePerCh+1 samples (pair) will be generated

	DbgPin2Up;
	//step 1: calculate the averaged Amound Of Data of CirBufUsbAudioDnStreamingL --- to get AmountOfDataAcc
	for(int i=0;i<(AOD_AverageLength-1);i++)
	{
		AsrcBufAmountOfData[i]=AsrcBufAmountOfData[i+1];
		AmountOfDataAcc+=AsrcBufAmountOfData[i+1];
	}
	AsrcBufAmountOfData[AOD_AverageLength-1]=BufAmountOfData;
	AmountOfDataAcc+=BufAmountOfData;

	//step 2: PID calculation (P control or PID control) --- to get AsrcDriftingValueTarget
	int Err;

	#if AOD_AverageLength==16
		Err=(float)(AUDIO_SPEAKER_UsbDnBufCenterLevelInSamples)-(float)(AmountOfDataAcc>>4);
	#endif
	#if AOD_AverageLength==8
		Err=(float)(AUDIO_SPEAKER_UsbDnBufCenterLevelInSamples)-(float)(AmountOfDataAcc>>3);
	#endif
	#if AOD_AverageLength==4
		Err=(float)(AUDIO_SPEAKER_UsbDnBufCenterLevelInSamples)-(float)(AmountOfDataAcc>>2);
	#endif

	//update PID coefficients if needed
	if(ControlPara[3])
	{
		if(ControlPara[3]==1)
		{
			//need to update Kp Ki Kd values from CM33 side VCOM
			KpValueInPNoIControl=1.0f/ControlPara[4];

			KpValueAtFsIn48KHz=1.0f/(int)ControlPara[4];
			KiValueAtFsIn48KHz=1.0f/(int)ControlPara[5];
			KdValueAtFsIn48KHz=1.0f/(int)ControlPara[6];
			PI_ErrAccMaxValueAtFsIn48KHz=(int)ControlPara[7];

			ControlPara[3]=0;
		}else if(ControlPara[3]<=3)
		{
			//need to update interp method, cubic or linear
			UpdateAsrcInterp(&AsrcProcessor1,ControlPara[3]==3);		//ControlPara[3]=3: cubic, ControlPara[3]=[2]: linear
			ControlPara[3]=0;
		}
	}

	if(ControlPara[2]==1)
	{
		//not in USB audio downstreaming playing, PNoI control --- no need to decrease Kp by *0.05
		#if LocalI2SFsIs44p1KHz==1
			Kp=KpValueInPNoIControl*4;
		#else
			Kp=KpValueInPNoIControl*0.02;
		#endif		
		AsrcDriftingValueTarget=Kp*Err;
		PI_Control_SampleAOD_ErrAccumulated=0;
		PreErr=0;
	}else
	{
		//PID control
		if(AsrcFsInTarget==48000)
		{
			#if LocalI2SFsIs44p1KHz==1
				Kp=KpValueAtFsIn48KHz*0.5f;
				Ki=KiValueAtFsIn48KHz*0.1f;
				PI_ErrAccMaxValue=PI_ErrAccMaxValueAtFsIn48KHz*50;
			#else
				Kp=KpValueAtFsIn48KHz;
				Ki=KiValueAtFsIn48KHz;
				PI_ErrAccMaxValue=PI_ErrAccMaxValueAtFsIn48KHz;
			#endif

			Kd=KdValueAtFsIn48KHz;
		}else
		{
			//in this demo, we only test with AsrcFsIn=48000
		}


		PI_Control_SampleAOD_ErrAccumulated+=Err;
		if(PI_Control_SampleAOD_ErrAccumulated>PI_ErrAccMaxValue)
			PI_Control_SampleAOD_ErrAccumulated=PI_ErrAccMaxValue;
		if(PI_Control_SampleAOD_ErrAccumulated<-PI_ErrAccMaxValue)
			PI_Control_SampleAOD_ErrAccumulated=-PI_ErrAccMaxValue;

		#if EnableKd==1
			AsrcDriftingValueTarget=Kp*Err+Ki*PI_Control_SampleAOD_ErrAccumulated+Kd*(Err-PreErr);
		#else
			AsrcDriftingValueTarget=Kp*Err+Ki*PI_Control_SampleAOD_ErrAccumulated;
		#endif

		PreErr=Err;
	}

	if(ControlPara[0]==AudioDataRateMathing_NxpCM7Asrc)
	{
		//software ASRC can do +/-0.04 drifting

		#if LocalI2SFsIs44p1KHz==1
			//no more than +/-15%
			if(AsrcDriftingValueTarget> 0.14999f) AsrcDriftingValueTarget= 0.14999f;
			if(AsrcDriftingValueTarget<-0.14999f) AsrcDriftingValueTarget=-0.14999f;
		#else
			//no more than +/-4%
			if(AsrcDriftingValueTarget> 0.03999f) AsrcDriftingValueTarget= 0.03999f;
			if(AsrcDriftingValueTarget<-0.03999f) AsrcDriftingValueTarget=-0.03999f;
		#endif

		if(AsrcDriftingValueCurrent!=AsrcDriftingValueTarget)
		{
			UpdateAsrcRatio(&AsrcProcessor1,1.0+AsrcDriftingValueTarget);
			AsrcDriftingValueCurrent=AsrcDriftingValueTarget;
		}
	}else
	{
		//Audio PLL adjust can do -0.023 ~ + 0.037 drifting
		//can only adjust Audio PLL Numerator, so it makes a ration between 32.000~33.999 this is +3.7% of 32.768 and -2.3% of 32.768
		//so, adjust audio PLL range is a smaller than software ASRC
		if(AsrcDriftingValueTarget> 0.037f) AsrcDriftingValueTarget= 0.037f;
		if(AsrcDriftingValueTarget<-0.023) AsrcDriftingValueTarget=-0.023f;
		if(AsrcDriftingValueCurrent!=AsrcDriftingValueTarget)
		{
			UpdateAudioPll(1-AsrcDriftingValueTarget);
			AsrcDriftingValueCurrent=AsrcDriftingValueTarget;
		}
		UpdateAsrcRatio(&AsrcProcessor1,1.0);
	}

	DbgPin2Dn;

	//step 3: check if need to use internal signal generator to over-write the received external I2S audio data
	if(ControlPara[1])
	{
		DbgPin4Up;
		//to use internal signal generator as filter testing signal source
		if(ControlPara[1]==1)
			//sweeping is selected
			GenerateSineTone(&SineToneGenerator1, SrcPtr, AudioFrameSizeInSamplePerCh,1);
		else
			//1KHz is selected
			GenerateSineToneSingleFreq(&SineToneGenerator2, SrcPtr, AudioFrameSizeInSamplePerCh,1);
		DbgPin4Dn;

		//copy the tone of the first generator to the next ch with *-1, and make LR mixed
		#if EnableAsrcLatencyTest==1
			if((AsrcLatencyTestCnt%20000)<1000)
				for(int i=AudioFrameSizeInSamplePerCh-1;i>=0;i--)
				{
					SrcPtr[2*i+1]=-SrcPtr[i];
					SrcPtr[2*i+0]= SrcPtr[i]*0.5f;
				}
			else
				for(int i=AudioFrameSizeInSamplePerCh-1;i>=0;i--)
				{
					SrcPtr[2*i+1]=-SrcPtr[i];
					SrcPtr[2*i+0]= SrcPtr[i];
				}

		#else
			for(int i=AudioFrameSizeInSamplePerCh-1;i>=0;i--)
			{
				SrcPtr[2*i+1]=-SrcPtr[i];
				SrcPtr[2*i+0]= SrcPtr[i];
			}
		#endif
	}

	//step 4: call the main ASRC process
	GET_CYCLE_COUNTER(CycCntA);
		DbgPin2Up;

		if(ControlPara[0]==AudioDataRateMathing_NxpCM7Asrc)
		{
			//need to run ASRC, convert SrcPtr to TmpS32MixedBuf, input sample number is 6, output sample number will be written in AsrcOutputSampleNum
			AsrcOutputSampleNum=AsrcConvertOneFrame(&AsrcProcessor1,(float *)TmpS32MixedBuf,SrcPtr);
		}else
		{
			//now using audio PLL adjusting, no need to run ASRC, just copy 6 samples in SrcPtr to TmpS32MixedBuf
			memcpy(TmpS32MixedBuf,SrcPtr,AsrcProcessor1.InputFrameLengthInSample*sizeof(float)*AsrcProcessor1.ChNumber);
			AsrcOutputSampleNum=AsrcProcessor1.InputFrameLengthInSample;
		}

		DbgPin2Dn;
	GET_CYCLE_COUNTER(CycCntB);

	//step 5: put the generated audio to circular buffer
	switch(AsrcOutputSampleNum)
	{
		case 5:
			MoveFloatSamplesToS32Buf_5SampleGrouped((int *)TmpS32MixedBuf,(float *)TmpS32MixedBuf,AsrcProcessor1.ChNumber);	//TmpS32MixedBuf contains LR mixed samples, actually convert 10 samples
			break;
		case 6:
			MoveFloatSamplesToS32Buf_6SampleGrouped((int *)TmpS32MixedBuf,(float *)TmpS32MixedBuf,AsrcProcessor1.ChNumber);	//TmpS32MixedBuf contains LR mixed samples, actually convert 12 samples
			break;
		case 7:
			MoveFloatSamplesToS32Buf_7SampleGrouped((int *)TmpS32MixedBuf,(float *)TmpS32MixedBuf,AsrcProcessor1.ChNumber);	//TmpS32MixedBuf contains LR mixed samples, actually convert 14 samples
			break;
		default:
			PRINTF("8\r\n");		//should not happen
			break;
	}

	//save the current USB audio packet to the circular buffer
	DbgPin2Up;
	USB_AUDIO_ENTER_CRITICAL();
	if (CirAudioBuf_SpaceAvailableInSamples_S64(&UsbDnStrmCirBuf) >= AsrcOutputSampleNum)
	{
		CirAudioBuf_WriteSamples_S64(&UsbDnStrmCirBuf, AsrcOutputSampleNum, TmpS32MixedBuf);
	}else
	{
		PRINTF("F\r\n");
	}
	USB_AUDIO_EXIT_CRITICAL();

	//step 6: put some debug values to ComWatchValue, and will be printed by MCU side to the USB COM host
	ComWatchValue[0]= PI_Control_SampleAOD_ErrAccumulated;
    ComWatchValue[1]= AsrcDriftingValueTarget*1000000;
    ComWatchValue[2]=BufAmountOfData;
    ComWatchValue[3]=CycCntB-CycCntA;

	DbgPin2Dn;
}


