/*
 * Copyright 2020-2025 NXP
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */


#include "usb_device_config.h"
#include "usb.h"
#include "usb_device.h"

#include "usb_device_class.h"
#include "usb_device_audio.h"

#include "usb_device_descriptor.h"

#include "GlobalDef.h"

#include "main.h"

#include "fsl_device_registers.h"
#include "clock_config.h"
#include "board.h"
#include <stdio.h>
#include <stdlib.h>

#include "CircularBuf.h"
#include "SubFunc.h"
#include "AudioDmaIO.h"
#include "NxpCm7ASRC.h"
#include "AsrcProcess.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define USB_AUDIO_ENTER_CRITICAL() 			\
				OSA_SR_ALLOC();            	\
				OSA_ENTER_CRITICAL()
#define USB_AUDIO_EXIT_CRITICAL() OSA_EXIT_CRITICAL()
/*******************************************************************************
 * Prototypes
 ******************************************************************************/
usb_status_t USB_DeviceAudioCallback(class_handle_t handle, uint32_t event, void *param);
usb_status_t USB_DeviceCallback(usb_device_handle handle, uint32_t event, void *param);

/*******************************************************************************
 * Variables
 ******************************************************************************/
USB_DMA_NONINIT_DATA_ALIGN(USB_DATA_ALIGN_SIZE)

#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
	USB_DMA_NONINIT_DATA_ALIGN(8)
	uint8_t audioPlayPacket[FS_ISO_OUT_ENDP_PACKET_SIZE];
#else
	USB_DMA_NONINIT_DATA_ALIGN(8)
	uint8_t audioPlayPacket[(FS_ISO_OUT_ENDP_PACKET_SIZE + AUDIO_OUT_FORMAT_CHANNELS * AUDIO_OUT_FORMAT_SIZE)];

	USB_DMA_NONINIT_DATA_ALIGN(USB_DATA_ALIGN_SIZE)
	uint8_t usbAudioFeedBackBuffer[4];

	USB_DMA_NONINIT_DATA_ALIGN(USB_DATA_ALIGN_SIZE)
	uint8_t audioFeedBackBuffer[4];

	volatile uint8_t feedbackValueUpdating=0;
#endif

volatile int AsrcInputBufIsReady=0;

usb_device_composite_struct_t *g_deviceAudioComposite;
volatile bool g_CodecMuteUnmute = false;
extern usb_device_composite_struct_t g_composite;

//extern uint8_t s_wavBuff[];
/*******************************************************************************
 * Code
 ******************************************************************************/

#if ((defined(USB_DEVICE_CONFIG_AUDIO)) && (USB_DEVICE_CONFIG_AUDIO > 0U))

/*!
 * @brief Audio class specific request function.
 *
 * This function handles the Audio class specific requests.
 *
 * @param handle           The USB device handle.
 * @param event            The USB device event type.
 * @param param            The parameter of the device specific request.
 *
 * @return A USB error code or kStatus_USB_Success.
 */
__attribute__((__section__("CodeQuickAccess")))
usb_status_t USB_DeviceAudioRequest(class_handle_t handle, uint32_t event, void *param) {
    usb_device_control_request_struct_t *request = (usb_device_control_request_struct_t *)param;
    usb_status_t error                           = kStatus_USB_Success;
#if (!USB_DEVICE_CONFIG_AUDIO_CLASS_2_0)
    uint16_t volume;
#endif

    switch (event)
    {
        case USB_DEVICE_AUDIO_FU_GET_CUR_MUTE_CONTROL:
#if (USB_DEVICE_CONFIG_AUDIO_CLASS_2_0)
            request->buffer = (uint8_t *)&g_deviceAudioComposite->audioUnified.curMute20;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.curMute20);
#else
            request->buffer = &g_deviceAudioComposite->audioUnified.curMute;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.curMute);
#endif
            break;
        case USB_DEVICE_AUDIO_FU_GET_CUR_VOLUME_CONTROL:
#if (USB_DEVICE_CONFIG_AUDIO_CLASS_2_0)
            request->buffer = (uint8_t *)&g_deviceAudioComposite->audioUnified.curVolume20;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.curVolume20);
#else
            request->buffer = g_deviceAudioComposite->audioUnified.curVolume;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.curVolume);
#endif
            break;
        case USB_DEVICE_AUDIO_FU_GET_CUR_BASS_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.curBass;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.curBass);
            break;
        case USB_DEVICE_AUDIO_FU_GET_CUR_MID_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.curMid;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.curMid);
            break;
        case USB_DEVICE_AUDIO_FU_GET_CUR_TREBLE_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.curTreble;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.curTreble);
            break;
        case USB_DEVICE_AUDIO_FU_GET_CUR_AUTOMATIC_GAIN_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.curAutomaticGain;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.curAutomaticGain);
            break;
        case USB_DEVICE_AUDIO_FU_GET_CUR_DELAY_CONTROL:
            request->buffer = g_deviceAudioComposite->audioUnified.curDelay;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.curDelay);
            break;
        case USB_DEVICE_AUDIO_FU_GET_MIN_VOLUME_CONTROL:
            request->buffer = g_deviceAudioComposite->audioUnified.minVolume;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.minVolume);
            break;
        case USB_DEVICE_AUDIO_FU_GET_MIN_BASS_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.minBass;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.minBass);
            break;
        case USB_DEVICE_AUDIO_FU_GET_MIN_MID_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.minMid;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.minMid);
            break;
        case USB_DEVICE_AUDIO_FU_GET_MIN_TREBLE_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.minTreble;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.minTreble);
            break;
        case USB_DEVICE_AUDIO_FU_GET_MIN_DELAY_CONTROL:
            request->buffer = g_deviceAudioComposite->audioUnified.minDelay;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.minDelay);
            break;
        case USB_DEVICE_AUDIO_FU_GET_MAX_VOLUME_CONTROL:
            request->buffer = g_deviceAudioComposite->audioUnified.maxVolume;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.maxVolume);
            break;
        case USB_DEVICE_AUDIO_FU_GET_MAX_BASS_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.maxBass;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.maxBass);
            break;
        case USB_DEVICE_AUDIO_FU_GET_MAX_MID_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.maxMid;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.maxMid);
            break;
        case USB_DEVICE_AUDIO_FU_GET_MAX_TREBLE_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.maxTreble;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.maxTreble);
            break;
        case USB_DEVICE_AUDIO_FU_GET_MAX_DELAY_CONTROL:
            request->buffer = g_deviceAudioComposite->audioUnified.maxDelay;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.maxDelay);
            break;
        case USB_DEVICE_AUDIO_FU_GET_RES_VOLUME_CONTROL:
            request->buffer = g_deviceAudioComposite->audioUnified.resVolume;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.resVolume);
            break;
        case USB_DEVICE_AUDIO_FU_GET_RES_BASS_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.resBass;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.resBass);
            break;
        case USB_DEVICE_AUDIO_FU_GET_RES_MID_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.resMid;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.resMid);
            break;
        case USB_DEVICE_AUDIO_FU_GET_RES_TREBLE_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.resTreble;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.resTreble);
            break;
        case USB_DEVICE_AUDIO_FU_GET_RES_DELAY_CONTROL:
            request->buffer = g_deviceAudioComposite->audioUnified.resDelay;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.resDelay);
            break;
#if USB_DEVICE_CONFIG_AUDIO_CLASS_2_0
        case USB_DEVICE_AUDIO_CS_GET_CUR_SAMPLING_FREQ_CONTROL:
            request->buffer = (uint8_t *)&g_deviceAudioComposite->audioUnified.curSampleFrequency;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.curSampleFrequency);
            break;
        case USB_DEVICE_AUDIO_CS_SET_CUR_SAMPLING_FREQ_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = (uint8_t *)&g_deviceAudioComposite->audioUnified.curSampleFrequency;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curSampleFrequency);
            }
            break;
        case USB_DEVICE_AUDIO_CS_GET_CUR_CLOCK_VALID_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.curClockValid;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.curClockValid);
            break;
        case USB_DEVICE_AUDIO_CS_SET_CUR_CLOCK_VALID_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.curClockValid;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curClockValid);
            }
            break;
        case USB_DEVICE_AUDIO_CS_GET_RANGE_SAMPLING_FREQ_CONTROL:
            request->buffer = (uint8_t *)&g_deviceAudioComposite->audioUnified.freqControlRange;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.freqControlRange);
            break;
        case USB_DEVICE_AUDIO_FU_GET_RANGE_VOLUME_CONTROL:
            request->buffer = (uint8_t *)&g_deviceAudioComposite->audioUnified.volumeControlRange;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.volumeControlRange);
            break;
#else
        case USB_DEVICE_AUDIO_EP_GET_CUR_SAMPLING_FREQ_CONTROL:
            request->buffer = g_deviceAudioComposite->audioUnified.curSamplingFrequency;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.curSamplingFrequency);
            break;
        case USB_DEVICE_AUDIO_EP_GET_MIN_SAMPLING_FREQ_CONTROL:
            request->buffer = g_deviceAudioComposite->audioUnified.minSamplingFrequency;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.minSamplingFrequency);
            break;
        case USB_DEVICE_AUDIO_EP_GET_MAX_SAMPLING_FREQ_CONTROL:
            request->buffer = g_deviceAudioComposite->audioUnified.maxSamplingFrequency;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.maxSamplingFrequency);
            break;
        case USB_DEVICE_AUDIO_EP_GET_RES_SAMPLING_FREQ_CONTROL:
            request->buffer = g_deviceAudioComposite->audioUnified.resSamplingFrequency;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.resSamplingFrequency);
            break;
        case USB_DEVICE_AUDIO_EP_SET_CUR_SAMPLING_FREQ_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = g_deviceAudioComposite->audioUnified.curSamplingFrequency;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curSamplingFrequency);
            }
            break;
        case USB_DEVICE_AUDIO_EP_SET_RES_SAMPLING_FREQ_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = g_deviceAudioComposite->audioUnified.resSamplingFrequency;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.resSamplingFrequency);
            }
            break;
        case USB_DEVICE_AUDIO_EP_SET_MAX_SAMPLING_FREQ_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = g_deviceAudioComposite->audioUnified.maxSamplingFrequency;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.maxSamplingFrequency);
            }
            break;
        case USB_DEVICE_AUDIO_EP_SET_MIN_SAMPLING_FREQ_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = g_deviceAudioComposite->audioUnified.minSamplingFrequency;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.minSamplingFrequency);
            }
            break;
#endif
        case USB_DEVICE_AUDIO_FU_SET_CUR_VOLUME_CONTROL:
#if (USB_DEVICE_CONFIG_AUDIO_CLASS_2_0)
            if (request->isSetup == 1U)
            {
                request->buffer = g_deviceAudioComposite->audioUnified.curVolume20;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curVolume20);
            }
            else
            {
                g_deviceAudioComposite->audioUnified.codecTask |= VOLUME_CHANGE_TASK;
            }
#else
            if (request->isSetup == 1U)
            {
                request->buffer = g_deviceAudioComposite->audioUnified.curVolume;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curVolume);
            }
            else
            {
                volume = (uint16_t)((uint16_t)g_deviceAudioComposite->audioUnified.curVolume[1] << 8U);
                volume |= (uint8_t)(g_deviceAudioComposite->audioUnified.curVolume[0]);
                g_deviceAudioComposite->audioUnified.codecTask |= VOLUME_CHANGE_TASK;
            }
#endif
            break;
        case USB_DEVICE_AUDIO_FU_SET_CUR_MUTE_CONTROL:
#if (USB_DEVICE_CONFIG_AUDIO_CLASS_2_0)
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.curMute20;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curMute20);
            }
            else
            {
                if (g_deviceAudioComposite->audioUnified.curMute20)
                {
                    g_deviceAudioComposite->audioUnified.codecTask |= MUTE_CODEC_TASK;
                }
                else
                {
                    g_deviceAudioComposite->audioUnified.codecTask |= UNMUTE_CODEC_TASK;
                }
            }
#else
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.curMute;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curMute);
            }
            else
            {
                if (g_deviceAudioComposite->audioUnified.curMute)
                {
                    g_deviceAudioComposite->audioUnified.codecTask |= MUTE_CODEC_TASK;
                }
                else
                {
                    g_deviceAudioComposite->audioUnified.codecTask |= UNMUTE_CODEC_TASK;
                }
            }
#endif
            break;
        case USB_DEVICE_AUDIO_FU_SET_CUR_BASS_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.curBass;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curBass);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_CUR_MID_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.curMid;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curMid);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_CUR_TREBLE_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.curTreble;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curTreble);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_CUR_AUTOMATIC_GAIN_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.curAutomaticGain;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curAutomaticGain);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_CUR_DELAY_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = g_deviceAudioComposite->audioUnified.curDelay;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curDelay);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_MIN_VOLUME_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = g_deviceAudioComposite->audioUnified.minVolume;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.minVolume);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_MIN_BASS_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.minBass;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.minBass);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_MIN_MID_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.minMid;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.minMid);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_MIN_TREBLE_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.minTreble;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.minTreble);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_MIN_DELAY_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = g_deviceAudioComposite->audioUnified.minDelay;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.minDelay);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_MAX_VOLUME_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = g_deviceAudioComposite->audioUnified.maxVolume;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.maxVolume);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_MAX_BASS_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.maxBass;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.maxBass);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_MAX_MID_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.maxMid;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.maxMid);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_MAX_TREBLE_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.maxTreble;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.maxTreble);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_MAX_DELAY_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = g_deviceAudioComposite->audioUnified.maxDelay;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.maxDelay);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_RES_VOLUME_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = g_deviceAudioComposite->audioUnified.resVolume;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.resVolume);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_RES_BASS_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.resBass;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.resBass);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_RES_MID_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.resMid;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.resMid);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_RES_TREBLE_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.resTreble;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.resTreble);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_RES_DELAY_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = g_deviceAudioComposite->audioUnified.resDelay;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.resDelay);
            }
            break;
        default:
            error = kStatus_USB_InvalidRequest;
            break;
    }
    return error;
}


#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
#else
/* The USB_AudioFeedbackDataUpdate() function calculates the feedback data */
__attribute__((__section__(".data.$SRAM_DTC")))
uint32_t USBAudio_FeedBackEp_feedbackValue=(AUDIO_OUT_SAMPLING_RATE_KHZ*1024);
__attribute__((__section__("CodeQuickAccess")))
void USB_AudioFeedbackDataUpdate()
{

    uint32_t audioSpeakerUsedSpace;


    /* feedback interval is AUDIO_CALCULATE_Ff_INTERVAL */
    if (USB_SPEED_HIGH == g_deviceAudioComposite->speed)
    {
#if (USB_DEVICE_CONFIG_AUDIO_CLASS_2_0) /* high speed, feedback interval is AUDIO_CALCULATE_Ff_INTERVAL ms */
        if (g_deviceAudioComposite->audioUnified.speakerIntervalCount !=
            AUDIO_CALCULATE_Ff_INTERVAL *
                (AUDIO_PLAY_BUFFER_SIZE_ONE_FRAME / g_deviceAudioComposite->audioUnified.audioPlayTransferSize))
#else
        if (g_deviceAudioComposite->audioUnified.speakerIntervalCount != AUDIO_CALCULATE_Ff_INTERVAL)
#endif
        {
            g_deviceAudioComposite->audioUnified.speakerIntervalCount++;
            return;
        }
    }
    else /* full speed, feedback interval is AUDIO_CALCULATE_Ff_INTERVAL ms */
    {
        if (g_deviceAudioComposite->audioUnified.speakerIntervalCount != AUDIO_CALCULATE_Ff_INTERVAL)
        {
            g_deviceAudioComposite->audioUnified.speakerIntervalCount++;
            return;
        }
    }

    g_deviceAudioComposite->audioUnified.speakerIntervalCount = 1;

    g_deviceAudioComposite->audioUnified.timesFeedbackCalculate++;

    if (g_deviceAudioComposite->audioUnified.timesFeedbackCalculate == 2)
    {
        USBAudio_FeedBackEp_feedbackValue = (AUDIO_OUT_SAMPLING_RATE_KHZ*1024);
        //PRINTF("%c%c",(USBAudio_FeedBackEp_feedbackValue>>8)&0xff,USBAudio_FeedBackEp_feedbackValue&0xff);//�䨰��?originFeedbackValue��?�̨�16??

		if (USB_SPEED_HIGH == g_composite.speed)
		{
	        AUDIO_UPDATE_FEEDBACK_DATA(audioFeedBackBuffer, USBAudio_FeedBackEp_feedbackValue/8);
		}
		if (USB_SPEED_FULL == g_composite.speed)
		{
	        AUDIO_UPDATE_FEEDBACK_DATA(audioFeedBackBuffer, USBAudio_FeedBackEp_feedbackValue);
		}
    }
    else if (g_deviceAudioComposite->audioUnified.timesFeedbackCalculate > 2)
    {
        int Delta;
        audioSpeakerUsedSpace = CirAudioBuf_SpaceOccupiedInSamples_S64(&UsbDnStrmCirBuf);

        Delta=audioSpeakerUsedSpace-AUDIO_SPEAKER_UsbDnBufCenterLevelInSamples;

#if (USB_DEVICE_CONFIG_AUDIO_CLASS_2_0)
		if (USB_SPEED_HIGH == g_composite.speed)
		{
	        USBAudio_FeedBackEp_feedbackValue = (AUDIO_OUT_SAMPLING_RATE_KHZ*1024)-(Delta/2);
	        AUDIO_UPDATE_FEEDBACK_DATA(audioFeedBackBuffer, USBAudio_FeedBackEp_feedbackValue/8);
		}
		if (USB_SPEED_FULL == g_composite.speed)
		{
	        USBAudio_FeedBackEp_feedbackValue = (AUDIO_OUT_SAMPLING_RATE_KHZ*1024)-2*(Delta/48);
	        AUDIO_UPDATE_FEEDBACK_DATA(audioFeedBackBuffer, USBAudio_FeedBackEp_feedbackValue);
		}
#else
		USBAudio_FeedBackEp_feedbackValue = (AUDIO_OUT_SAMPLING_RATE_KHZ*1024)-2*(Delta/48);
		AUDIO_UPDATE_FEEDBACK_DATA(audioFeedBackBuffer, USBAudio_FeedBackEp_feedbackValue);
#endif
    }
    else
    {
        USBAudio_FeedBackEp_feedbackValue = (AUDIO_OUT_SAMPLING_RATE_KHZ*1024);

		if (USB_SPEED_HIGH == g_composite.speed)
		{
	        AUDIO_UPDATE_FEEDBACK_DATA(audioFeedBackBuffer, USBAudio_FeedBackEp_feedbackValue/8);
		}
		if (USB_SPEED_FULL == g_composite.speed)
		{
	        AUDIO_UPDATE_FEEDBACK_DATA(audioFeedBackBuffer, USBAudio_FeedBackEp_feedbackValue);
		}

    }

    g_deviceAudioComposite->audioUnified.lastAudioSendCount = g_deviceAudioComposite->audioUnified.audioSendCount;
}
#endif

__attribute__((__section__("CodeQuickAccess")))
uint32_t USB_MicUpStreamDataRateControl_AdjustPacketLength(T_CircularAudioBuf_MultiCh *CirBufPtr)
{
	uint16_t AvailableSamplesInCirBuf;
	uint32_t PacketSize;
	uint16_t HiLimitInSample, LoLimitInSample;

	if(CirBufPtr->LengthInSamples < AUDIO_IN_SAMPLING_RATE_KHZ * 16 )
	{	//buffer length is shorter than 16ms  --- assume 12ms
		HiLimitInSample=AUDIO_IN_SAMPLING_RATE_KHZ * 8;
		LoLimitInSample=AUDIO_IN_SAMPLING_RATE_KHZ * 4;
	}
	else if(CirBufPtr->LengthInSamples < AUDIO_IN_SAMPLING_RATE_KHZ * 32 )
	{	//buffer length is 16ms ~ 32ms        --- assume 16ms
		HiLimitInSample=AUDIO_IN_SAMPLING_RATE_KHZ * 10;
		LoLimitInSample=AUDIO_IN_SAMPLING_RATE_KHZ * 4;
	}
	else if(CirBufPtr->LengthInSamples < AUDIO_IN_SAMPLING_RATE_KHZ * 64 )
	{	//buffer length is 32ms ~ 64ms        --- assume 32ms
		HiLimitInSample=AUDIO_IN_SAMPLING_RATE_KHZ * 28;
		LoLimitInSample=AUDIO_IN_SAMPLING_RATE_KHZ * 16;
	}
	else if(CirBufPtr->LengthInSamples < AUDIO_IN_SAMPLING_RATE_KHZ * 128 )
	{	//buffer length is 64ms ~ 128ms       --- assume 64ms
		HiLimitInSample=AUDIO_IN_SAMPLING_RATE_KHZ * 50;
		LoLimitInSample=AUDIO_IN_SAMPLING_RATE_KHZ * 25;
	}
	else
	{	//buffer length is longer than 128ms  --- assume 128ms
		HiLimitInSample=AUDIO_IN_SAMPLING_RATE_KHZ * 100;
		LoLimitInSample=AUDIO_IN_SAMPLING_RATE_KHZ * 30;
	}

	AvailableSamplesInCirBuf = CirAudioBuf_SpaceOccupiedInSamples_MultiCh(CirBufPtr);
	if (AvailableSamplesInCirBuf > HiLimitInSample)
	{	//mic audio data is going to be too much
		PacketSize = AUDIO_IN_SAMPLING_RATE_KHZ + 1;	//increase one audio sample
		//usb_echo("+\r\n");
	} else if (AvailableSamplesInCirBuf > LoLimitInSample)
	{	//mic audio data is going to be not enough
		PacketSize = AUDIO_IN_SAMPLING_RATE_KHZ;	    //no increase no decrease
		//usb_echo("x\r\n");
	} else if (AvailableSamplesInCirBuf > AUDIO_IN_SAMPLING_RATE_KHZ)
	{	//mic audio data is going to be not enough
		PacketSize = AUDIO_IN_SAMPLING_RATE_KHZ - 1;	//decrease one audio sample
		//usb_echo("-\r\n");
	} else
	{
		PacketSize = 0;	    //this means there is almost no audio data to upstream
		//usb_echo("0\r\n");
	}
	return PacketSize;
}

uint16_t MicUpStreamDataRateAdjustCnt = 0;

/*!
 * @brief device Audio callback function.
 *
 * This function handle the Audio class specified event.
 * @param handle          The USB class  handle.
 * @param event           The USB device event type.
 * @param param           The parameter of the class specific event.
 * @return kStatus_USB_Success or error.
 */

__attribute__((__section__("CodeQuickAccess")))
usb_status_t USB_DeviceAudioCompositeCallback(class_handle_t handle, uint32_t event, void *param)
{
	usb_status_t error = kStatus_USB_Error;
	usb_device_endpoint_callback_message_struct_t *ep_cb_param;
	ep_cb_param = (usb_device_endpoint_callback_message_struct_t*) param;
	switch (event) {
	case kUSB_DeviceAudioEventStreamSendResponse:
		if ((g_deviceAudioComposite->audioUnified.attach) && (ep_cb_param->length != (USB_UNINITIALIZED_VAL_32)))
		{
#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
			if (0)
			{
			}
#else
			if (ep_cb_param->length == ((USB_SPEED_HIGH == g_composite.speed) ? HS_ISO_FEEDBACK_ENDP_PACKET_SIZE : FS_ISO_FEEDBACK_ENDP_PACKET_SIZE))
			{
				USB_AUDIO_ENTER_CRITICAL();
			
                if (!feedbackValueUpdating)
                {
                    *((uint32_t *)&usbAudioFeedBackBuffer[0]) = *((uint32_t *)&audioFeedBackBuffer[0]);
                }
				error = USB_DeviceAudioSend(g_deviceAudioComposite->audioUnified.audioSpeakerHandle,
				USB_AUDIO_SPEAKER_FEEDBACK_ENDPOINT, usbAudioFeedBackBuffer, (USB_SPEED_HIGH == g_composite.speed) ? HS_ISO_FEEDBACK_ENDP_PACKET_SIZE :
				FS_ISO_FEEDBACK_ENDP_PACKET_SIZE);
				
				USB_AUDIO_EXIT_CRITICAL();
			}
#endif
			else
			{
				//DbgPin4Up;
				USB_AUDIO_ENTER_CRITICAL();
				UsbAudioUpStreamingIsStarted = 1;

				if (!(MicUpStreamDataRateAdjustCnt % 5))
				{
					//adjust mic upstream data rate once every 5 or 8 or 16 packets --- adjust packet size, one sample more ore one sample less
					uint32_t l = USB_MicUpStreamDataRateControl_AdjustPacketLength(&UsbUpStrmCirBuf);

					//l=48;

					if (l)
					{
						int *UsbUpStrmDataPtr;
						UsbUpStrmDataPtr = (int *)CirAudioBuf_ReadSamples_GetRdPtr_MultiCh(&UsbUpStrmCirBuf, l);
						error = USB_DeviceAudioSend(g_deviceAudioComposite->audioUnified.audioRecorderHandle,
									USB_AUDIO_RECORDER_STREAM_ENDPOINT,
									(uint8_t*)UsbUpStrmDataPtr,
									l * AUDIO_IN_FORMAT_CHANNELS * AUDIO_IN_FORMAT_SIZE);
					} else
					{
						error = USB_DeviceAudioSend(g_deviceAudioComposite->audioUnified.audioRecorderHandle,
									USB_AUDIO_RECORDER_STREAM_ENDPOINT,
									(uint8_t *)AllZeroBuf_48PointsSingleCh_32Bit,
									FS_ISO_IN_ENDP_PACKET_SIZE);
					}
				} else
				{
					uint32_t space_occupied = CirAudioBuf_SpaceOccupiedInSamples_MultiCh(&UsbUpStrmCirBuf);
					if (space_occupied >= AUDIO_IN_SAMPLING_RATE_KHZ)
					{
						int *UsbUpStrmDataPtr;
						UsbUpStrmDataPtr = (int *)CirAudioBuf_ReadSamples_GetRdPtr_MultiCh(&UsbUpStrmCirBuf, AUDIO_IN_SAMPLING_RATE_KHZ);
						error = USB_DeviceAudioSend(g_deviceAudioComposite->audioUnified.audioRecorderHandle,
									USB_AUDIO_RECORDER_STREAM_ENDPOINT,
									(uint8_t*)UsbUpStrmDataPtr,
									FS_ISO_IN_ENDP_PACKET_SIZE);
					} else
					{
						error = USB_DeviceAudioSend(g_deviceAudioComposite->audioUnified.audioRecorderHandle,
									USB_AUDIO_RECORDER_STREAM_ENDPOINT,
									(uint8_t *)AllZeroBuf_48PointsSingleCh_32Bit,
									FS_ISO_IN_ENDP_PACKET_SIZE);
					}
				}
				MicUpStreamDataRateAdjustCnt++;
				USB_AUDIO_EXIT_CRITICAL();
				//DbgPin4Dn;
			}
		}
		break;
	case kUSB_DeviceAudioEventStreamRecvResponse:
		if ((g_deviceAudioComposite->audioUnified.attach) && (ep_cb_param->length != (USB_UNINITIALIZED_VAL_32)))
		{
			//DbgPin4Up;
			if (g_deviceAudioComposite->audioUnified.startPlay == 0)
			{
				g_deviceAudioComposite->audioUnified.startPlay = 1;
			}

			USB_AUDIO_ENTER_CRITICAL();
			if ((CirAudioBuf_SpaceOccupiedInSamples_S64(&UsbDnStrmCirBuf) >= AUDIO_SPEAKER_UsbDnBufCenterLevelInSamples) && (g_deviceAudioComposite->audioUnified.startPlayHalfFull == 0))
			{
				g_deviceAudioComposite->audioUnified.startPlayHalfFull = 1;
			}

			g_deviceAudioComposite->audioUnified.usbRecvCount += ep_cb_param->length;
			//g_deviceAudioComposite->audioUnified.usbRecvTimes++;
#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
			if (USB_SPEED_HIGH == g_composite.speed)
			{
				error = USB_DeviceAudioRecv(handle, USB_AUDIO_SPEAKER_STREAM_ENDPOINT, &audioPlayPacket[0], (HS_ISO_OUT_ENDP_PACKET_SIZE));
			}
			if (USB_SPEED_FULL == g_composite.speed)
			{
				error = USB_DeviceAudioRecv(handle, USB_AUDIO_SPEAKER_STREAM_ENDPOINT, &audioPlayPacket[0], (FS_ISO_OUT_ENDP_PACKET_SIZE));
			}
#else

			if(UacDnStreamSyncSelect==UsbAudioDnSyncByFeedbackEP)
			{
				//when using FeedbackEP for USB audio synchronization, call the feedback update function
				//USB_AUDIO_ENTER_CRITICAL();
				USB_AudioFeedbackDataUpdate();
				//USB_AUDIO_EXIT_CRITICAL();
			}else
			{
				//when not using FeedbackEP for USB audio synchronization, just feedback default value
				AUDIO_UPDATE_FEEDBACK_DATA(audioFeedBackBuffer, AUDIO_OUT_SAMPLING_RATE_KHZ*1024/8);
			}

			if (USB_SPEED_HIGH == g_composite.speed)
			{
				error = USB_DeviceAudioRecv(handle, USB_AUDIO_SPEAKER_STREAM_ENDPOINT, &audioPlayPacket[0], (HS_ISO_OUT_ENDP_PACKET_SIZE + AUDIO_OUT_FORMAT_CHANNELS * AUDIO_OUT_FORMAT_SIZE));
			}
			if (USB_SPEED_FULL == g_composite.speed)
			{
				error = USB_DeviceAudioRecv(handle, USB_AUDIO_SPEAKER_STREAM_ENDPOINT, &audioPlayPacket[0], (FS_ISO_OUT_ENDP_PACKET_SIZE + AUDIO_OUT_FORMAT_CHANNELS * AUDIO_OUT_FORMAT_SIZE));
			}
#endif
			BufAmountOfData=CirAudioBuf_SpaceOccupiedInSamples_S64(&UsbDnStrmCirBuf);

			if(UacDnStreamSyncSelect!=UsbAudioDnSyncByFeedbackEP)
			{
				//to ASRC process and save ASRC processed audio to the circular buffer in the main loop

				//when not half full, force all received audio to zeros
				if (g_deviceAudioComposite->audioUnified.startPlayHalfFull)
					memcpy(AsrcInputAudioBuf, audioPlayPacket, ep_cb_param->length);
				else
					memset(AsrcInputAudioBuf, 0, ep_cb_param->length);

				#if DoAsrcInUsbIntr==1
					MoveS32SamplesToFloatBuf_6SampleGrouped(AsrcInputAudioBuf, (int *)AsrcInputAudioBuf, 2);	//each frame, there are 6 sample pairs --- 12 samples total
					AsrcProcessAndSaveToCirBuf(AsrcInputAudioBuf);
				#else
					AsrcInputBufIsReady=1;
				#endif
			}else
			{
				//save the current USB audio packet to the circular buffer
				if (CirAudioBuf_SpaceAvailableInSamples_S64(&UsbDnStrmCirBuf) >= ep_cb_param->length/(AUDIO_OUT_FORMAT_CHANNELS * AUDIO_OUT_FORMAT_SIZE))
				{
					if (g_deviceAudioComposite->audioUnified.startPlayHalfFull == 1)
					{
						//half full play started --- copy real downstream audio in the data packet to the downstreaming (feedback calculation) buffer
						CirAudioBuf_WriteSamples_S64(&UsbDnStrmCirBuf, ep_cb_param->length/(AUDIO_OUT_FORMAT_CHANNELS * AUDIO_OUT_FORMAT_SIZE), (long long*)audioPlayPacket);
					} else
					{
						//half full play not started --- put 0 to buffer, this is to prevent dirty sampels from the host, seems there are dirty samples from the end of last time host playing
						CirAudioBuf_WriteSamples_S64(&UsbDnStrmCirBuf, ep_cb_param->length/(AUDIO_OUT_FORMAT_CHANNELS * AUDIO_OUT_FORMAT_SIZE), (long long*)AllZeroBuf_48PointsSingleCh_32Bit);	//this is to prevent play/pause click
																																					//--- starting playing may stream the audio data of the end of last playing
																																					//--- clr to 0 is to get rid of the tail of the last playing
					}
				}
			}
			USB_AUDIO_EXIT_CRITICAL();

			//DbgPin4Dn;
		}
		break;

	default:
		if (param && (event > 0xFF))
		{
			error = USB_DeviceAudioRequest(handle, event, param);
		}
		break;
	}

	return error;
}

/* The USB_DeviceAudioSpeakerStatusReset() function resets the audio speaker status to the initialized status */
void USB_DeviceAudioSpeakerStatusReset(void)
{
	g_deviceAudioComposite->audioUnified.startPlay = 0;
	g_deviceAudioComposite->audioUnified.startPlayHalfFull = 0;
	//g_deviceAudioComposite->audioUnified.UsbDnStrmBuf_WrPosInByte = 0;
	//g_deviceAudioComposite->audioUnified.UsbDnStrmBuf_RdPosInByte = 0;
	g_deviceAudioComposite->audioUnified.audioSendCount = 0;
	g_deviceAudioComposite->audioUnified.usbRecvCount = 0;
	g_deviceAudioComposite->audioUnified.lastAudioSendCount = 0;
	//g_deviceAudioComposite->audioUnified.audioSendTimes = 0;
	//g_deviceAudioComposite->audioUnified.usbRecvTimes = 0;
	g_deviceAudioComposite->audioUnified.speakerIntervalCount = 0;
	g_deviceAudioComposite->audioUnified.UsbDnStrmBuf_OccupiedSpaceInByte = 0;
	g_deviceAudioComposite->audioUnified.timesFeedbackCalculate = 0;
	g_deviceAudioComposite->audioUnified.speakerDetachOrNoInput = 0;
#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
    g_deviceAudioComposite->audioUnified.audioPllTicksPrev   = 0;
    g_deviceAudioComposite->audioUnified.audioPllTicksDiff   = 0;
    g_deviceAudioComposite->audioUnified.audioPllTicksEma    = AUDIO_PLL_USB1_SOF_INTERVAL_COUNT;
    g_deviceAudioComposite->audioUnified.audioPllTickEmaFrac = 0;
    g_deviceAudioComposite->audioUnified.audioPllStep        = AUDIO_PLL_FRACTIONAL_CHANGE_STEP;
#endif
}

/*!
 * @brief Audio set configuration function.
 *
 * This function sets configuration for msc class.
 *
 * @param handle The Audio class handle.
 * @param configure The Audio class configure index.
 *
 * @return A USB error code or kStatus_USB_Success.
 */
usb_status_t USB_DeviceAudioCompositeSetConfigure(class_handle_t handle, uint8_t configure)
{
	if (USB_COMPOSITE_CONFIGURE_INDEX == configure)
	{
		g_deviceAudioComposite->audioUnified.attach = 1U;
	}
	return kStatus_USB_Success;
}

usb_status_t USB_DeviceAudioRecorderSetInterface(class_handle_t handle, uint8_t interface, uint8_t alternateSetting)
{
    usb_status_t error = kStatus_USB_Success;

    if (alternateSetting == 1U)
	{
        MicUpStreamDataRateAdjustCnt = 0;

        CirAudioBuf_ClearAllSamples_MultiCh(&UsbUpStrmCirBuf);

        error = USB_DeviceAudioSend(g_deviceAudioComposite->audioUnified.audioRecorderHandle,
				USB_AUDIO_RECORDER_STREAM_ENDPOINT,
				//s_wavBuff,
				(uint8_t *)AllZeroBuf_48PointsSingleCh_32Bit,
                FS_ISO_IN_ENDP_PACKET_SIZE
				);
    }else
    {
        UsbAudioUpStreamingIsStarted = 0;
        CirAudioBuf_ClearAllSamples_MultiCh(&UsbUpStrmCirBuf);
        MicUpStreamDataRateAdjustCnt = 0;
    }
    return error;
}

usb_status_t USB_DeviceAudioSpeakerSetInterface(class_handle_t handle, uint8_t interface, uint8_t alternateSetting)
{
	usb_status_t error = kStatus_USB_Success;

	if (alternateSetting == 1U)
	{
		USB_DeviceAudioSpeakerStatusReset();
		if (USB_SPEED_HIGH == g_composite.speed)
		{
			error = USB_DeviceAudioRecv(g_deviceAudioComposite->audioUnified.audioSpeakerHandle,
										USB_AUDIO_SPEAKER_STREAM_ENDPOINT,
										//&audioPlayDataBuff[0],
										(unsigned char *)UsbDnStrmCirBuf.PtrWr,
										HS_ISO_OUT_ENDP_PACKET_SIZE
										);
		}
		if (USB_SPEED_FULL == g_composite.speed)
		{
			error = USB_DeviceAudioRecv(g_deviceAudioComposite->audioUnified.audioSpeakerHandle,
										USB_AUDIO_SPEAKER_STREAM_ENDPOINT,
										//&audioPlayDataBuff[0],
										(unsigned char *)UsbDnStrmCirBuf.PtrWr,
										FS_ISO_OUT_ENDP_PACKET_SIZE
										);
		}
		if (error != kStatus_USB_Success)
		{
			return error;
		} else
		{
#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
#else
			if (!feedbackValueUpdating)
			{
				*((uint32_t *)&usbAudioFeedBackBuffer[0]) = *((uint32_t *)&audioFeedBackBuffer[0]);
			}
			error = USB_DeviceAudioSend(g_deviceAudioComposite->audioUnified.audioSpeakerHandle,
			USB_AUDIO_SPEAKER_FEEDBACK_ENDPOINT, usbAudioFeedBackBuffer, (USB_SPEED_HIGH == g_composite.speed) ? HS_ISO_FEEDBACK_ENDP_PACKET_SIZE :
			FS_ISO_FEEDBACK_ENDP_PACKET_SIZE);
#endif
			UsbAudioDnStreamingIsStarted=1;
			CirAudioBuf_ClearAllSamples_S64(&UsbDnStrmCirBuf);
			g_deviceAudioComposite->audioUnified.audioSendCount = 0;
			g_deviceAudioComposite->audioUnified.usbRecvCount = 0;
			g_deviceAudioComposite->audioUnified.lastAudioSendCount = 0;
		}
	}else
	{
		UsbAudioDnStreamingIsStarted = 0;
		CirAudioBuf_ClearAllSamples_S64(&UsbDnStrmCirBuf);
		g_deviceAudioComposite->audioUnified.audioSendCount = 0;
		g_deviceAudioComposite->audioUnified.usbRecvCount = 0;
		g_deviceAudioComposite->audioUnified.lastAudioSendCount = 0;
	}
	return error;
}

/*!
 * @brief Audio init function.
 *
 * This function initializes the device with the composite device class information.
 *
 * @param device_composite          The pointer to the composite device structure.
 * @return kStatus_USB_Success .
 */
usb_status_t USB_DeviceAudioCompositeInit(usb_device_composite_struct_t *device_composite)
{
#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
    //SCTIMER_CaptureInit();
#else
	AUDIO_UPDATE_FEEDBACK_DATA(audioFeedBackBuffer, AUDIO_SAMPLING_RATE_TO_10_14);
#endif
	g_deviceAudioComposite = device_composite;
	g_deviceAudioComposite->audioUnified.copyProtect = 0x01U;
	g_deviceAudioComposite->audioUnified.curMute = 0x00U;
	g_deviceAudioComposite->audioUnified.curVolume[0] = 0x00U;
	g_deviceAudioComposite->audioUnified.curVolume[1] = 0x1fU;
	g_deviceAudioComposite->audioUnified.minVolume[0] = 0x00U;
	g_deviceAudioComposite->audioUnified.minVolume[1] = 0x00U;
	g_deviceAudioComposite->audioUnified.maxVolume[0] = 0x00U;
	g_deviceAudioComposite->audioUnified.maxVolume[1] = 0X43U;
	g_deviceAudioComposite->audioUnified.resVolume[0] = 0x01U;
	g_deviceAudioComposite->audioUnified.resVolume[1] = 0x00U;
	g_deviceAudioComposite->audioUnified.curBass = 0x00U;
	g_deviceAudioComposite->audioUnified.curBass = 0x00U;
	g_deviceAudioComposite->audioUnified.minBass = 0x80U;
	g_deviceAudioComposite->audioUnified.maxBass = 0x7FU;
	g_deviceAudioComposite->audioUnified.resBass = 0x01U;
	g_deviceAudioComposite->audioUnified.curMid = 0x00U;
	g_deviceAudioComposite->audioUnified.minMid = 0x80U;
	g_deviceAudioComposite->audioUnified.maxMid = 0x7FU;
	g_deviceAudioComposite->audioUnified.resMid = 0x01U;
	g_deviceAudioComposite->audioUnified.curTreble = 0x01U;
	g_deviceAudioComposite->audioUnified.minTreble = 0x80U;
	g_deviceAudioComposite->audioUnified.maxTreble = 0x7FU;
	g_deviceAudioComposite->audioUnified.resTreble = 0x01U;
	g_deviceAudioComposite->audioUnified.curAutomaticGain = 0x01U;
	g_deviceAudioComposite->audioUnified.curDelay[0] = 0x00U;
	g_deviceAudioComposite->audioUnified.curDelay[1] = 0x40U;
	g_deviceAudioComposite->audioUnified.minDelay[0] = 0x00U;
	g_deviceAudioComposite->audioUnified.minDelay[1] = 0x00U;
	g_deviceAudioComposite->audioUnified.maxDelay[0] = 0xFFU;
	g_deviceAudioComposite->audioUnified.maxDelay[1] = 0xFFU;
	g_deviceAudioComposite->audioUnified.resDelay[0] = 0x00U;
	g_deviceAudioComposite->audioUnified.resDelay[1] = 0x01U;
	g_deviceAudioComposite->audioUnified.curLoudness = 0x01U;
	g_deviceAudioComposite->audioUnified.curSamplingFrequency[0] = 0x00U;
	g_deviceAudioComposite->audioUnified.curSamplingFrequency[1] = 0x00U;
	g_deviceAudioComposite->audioUnified.curSamplingFrequency[2] = 0x01U;
	g_deviceAudioComposite->audioUnified.minSamplingFrequency[0] = 0x00U;
	g_deviceAudioComposite->audioUnified.minSamplingFrequency[1] = 0x00U;
	g_deviceAudioComposite->audioUnified.minSamplingFrequency[2] = 0x01U;
	g_deviceAudioComposite->audioUnified.maxSamplingFrequency[0] = 0x00U;
	g_deviceAudioComposite->audioUnified.maxSamplingFrequency[1] = 0x00U;
	g_deviceAudioComposite->audioUnified.maxSamplingFrequency[2] = 0x01U;
	g_deviceAudioComposite->audioUnified.resSamplingFrequency[0] = 0x00U;
	g_deviceAudioComposite->audioUnified.resSamplingFrequency[1] = 0x00U;
	g_deviceAudioComposite->audioUnified.resSamplingFrequency[2] = 0x01U;
#if USB_DEVICE_CONFIG_AUDIO_CLASS_2_0
    g_deviceAudioComposite->audioUnified.curMute20          = 0U;
    g_deviceAudioComposite->audioUnified.curClockValid      = 1U;
    g_deviceAudioComposite->audioUnified.curVolume20[0]     = 0x00U;
    g_deviceAudioComposite->audioUnified.curVolume20[1]     = 0x1FU;
    g_deviceAudioComposite->audioUnified.curSampleFrequency = (AUDIO_IN_SAMPLING_RATE_KHZ*1000);

    g_deviceAudioComposite->audioUnified.freqControlRange.wNumSubRanges = 1U;
    g_deviceAudioComposite->audioUnified.freqControlRange.wMIN          = (AUDIO_IN_SAMPLING_RATE_KHZ*1000);	//now, Dn stream and up stream must be the same FS
    g_deviceAudioComposite->audioUnified.freqControlRange.wMAX          = (AUDIO_IN_SAMPLING_RATE_KHZ*1000);	//now, Dn stream and up stream must be the same FS
    g_deviceAudioComposite->audioUnified.freqControlRange.wRES          = 0U;

    g_deviceAudioComposite->audioUnified.volumeControlRange.wNumSubRanges = 1U;
    g_deviceAudioComposite->audioUnified.volumeControlRange.wMIN          = 0x8001U;
    g_deviceAudioComposite->audioUnified.volumeControlRange.wMAX          = 0x7FFFU;
    g_deviceAudioComposite->audioUnified.volumeControlRange.wRES          = 1U;

#endif
	//g_deviceAudioComposite->audioUnified.UsbDnStrmBuf_WrPosInByte = 0;
	//g_deviceAudioComposite->audioUnified.UsbDnStrmBuf_RdPosInByte = 0;
	g_deviceAudioComposite->audioUnified.audioSendCount = 0;
	g_deviceAudioComposite->audioUnified.lastAudioSendCount = 0;
	g_deviceAudioComposite->audioUnified.usbRecvCount = 0;
	//g_deviceAudioComposite->audioUnified.audioSendTimes = 0;
	//g_deviceAudioComposite->audioUnified.usbRecvTimes = 0;
	g_deviceAudioComposite->audioUnified.startPlay = 0;
	g_deviceAudioComposite->audioUnified.startPlayHalfFull = 0;
	g_deviceAudioComposite->audioUnified.speakerIntervalCount = 0;
	g_deviceAudioComposite->audioUnified.UsbDnStrmBuf_OccupiedSpaceInByte = 0;
	g_deviceAudioComposite->audioUnified.timesFeedbackCalculate = 0;
	g_deviceAudioComposite->audioUnified.speakerDetachOrNoInput = 0;
#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
    g_deviceAudioComposite->audioUnified.audioPllTicksPrev   = 0;
    g_deviceAudioComposite->audioUnified.audioPllTicksDiff   = 0;
    g_deviceAudioComposite->audioUnified.audioPllTicksEma    = AUDIO_PLL_USB1_SOF_INTERVAL_COUNT;
    g_deviceAudioComposite->audioUnified.audioPllTickEmaFrac = 0;
    g_deviceAudioComposite->audioUnified.audioPllStep        = AUDIO_PLL_FRACTIONAL_CHANGE_STEP;
#endif

    return kStatus_USB_Success;
}
extern void BOARD_SetCodecMuteUnmute(bool mute);
void USB_AudioCodecTask(void)
{
    if (g_deviceAudioComposite->audioUnified.codecTask & MUTE_CODEC_TASK)
    {
        g_deviceAudioComposite->audioUnified.codecTask &= ~MUTE_CODEC_TASK;
        g_CodecMuteUnmute = true;
    }
    if (g_deviceAudioComposite->audioUnified.codecTask & UNMUTE_CODEC_TASK)
    {
        g_deviceAudioComposite->audioUnified.codecTask &= ~UNMUTE_CODEC_TASK;
        g_CodecMuteUnmute = true;
    }
    if (g_deviceAudioComposite->audioUnified.codecTask & VOLUME_CHANGE_TASK)
    {
        g_deviceAudioComposite->audioUnified.codecTask &= ~VOLUME_CHANGE_TASK;
    }
}

void USB_AudioSpeakerResetTask(void)
{
	if (g_deviceAudioComposite->audioUnified.speakerDetachOrNoInput)
	{
		USB_DeviceAudioSpeakerStatusReset();
	}
}
#endif
