/*
 * Copyright 2025 NXP
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "dimage.h"

void dump_hdr(const ihdr_t *hdr)
{
    DIMAGE_TRACE("header_marker :0x%08X\r\n", hdr->header_marker);
    DIMAGE_TRACE("image_type    :0x%08X\r\n", hdr->img_type);
    DIMAGE_TRACE("reserved      :0x%08X\r\n", hdr->reserved);
    DIMAGE_TRACE("img_len       :0x%08X\r\n", hdr->img_len);
    DIMAGE_TRACE("crc_value     :0x%08X\r\n", hdr->crc_value);
    DIMAGE_TRACE("version       :0x%08X\r\n", hdr->version);
}

/* get a image header from actual addr and load addr */
int image_get_hdr(uint32_t addr, uint32_t load_addr, ihdr_t *hdr)
{
    uint32_t hdr_addr;
    
    memory_read(addr + DUAL_IMAGE_HDR_ADDR, (uint8_t*)&hdr_addr, sizeof(hdr_addr));
    
    hdr_addr = hdr_addr + addr;
    //DIMAGE_TRACE("dhr_addr:0x%X\r\n", hdr_addr);
    memory_read(hdr_addr, (uint8_t*)hdr, sizeof(ihdr_t));

    return hdr_addr;
}


/* do image crc checking */
static int _crc_check(uint32_t addr, uint32_t load_addr)
{
    int i;
    uint8_t buf[256];
    int ret;
    int crc_len, crc_start;
    uint32_t cal_crc, crc_offset;
    ihdr_t hdr;
    
    ret = 1;
    
    /* get image header */
    crc_offset = image_get_hdr(addr, load_addr, &hdr);
    
    if(hdr.header_marker == HEADER_BLOCK_MARKER)
    {
        switch(hdr.img_type)
        {
            case 0: /* need crc check */
                crc_offset = crc_offset - addr + sizeof(ihdr_t) - 2*sizeof(uint32_t);
                //DIMAGE_TRACE("crc_offset:0x%X\r\n", crc_offset);
                
                if(crc_offset > hdr.img_len)
                {
                    break;
                }
                
                crc32_init(&cal_crc);
                
                /* calcuate data before crc */
                crc_len = crc_offset;
                crc_start = addr;
                
                for(i=0; i<crc_len / sizeof(buf); i++)
                {
                    memory_read(crc_start + i*sizeof(buf), buf, sizeof(buf));
                    crc32_generate(&cal_crc, buf, sizeof(buf));
                }
                
                memory_read(crc_start + i*sizeof(buf), buf, crc_len % sizeof(buf));
                crc32_generate(&cal_crc, buf, crc_len % sizeof(buf));
                    
                /* calcuate data after crc */
                crc_len = hdr.img_len - crc_offset - sizeof(uint32_t);
                crc_start = crc_offset + sizeof(uint32_t) + addr;
                  
                for(i=0; i<crc_len / sizeof(buf); i++)
                {
                    memory_read(crc_start + i*sizeof(buf), buf, sizeof(buf));
                    crc32_generate(&cal_crc, buf, sizeof(buf));
                }
                
                memory_read(crc_start + i*sizeof(buf), buf, crc_len % sizeof(buf));
                crc32_generate(&cal_crc, buf, crc_len % sizeof(buf));
                
                crc32_complete(&cal_crc);
                
                if(cal_crc == hdr.crc_value)
                {
                    ret = 0;
                }
                break;
            case 1: /* no crc check */
                ret = 0;
                break;
            default:
                break;
        }
    }
    return ret;
}

/* scan flash to find vailid image */
int image_scan(uint32_t start_addr, uint32_t load_addr, uint32_t len, uint32_t *image_addr, uint32_t max_image_cnt)
{
    uint32_t marker;
    uint32_t image_cnt;
    ihdr_t hdr;
    
    /* scan dual image marker and header */
    image_cnt = 0;
    
    for(uint32_t i = 0; i < max_image_cnt; i++)
    {
        memory_read(start_addr + DUAL_IMAGE_MARKER_OFFSET, (uint8_t*)&marker, sizeof(marker));

        if(marker == DUAL_IMAGE_MAKRER)
        {
            image_get_hdr(start_addr, load_addr, &hdr);
            if(hdr.header_marker == HEADER_BLOCK_MARKER)
            {
                if(_crc_check(start_addr, load_addr) == 0)
                {
                    /* image found */
                    image_addr[image_cnt] = start_addr;
                    image_cnt++;
                }
                else
                {
                    DIMAGE_TRACE("crc check failed\r\n");
                }
            }
        }
        start_addr += len;
    }
    return image_cnt;
}

static int image_load(uint32_t to, uint32_t from, uint32_t len)
{
    int ret;
    uint8_t buf[256];

    ret = memory_erase(to, ALIGN_UP(len, 8));
    if(ret)
    {
        return ret;
    }

    while(len > 0)
    {
        uint32_t min_len;
        min_len = MIN(len, 256);
        memory_read(from, buf, min_len);
        ret = memory_write(to, buf, min_len);
        if(ret)
        {
            return ret;
        }

        len -= min_len;
        from += min_len;
        to += min_len;
    }

    return ret;
}

int image_verify(uint32_t to, uint32_t from, uint32_t len)
{
    int ret = 0;
    uint8_t buf[256];

    while(len > 0)
    {
        uint32_t min_len;
        min_len = MIN(len, 256);
        memory_read(from, buf, min_len);
        ret = memcmp(buf, (uint8_t *)to, min_len);
        if(ret)
        {
            return ret;
        }

        len -= min_len;
        from += min_len;
        to += min_len;
    }

    return ret;
}

/* do dual image policy and boot application if everything ok */
int image_check_and_boot(void)
{
    /*
        gaddr: golden image addr
        baddr: backup image addr 
        gimage_cnt: how many golden image in golden area
        bimage_cnt: how many backup image in backup area
    */
  
    int gimage_cnt, bimage_cnt, len;
    uint32_t gaddr, baddr, baddr_array[BACKUP_IMAGE_CNT];
    ihdr_t ghdr, bhdr, bhdr_temp;
    
    /* scan a image in golden region */
    DIMAGE_TRACE("scan golden region...\r\n");
    gimage_cnt = image_scan(GOLDEN_REGION_START, GOLDEN_REGION_START, 0, &gaddr, GOLDEN_IMAGE_CNT);
    if(gimage_cnt)
    {
        DIMAGE_TRACE("image found: 0x%08X\r\n", gaddr);
        image_get_hdr(gaddr, GOLDEN_REGION_START, &ghdr);
        dump_hdr(&ghdr);
    }
    
    /* scan a image in backup region */
    DIMAGE_TRACE("scan backup region...\r\n");
    memset(baddr_array, 0, sizeof(baddr_array));
    bimage_cnt = image_scan(BACKUP_REGION_START, GOLDEN_REGION_START, BACKUP_REGION_LEN, baddr_array, BACKUP_IMAGE_CNT);

    for(uint32_t i = 0; i < bimage_cnt; i++)
    {
        DIMAGE_TRACE("image found: 0x%08X\r\n", baddr_array[i]);
        image_get_hdr(baddr_array[i], GOLDEN_REGION_START, &bhdr_temp);
        dump_hdr(&bhdr_temp);
        if(i == 0)
        {
            bhdr = bhdr_temp;
            baddr = baddr_array[i];
        }
        else
        {
            if(bhdr_temp.version > bhdr.version)
            {
                bhdr = bhdr_temp;
                baddr = baddr_array[i];
            }
        }
    }

    if((!gimage_cnt) && (bimage_cnt))
    {
        DIMAGE_TRACE("golden image bad, backup ok\r\n");
        
        /* copy to golden image and boot */
        len = bhdr.img_len;
        image_load(GOLDEN_REGION_START, baddr, len);
        if(image_verify(GOLDEN_REGION_START, baddr, len) == 0)
        {
            return 0;
        }
    }
    
    if(gimage_cnt && (!bimage_cnt))
    {
        DIMAGE_TRACE("golden image ok, no backup image, boot\r\n");
        
        return 0;
    }
    
    if(gimage_cnt && bimage_cnt)
    {
        if(ghdr.version < bhdr.version)
        {
            DIMAGE_TRACE("backup image version higher, copy to golden area\r\n");
            
            /* copy backup into golden */
            len = bhdr.img_len;
            image_load(GOLDEN_REGION_START, baddr, len);
            if(image_verify(GOLDEN_REGION_START, baddr, len) == 0)
            {
                return 0;
            }
        }
        else
        {
            DIMAGE_TRACE("golden image has same or higher version than backup, boot golden image\r\n");
            return 0;
        }
    }
    DIMAGE_TRACE("No valid firmware.\r\n");
    return 1;
}

int image_program(const uint8_t *image, uint32_t len)
{
    int bimage_cnt;
    uint32_t baddr, baddr_array[BACKUP_IMAGE_CNT];
    ihdr_t bhdr, bhdr_temp;

    memset(baddr_array, 0, sizeof(baddr_array));
    bimage_cnt = image_scan(BACKUP_REGION_START, GOLDEN_REGION_START, BACKUP_REGION_LEN, baddr_array, BACKUP_IMAGE_CNT);
    if(bimage_cnt == BACKUP_IMAGE_CNT)
    {
        /* find the oldest image */
        for(uint32_t i = 0; i < BACKUP_IMAGE_CNT; i++)
        {
            DIMAGE_TRACE("image found: 0x%08X\r\n", baddr_array[i]);
            image_get_hdr(baddr_array[i], GOLDEN_REGION_START, &bhdr_temp);
            dump_hdr(&bhdr_temp);
            if(i == 0)
            {
                bhdr = bhdr_temp;
                baddr = baddr_array[i];
            }
            else
            {
                if(bhdr_temp.version < bhdr.version)
                {
                    bhdr = bhdr_temp;
                    baddr = baddr_array[i];
                }
            }
        }
    }
    else 
    {
        /* find the first invalid bank */
        for(uint32_t i = 0; i < BACKUP_IMAGE_CNT; i++)
        {
            DIMAGE_TRACE("image found: 0x%08X\r\n", baddr_array[i]);
            baddr = BACKUP_REGION_START + i * BACKUP_REGION_LEN;
            if(baddr != baddr_array[i])
            {
                break;
            }
        }
    }
    
    memory_erase(baddr, BACKUP_REGION_LEN);
    memory_write(baddr, image, len);
    return baddr;
}

int image_erase(uint32_t bank)
{
    memory_erase(BACKUP_REGION_START + bank * BACKUP_REGION_LEN, BACKUP_REGION_LEN);
}
