/********************************************************************* 
 * 
 * Copyright: 
 *      MOTOROLA, INC. All Rights Reserved.  
 *  You are hereby granted a copyright license to use, modify, and 
 *  distribute the SOFTWARE so long as this entire notice is 
 *  retained without alteration in any modified and/or redistributed 
 *  versions, and that such modified versions are clearly identified 
 *  as such. No licenses are granted by implication, estoppel or 
 *  otherwise under any patents or trademarks of Motorola, Inc. This 
 *  software is provided on an "AS IS" basis and without warranty. 
 * 
 *  To the maximum extent permitted by applicable law, MOTOROLA 
 *  DISCLAIMS ALL WARRANTIES WHETHER EXPRESS OR IMPLIED, INCLUDING 
 *  IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR 
 *  PURPOSE AND ANY WARRANTY AGAINST INFRINGEMENT WITH REGARD TO THE 
 *  SOFTWARE (INCLUDING ANY MODIFIED VERSIONS THEREOF) AND ANY 
 *  ACCOMPANYING WRITTEN MATERIALS. 
 * 
 *  To the maximum extent permitted by applicable law, IN NO EVENT 
 *  SHALL MOTOROLA BE LIABLE FOR ANY DAMAGES WHATSOEVER (INCLUDING 
 *  WITHOUT LIMITATION, DAMAGES FOR LOSS OF BUSINESS PROFITS, BUSINESS 
 *  INTERRUPTION, LOSS OF BUSINESS INFORMATION, OR OTHER PECUNIARY 
 *  LOSS) ARISING OF THE USE OR INABILITY TO USE THE SOFTWARE.   
 * 
 *  Motorola assumes no responsibility for the maintenance and support 
 *  of this software 
 ********************************************************************/
/***************************************************************************/
/* FILE NAME: tpu_nitc.c                       COPYRIGHT (c) MOTOROLA 2002 */
/* VERSION: 1.0                                   All Rights Reserved      */
/*                                                                         */
/* DESCRIPTION: This file contains the TPU NITC functions. These functions */
/* allow you to completely control TPU channels running the NITC function. */
/* They provide a simple interface requiring the minimum amount of         */
/* configuration by the user. The functions are:                           */
/* tpu_nitc_init_tcr_mode                                                  */         
/* tpu_nitc_init_parameter_mode							   */
/* tpu_nitc_write_max_count                                                */	
/* tpu_nitc_write_trans_count								   */
/* tpu_nitc_read_max_count								   */
/* tpu_nitc_read_trans_count								   */
/* tpu_nitc_read_final_trans_time					               */
/* tpu_nitc_read_last_trans_time					               */
/*=========================================================================*/
/* HISTORY           ORIGINAL AUTHOR: Vernon Goler                         */
/* REV      AUTHOR      DATE       DESCRIPTION OF CHANGE                   */
/* ---   -----------  ---------    ---------------------                   */
/* 1.0   V. Goler  24/Aug/02    Initial version of function.               */
/***************************************************************************/
#include "tpu_nitc.h"
#include "mpc500_util.h"

/*******************************************************************************
FUNCTION      : 			tpu_nitc_init_tcr_mode
PURPOSE       : 			To initialize a channel to run the nitc function 					in tcr capture mode

INPUTS NOTES  : This function has 11 parameters:

       		*tpu - 	This is a pointer to the TPU3 module to use. It is 					of type TPU3_tag which is defined in m_tpu3.h

                 	channel - 	This is the channel number of the channel that 						will perform the nitc function.

                 	priority - 	This is the priority assigned to the channel.
                            	This parameter should be assigned a value of:
                            	TPU_PRIORITY_HIGH, TPU_PRIORITY_MIDDLE or
                            	TPU_PRIORITY_LOW.

         		detect_edge - This is the transition edge that will be                           					detected. The edge detected is either rising, 						falling, or both.This parameter should be assigned 					a value of:
					TPU_NITC_RISING
					TPU_NITC_FALLING
					TPU_NITC_RISING_FALLING

			max_count - This defines the maximum number of transitions to 					be detected.  When max_count transifions have been 					detected either an interrupt may be generated, and 					or a message sent to other channels via a link 						mechanism.

			single_continuous_operation - This defines whether the channel      					counts input transitions until a programmable 						maximum number of     transitions is reached once 					and then ceases operation until reinitialized, or 					performs this operation continuously.  This  						parameter should be assigned a value of:
					TPU_NITC_SINGLE
					TPU_NITC_CONTINUOUS

			tcr - 	Determines which TCR to capture when programmable 					maximum number of transistions have been detected. 					Either TCR1 or TCR2can be configured to be 						captured.  This parameter should be assigned a 						value of:
					TPU_NITC_TCR1
					TPU_NITC_TCR2

			nolink_link - This determines whether a message is sent to   					other channels by link mechanism, when the 						programmable maximum number of transistions have 					been detected.  This parameter should be assigned 					a value of:
					TPU_NITC_NOLINK
					TPU_NITC_LINK

			start_link_channel - This is the channel where linking begins
					This parameter should be assigned a value of:
					TPU_NITC_START_LINK_CHANNEL_0
					TPU_NITC_START_LINK_CHANNEL_1
					TPU_NITC_START_LINK_CHANNEL_2
					TPU_NITC_START_LINK_CHANNEL_3
					TPU_NITC_START_LINK_CHANNEL_4
					TPU_NITC_START_LINK_CHANNEL_5
					TPU_NITC_START_LINK_CHANNEL_6
					TPU_NITC_START_LINK_CHANNEL_7
					TPU_NITC_START_LINK_CHANNEL_8
					TPU_NITC_START_LINK_CHANNEL_9
					TPU_NITC_START_LINK_CHANNEL_10
					TPU_NITC_START_LINK_CHANNEL_11
					TPU_NITC_START_LINK_CHANNEL_12
					TPU_NITC_START_LINK_CHANNEL_13
					TPU_NITC_START_LINK_CHANNEL_14
					TPU_NITC_START_LINK_CHANNEL_15

			link_channel_count - This is the number of sequential channels 					to link to from the start link channel.  The 						maximum number of channels that can be linked is 					eight channels.  This parameter should be assigned
					a value of:
					TPU_NITC_LINK_ONE
					TPU_NITC_LINK_TWO
					TPU_NITC_LINK_THREE
					TPU_NITC_LINK_FOUR
					TPU_NITC_LINK_FIVE
					TPU_NITC_LINK_SIX
					TPU_NITC_LINK_SEVEN
					TPU_NITC_LINK_EIGHT






			nointerrupt_interrupt - This determines whether an interrupt 					is enabled when the programmable maximum number of 					transitions are detected. This parameter should be
					assigned a value of:
					TPU_NITC_NOINTERRUPT
					TPU_NITC_INTERRUPT

RETURNS NOTES : 	none
WARNING       : 	The channel must be stopped before it is reconfigured. The
                	function disables the channel, but if the channel is currently
                	being serviced it will continue to be serviced. The delay for    			assigning the parameter pointer may to enough to allow channel 			service to complete, but depends on system loading.
*******************************************************************************/
void tpu_nitc_init_tcr_mode(struct TPU3_tag *tpu, UINT8 channel, \
UINT8 priority, UINT8 detect_edge, INT16 max_count, \
UINT8 single_continuous_operation, \
UINT8 tcr, UINT8 nolink_link, UINT8 start_link_channel, \
UINT8 link_channel_count, UINT8 nointerrupt_interrupt)
{

   	/* disable channel so channel can be configured safely 			*/
    	tpu_disable( tpu, channel);
    
   	/* select NITC function for channel 						*/
    	tpu_func( tpu, channel, TPU_FUNCTION_NITC);
    
    	/* Initialize parameter RAM                                      		*/
	/* setup channel as input, tcr capture, and edge detect			*/
    	tpu->PARM.R[channel][TPU_NITC_CHANNEL_CONTROL] = ((tcr << 6) | 	(detect_edge << 2) | (0x3));

    	tpu->PARM.R[channel][TPU_NITC_CHANNEL_ATTRIBUTES] = \
	((start_link_channel << 12) | (link_channel_count << 8));

   	tpu->PARM.R[channel][TPU_NITC_MAX_COUNT] = max_count;

    	/* Configure channel for single or continuous operation                 */
    	/* Configure channel for no linking or linking                          */
    	tpu_hsq(tpu, channel, ((nolink_link << 1) | single_continuous_operation));

	/* If nointerrupt_interupt flag is set enable interrupts			*/
	if(nointerrupt_interrupt) {
		tpu_interrupt_enable(tpu, channel);
	}
        
    	/* Initialize channel */
    	tpu_hsr(tpu, channel, TPU_NITC_INIT_TCR);


    	/* Enable channel by assigning a priority. 					*/
    	tpu_enable(tpu, channel, priority);
}


/*******************************************************************************
FUNCTION      : 			tpu_nitc_init_parameter_mode
PURPOSE       : 			To initialize a channel to run the nitc function 					in parameter capture mode

INPUTS NOTES  : This function has 11 parameters:

       		*tpu - 	This is a pointer to the TPU3 module to use. It is 					of type TPU3_tag which is defined in m_tpu3.h

                 	channel - 	This is the channel number of the channel that 						will perform the nitc function.

                 	priority - 	This is the priority assigned to the channel.
                            	This parameter should be assigned a value of:
                            	TPU_PRIORITY_HIGH, TPU_PRIORITY_MIDDLE or
                            	TPU_PRIORITY_LOW.

         		detect_edge - This is the transition edge that will be                           					detected. The edge detected is either rising, 						falling, or both.This parameter should be assigned 					a value of:
					TPU_NITC_RISING
					TPU_NITC_FALLING
					TPU_NITC_RISING_FALLING

			max_count - This defines the maximum number of transitions to 					be detected.  When max_count transifions have been 					detected either an interrupt may be generated, and 					or a message sent to other channels via a link 						mechanism.

			single_continuous_operation - This defines whether the channel      					counts input transitions until a programmable 						maximum number of     transitions is reached once 					and then ceases operation until reinitialized, or 					performs this operation continuously.  This  						parameter should be assigned a value of:
					TPU_NITC_SINGLE
					TPU_NITC_CONTINUOUS

			parameter_address - This defines the address of the TPU 						parameter.  The value pointed to by this 							parameter is "captured" into parameter locations:
					TPU_NITC_FINAL_TRANS_TIME
					TPU_NITC_LAST_TRANS_TIME
					Depending on max_count transistions being detected
					or less than max_count transistions being 						detected.  This address needs to be aligned to a 					16 bit boundary.

			nolink_link - This determines whether a message is sent to   					other channels by link mechanism, when the 						programmable maximum number of transistions have 					been detected.  This parameter should be assigned 					a value of:
					TPU_NITC_NOLINK
					TPU_NITC_LINK

			start_link_channel - This is the channel where linking begins
					This parameter should be assigned a value of:
					TPU_NITC_START_LINK_CHANNEL_0
					TPU_NITC_START_LINK_CHANNEL_1
					TPU_NITC_START_LINK_CHANNEL_2
					TPU_NITC_START_LINK_CHANNEL_3
					TPU_NITC_START_LINK_CHANNEL_4
					TPU_NITC_START_LINK_CHANNEL_5
					TPU_NITC_START_LINK_CHANNEL_6
					TPU_NITC_START_LINK_CHANNEL_7
					TPU_NITC_START_LINK_CHANNEL_8
					TPU_NITC_START_LINK_CHANNEL_9
					TPU_NITC_START_LINK_CHANNEL_10
					TPU_NITC_START_LINK_CHANNEL_11
					TPU_NITC_START_LINK_CHANNEL_12
					TPU_NITC_START_LINK_CHANNEL_13
					TPU_NITC_START_LINK_CHANNEL_14
					TPU_NITC_START_LINK_CHANNEL_15

			link_channel_count - This is the number of sequential channels 					to link to from the start link channel.  The 						maximum number of channels that can be linked is 					eight channels.  This parameter should be assigned
					a value of:
					TPU_NITC_LINK_ONE
					TPU_NITC_LINK_TWO
					TPU_NITC_LINK_THREE
					TPU_NITC_LINK_FOUR
					TPU_NITC_LINK_FIVE
					TPU_NITC_LINK_SIX
					TPU_NITC_LINK_SEVEN
					TPU_NITC_LINK_EIGHT


			nointerrupt_interrupt - This determines whether an interrupt 					is enabled when the programmable maximum number of 					transitions are detected. This parameter should be
					assigned a value of:
					TPU_NITC_NOINTERRUPT
					TPU_NITC_INTERRUPT

RETURNS NOTES : 	none
WARNING       : 	The channel must be stopped before it is reconfigured. The
                	function disables the channel, but if the channel is currently
                	being serviced it will continue to be serviced. The delay for    			assigning the parameter pointer may to enough to allow channel 			service to complete, but depends on system loading.
*******************************************************************************/

void tpu_nitc_init_parameter_mode(struct TPU3_tag *tpu, UINT8 channel, \
UINT8 priority, UINT8 detect_edge, INT16 max_count, \
UINT8 single_continuous_operation, \
UINT8 parameter_address, UINT8 nolink_link, UINT8 start_link_channel, \
UINT8 link_channel_count, UINT8 nointerrupt_interrupt)
{

   	/* disable channel so channel can be configured safely 			*/
    	tpu_disable( tpu, channel);
    
   	/* select NITC function for channel 						*/
    	tpu_func( tpu, channel, TPU_FUNCTION_NITC);
    
    	/* Initialize parameter RAM                                      		*/
	/* setup channel as input, tcr capture, and edge detect			*/
    	tpu->PARM.R[channel][TPU_NITC_CHANNEL_CONTROL] = \
	((detect_edge << 2) | (0x3));

    	tpu->PARM.R[channel][TPU_NITC_CHANNEL_ATTRIBUTES] = \
	((start_link_channel << 12) | (link_channel_count << 8) \
	| (parameter_address));

   	tpu->PARM.R[channel][TPU_NITC_MAX_COUNT] = max_count;
	
	/* Configure channel for single or continuous operation                 */
    	/* Configure channel for no linking or linking                          */
    	tpu_hsq(tpu, channel, ((nolink_link << 1) | single_continuous_operation));

	/* If nointerrupt_interupt flag is set enable interrupts			*/
	if(nointerrupt_interrupt) {
		tpu_interrupt_enable(tpu, channel);
	}
        
    	/* Initialize channel */
    	tpu_hsr(tpu, channel, TPU_NITC_INIT_PARAMETER);


    	/* Enable channel by assigning a priority. 					*/
    	tpu_enable(tpu, channel, priority);
}




/*******************************************************************************
FUNCTION      : 			tpu_nitc_write_max_count 
PURPOSE       : 			To allow update of parameter max_count

INPUTS NOTES  : This function has 3 parameters:

       		*tpu - 	This is a pointer to the TPU3 module to use. It is 					of type TPU3_tag which is defined in m_tpu3.h

                 	channel - 	This is the channel number of the channel that 						will perform the nitc function.

               

			max_count - This defines the maximum number of transitions to 					be detected.  When max_count transitions have been 					detected either an interrupt may be generated, and 					or a message sent to other channels via a link 						mechanism.

			

RETURNS NOTES : 	none
WARNING       : 	none


*******************************************************************************/
void tpu_nitc_write_max_count(struct TPU3_tag *tpu, UINT8 channel, \
INT16 max_count)

{

	tpu->PARM.R[channel][TPU_NITC_MAX_COUNT] = max_count; 
}



/*******************************************************************************
FUNCTION      : 			tpu_nitc_write_trans_count 
PURPOSE       : 			To allow update of parameter trans_count

INPUTS NOTES  : This function has 3 parameters:

       		*tpu - 	This is a pointer to the TPU3 module to use. It is 					of type TPU3_tag which is defined in m_tpu3.h

                 	channel - 	This is the channel number of the channel that 						will perform the nitc function.

			trans_count - This is the current count of transisitons that have been detected. 


			

RETURNS NOTES : 	none
WARNING       : 	The trans_count parameter should be written to a value less than the max_count parameter.


*******************************************************************************/
void tpu_nitc_write_trans_count(struct TPU3_tag *tpu, UINT8 channel, \
INT16 trans_count)

{
	tpu->PARM.R[channel][TPU_NITC_TRANS_COUNT] = trans_count; 
}




/*******************************************************************************
FUNCTION      : 			tpu_nitc_read_final_trans_time 
PURPOSE       : 			To allow reading of parameter:
					TPU_NITC_FINAL_TRANS_TIME 

INPUTS NOTES  : This function has 2 parameters:

       		*tpu - 	This is a pointer to the TPU3 module to use. It is 					of type TPU3_tag which is defined in m_tpu3.h

                 	channel - 	This is the channel number of the channel that 						will perform the nitc function.

			 

RETURNS NOTES : 	Returns final transition time or the value pointed to by the parameter address corresponding to the last transition.


WARNING       : 	none

*******************************************************************************/

INT16 tpu_nitc_read_final_trans_time(struct TPU3_tag *tpu, UINT8 channel)

    
{
	return (tpu->PARM.R[channel][TPU_NITC_FINAL_TRANS_TIME]);
}

/*******************************************************************************
FUNCTION      : 			tpu_nitc_read_last_trans_time 
PURPOSE       : 			To allow reading of parameter:
					TPU_NITC_LAST_TRANS_TIME 

INPUTS NOTES  : This function has 2 parameters:

       		*tpu - 	This is a pointer to the TPU3 module to use. It is 					of type TPU3_tag which is defined in m_tpu3.h

                 	channel - 	This is the channel number of the channel that 						will perform the nitc function.

			 

RETURNS NOTES : 	Returns last transition time or last value point to by the parameter address corresponding to the last transition.


WARNING       : 	none

*******************************************************************************/

INT16 tpu_nitc_read_last_trans_time(struct TPU3_tag *tpu, UINT8 channel)

    
{
	return (tpu->PARM.R[channel][TPU_NITC_LAST_TRANS_TIME]);
}



/*******************************************************************************
FUNCTION      : 			tpu_nitc_read_max_count 
PURPOSE       : 			To allow reading of parameter:
					TPU_NITC_MAX_COUNT 

INPUTS NOTES  : This function has 2 parameters:

       		*tpu - 	This is a pointer to the TPU3 module to use. It is 					of type TPU3_tag which is defined in m_tpu3.h

                 	channel - 	This is the channel number of the channel that 						will perform the nitc function.

RETURNS NOTES : 	Returns the max_count value 


WARNING       : 	none


*******************************************************************************/
INT16 tpu_nitc_read_max_count(struct TPU3_tag *tpu, UINT8 channel)

    
{
	return (tpu->PARM.R[channel][TPU_NITC_MAX_COUNT]);
}

/*******************************************************************************
FUNCTION      : 			tpu_nitc_read_trans_count 
PURPOSE       : 			To allow reading of parameter:
					TPU_NITC_TRANS_COUNT 

INPUTS NOTES  : This function has 2 parameters:

       		*tpu - 	This is a pointer to the TPU3 module to use. It is 					of type TPU3_tag which is defined in m_tpu3.h

                 	channel - 	This is the channel number of the channel that 						will perform the nitc function.

RETURNS NOTES : 	Returns the trans_count value 


WARNING       : 	none


*******************************************************************************/
INT16 tpu_nitc_read_trans_count(struct TPU3_tag *tpu, UINT8 channel)

    
{
	return (tpu->PARM.R[channel][TPU_NITC_TRANS_COUNT]);
}


/******************************************************************************/












/*****************************************************************************/
/* Motorola reserves the right to make changes without further notice to any */
/* product herein to improve reliability, function, or design. Motorola does */
/* not assume any  liability arising  out  of the  application or use of any */
/* product,  circuit, or software described herein;  neither  does it convey */
/* any license under its patent rights  nor the  rights of others.  Motorola */
/* products are not designed, intended,  or authorized for use as components */
/* in  systems  intended  for  surgical  implant  into  the  body, or  other */
/* applications intended to support life, or  for any  other application  in */
/* which the failure of the Motorola product  could create a situation where */
/* personal injury or death may occur. Should Buyer purchase or use Motorola */
/* products for any such intended  or unauthorized  application, Buyer shall */
/* indemnify and  hold  Motorola  and its officers, employees, subsidiaries, */
/* affiliates,  and distributors harmless against all claims costs, damages, */
/* and expenses, and reasonable  attorney  fees arising  out of, directly or */
/* indirectly,  any claim of personal injury  or death  associated with such */
/* unintended or unauthorized use, even if such claim alleges that  Motorola */
/* was negligent regarding the  design  or manufacture of the part. Motorola */
/* and the Motorola logo* are registered trademarks of Motorola Inc.         */
/* Motorola is an Equal Opportunity/Affirmative Action Employer.             */
/*****************************************************************************/

